<?php 

namespace App\Helpers;

use Hash;

use App\Admin;

use App\User;

use App\Requests;

use App\RequestsMeta;

use App\Provider;

use App\ProviderPayment;

use App\Jobs\sendPushNotification;

use Mail;

use File;

use Log;

use Storage;

use Setting;

use DB;

use Braintree_Transaction;

use Braintree_Customer;

use Braintree_WebhookNotification;

use Braintree_Subscription;

use Braintree_CreditCard;

class Helper {

    public static function clean($string) {

        $string = str_replace(' ', '-', $string); // Replaces all spaces with hyphens.

        return preg_replace('/[^A-Za-z0-9\-]/', '', $string); // Removes special chars.
    }

    public static function web_url() {
        return url('/');
    }

    public static function generate_email_code($value = "")
    {
        return uniqid($value);
    }

    public static function generate_email_expiry()
    {
        return time() + 24*3600*30;  // 30 days
    }

     // Check whether email verification code and expiry

    public static function check_email_verification($verification_code , $data , &$error, $type) 
    {

        // Check the data exists

        if($data) {

            // Check whether verification code is empty or not

            if($verification_code) {

                if ($verification_code !=  $data->verification_code ) {

                    $error = tr('verification_code_mismatched');

                    return FALSE;

                }

            }
                
            // Check whether verification code expiry 

            if ($data->verification_code_expiry > time()) {

                // Token is valid

                $error = NULL;

                return true;

            } else {

                $data->verification_code = Helper::generate_email_code();

                $data->verification_code_expiry = Helper::generate_email_expiry();

                $data->save();

                // If code expired means send mail to that user

                $subject = tr('verification_code_title');
                $email_data = $data;
                $page = ($type == USER) ? "emails.user.welcome" : "emails.provider.welcome";
                $email = $data['email'];

                $result = Helper::send_email($page,$subject,$email,$email_data);

                $error = tr('verification_code_expired');

                return FALSE;
            }
        }
    }

    // Note: $error is passed by reference
    public static function is_token_valid($entity, $id, $token, &$error) {
        if (
            ( $entity== 'USER' && ($row = User::where('id', '=', $id)->where('token', '=', $token)->first()) ) ||
            ( $entity== 'PROVIDER' && ($row = Provider::where('id', '=', $id)->where('token', '=', $token)->first()) )
        ) {
            if ($row->token_expiry > time()) {
                // Token is valid
                $error = NULL;
                return $row;
            } else {
                $error = array('success' => false, 'error' => Helper::error_message(103), 'error_code' => 103);
                return FALSE;
            }
        }
        $error = array('success' => false, 'error' => Helper::error_message(104), 'error_code' => 104);
        return FALSE;
    }

    // Convert all NULL values to empty strings
    public static function null_safe($data) {

        $result = array();

        foreach ($data as $key => $value) {

            if(is_array($value)) {

                $result[$key] = empty($value) ? [] : $value;

            } else {

                $result[$key] = ($value == NULL && $value != 0) ? "" : $value;
            }
            
        }
        
        return $result;
    }

    public static function generate_token() {
        return Helper::clean(Hash::make(rand() . time() . rand()));
    }

    public static function generate_token_expiry() {
        return time() + 24*3600*30;  // 30 days
    }

    public static function generate_password() {
        $new_password = time();
        $new_password .= rand();
        $new_password = sha1($new_password);
        $new_password = substr($new_password,0,8);
        return $new_password;
    }

    public static function upload_avatar($folder,$picture) {

        $file_name = Helper::file_name();

        $ext = $picture->getClientOriginalExtension();

        $local_url = $file_name . "." . $ext;

        $ext = $picture->getClientOriginalExtension();

        $picture->move(public_path()."/".$folder, $file_name . "." . $ext);

        $url = Helper::web_url().'/'.$folder."/".$local_url;

        return $url;
    
    }

    public static function delete_avatar($folder,$picture) {
        File::delete( public_path() . "/".$folder."/". basename($picture));
        return true;
    }

    public static function file_name() {

        $file_name = time();
        $file_name .= rand();
        $file_name = sha1($file_name);

        return $file_name;
    }

    public static function error_message($code) {

        switch($code) {
            case 101:
                $string = tr('invalid_input');
                break;
            case 102:
                $string = tr('email_already_use');
                break;
            case 103:
                $string = tr('token_expired');
                break;
            case 104:
                $string = tr('invalid_token');
                break;
            case 105:
                $string = tr('invalid_email_password');
                break;
            case 106:
                $string = tr('current_password_wrong');
                break;
            case 107:
                $string = tr('password_not_match');
                break;
            case 108:
                $string = tr('mail_configure_error');
                break;
            case 109: 
                $string = tr('request_status_mismatch');
                break;
            case 110:
                $string = tr('request_current_job_error');
                break;
            case 111:
                $string = tr('no_location_providers');
                break;
            case 112: 
                $string = tr('temporary_request_not_found');
                break;
            case 113:
                $string = tr('request_create_error');
                break;
            case 114: 
                $string = tr('provider_no_available');
                break;
            case 115:
                $string = tr('request_or_job_type_disabled');
                break;
            case 116:
                $string = tr('request_delete_not_allow');
                break;
            case 117:
                $string = tr('request_update_not_allow');
                break;
            case 118:
                $string = tr('request_rating_status_mismatch');
                break;
            case 119:
                $string = tr('no_payment_modes');
                break;
            case 120:
                $string = tr('request_cancel_already');
                break;
            case 121:
                $string = tr('request_cancel_status_mismatch');
                break;
            case 122:
                $string = tr('request_assign_status_already');
                break;
            case 123:
                $string = tr('request_assign_status_mismatch');
                break;
            case 124:
                $string = tr('request_not_found');
                break;
            case 125:
                $string = tr('request_not_offered_provider');
                break;
            case 126:
                $string = tr('request_provider_start_error');
                break;
            case 127:
                $string = tr('request_start_before_error');
                break;
            case 128;
                $string = tr('request_already_ongoing');
                break;
            case 129:
                $string = tr('provider_arrived_already');
                break;
            case 130:
                $string = tr('service_already_started');
                break;
            case 131:
                $string = tr('service_already_completed');
                break;
            case 132:
                $string = tr('request_rating_status_mismatch');
                break;
            case 133:
                $string = tr('provider_cancel_not_allow');
                break;
            case 134:
                $string = tr('provider_not_available');
                break;
            case 135:
                $string = tr('provider_rating_already_done');
                break;
            case 136:
                $string = tr('already_confirmed_payment');
                break;
            case 137:
                $string = tr('no_result_found');
                break;
            case 138:
                $string = tr('request_provider_date_already_job');
                break;
            case 139:
                $string = tr('request_provider_send_bid_status_mismatch');
                break;
            case 140:
                $string = tr('request_already_confirmed');
                break;
            case 141:
                $string = tr('selecetd_payment_disabled');
                break;
            case 142:
                $string = tr('payment_cancelled_tryagain');
                break;
            case 143:
                $string = tr('no_default_card_available');
                break;
            case 144:
                $string = tr('service_already_paid');
                break;
            case 145: 
                $string = tr('user_account_delete_success');
                break;
            case 146:
                $string = tr('payment_details_not_filled');
                break;
            case 147:
                $string = tr('selected_option_disabled_admin');
                break;
            case 148:
                $string = tr('select_ondemand_jobtype');
                break;
            case 149:
                $string = tr('select_later_jobtype');
                break;
            case 150:
                $string = tr('update_payment_mode');
                break;
            case 151:
                $string = tr('you_have_pending_payment');
                break;
            case 152:
                $string = tr('temporary_request_not_belongs_user');
                break;
            case 153:
                $string = tr('mail_not_found');
                break;
            case 154:
                $string = tr('password_not_match');
                break;
            case 155:
                $string = tr('request_date_greater_than');
                break;
            case 157:
                $string = tr('price_per_service');
                break;
            case 158:
                $string = tr('ondemand_request_cancel');
                break;
            case 159:
                $string = tr('agree_error');
                break;
            case 162:
                $string = tr('failed_to_upload');
                break;
            case 163:
                $string = tr('please_enter_amount');
                break;
            case 164:
                $string = tr('stripe_configuration_error');
                break;
            case 165:
                $string = tr('stripe_card_add_error');
                break;
            case 166:
                $string = tr('provider_service_not_exists');
                break;
            case 167:
                $string = tr('provider_service_already_off');
                break;
            case 168:
                $string = tr('coupon_code_not_found');
                break;
            case 169:
                $string = tr('coupon_inactive_status');
                break;
            case 170:
                $string = tr('total_no_of_users_maximum_limit_reached');
                break;
            case 171:
                $string = tr('per_users_limit_exceed');
                break;
            case 172:
                $string = tr('expired_coupon_code');
                break;
            case 173:
                $string = tr('requests_cancelled');
                break;
            case 174:
                $string = tr('gallery_not_found');
                break;
            case 175:
                $string = tr('fav_provider_not_found');
                break;
            case 176:
                $string = tr('requests_payment_failed');
                break;            

            case 177:
                $string = tr('category_not_available');
                break;
            
            case 178:
                $string = tr('user_not_available');
                break;
            // version 5.0

            case 50101:
                $string = tr('referral_code_invalid');
                break;


            case 201:
                $string = tr('redeem_disabled_by_admin');
                break;
            case 202:
                $string = tr('minimum_redeem_not_have');
                break;
            case 203:
                $string = tr('redeem_wallet_empty');
                break;
            case 204:
                $string = tr('redeem_request_status_mismatch');
                break;
            case 205:
                $string = tr('redeem_not_found');
                break;
            case 404:
                $string = tr('no_results');
                break;
            case 405:
                $string = tr('save_error');
                break;

            case 502:
                $string = tr('user_account_declined_by_admin');
                break;
            case 503:
                $string = tr('user_account_email_not_verified');
                break;
            case 504:
                $string = tr('login_account_record_not_found');
                break;
            case 505:
                $string = tr('forgot_password_fail_email_configuration_error');
                break;
            case 506:
                $string = tr('card_default_delete_error');
                break;
            case 507:
                $string = tr('record_not_found');
                break;
            case 508:
                $string = tr('user_account_not_allowed');
                break;

            default:
                $string = tr('unknown_error');
        }
        return $string;
    
    }

    /**
     *
     *
     *
     *
     */
    public static function success_message($code , $other_key = "") {

        switch($code) {
            case 200:
                $string = tr('success');
                break;
            case 202:
                $string = tr('login_success');
                break;
            case 203:
                $string = tr('logout_success');
                break;
            case 204:
                $string = tr('register_success');
                break;
            case 205:
                $string = tr('mail_sent_success');
                break;
            case 206;
                $string = tr('request_create_success');
                break;
            case 207;
                $string = tr('request_cancel_success');
                break;
            case 208:
                $string = tr('request_accepted');
                break;
            case 209:
                $string = tr('provider_started');
                break;
            case 210:
                $string = tr('provider_arrived');
                break;
            case 211:
                $string = tr('service_started');
                break;
            case 212:
                $string = tr('service_completed');
                break;
            case 213:
                $string = tr('service_rating_done');
                break;
            case 214:
                $string = tr('request_provider_assigned_success');
                break;
            case 215:
                $string = tr('request_user_reject_success');
                break;
            case 216:
                $string = tr('register_success');
                break;
            case 217:
                $string = tr('password_change_success');
                break;
            case 218:
                $string = tr('profile_updated');
                break;
            case 219:
                $string = tr('user_account_delete_success');
                break;
            case 220:
                $string = tr('service_updated_success');
                break;
            case 221:
                $string = tr('available_updated');
                break;
            case 222:
                $string = tr('bidding_sent_success');
                break;
            case 223:
                $string = tr('action_success');
                break;
            case 224;
                $string = tr('card_delete_success');
                break;
            case 225;
                $string = tr('card_default_success');
                break;
            case 226:
                $string = tr('request_rejected');
                break;
            case 227:
                $string = tr('request_payment_success');
                break;
            case 228:
                $string = tr('payment_mode_change_success');
                break;
            case 229:
                $string = tr('request_payment_confirm_success');
                break;
            case 230:
                $string = tr('request_bidding_cancel_success');
                break;
            case 231:
                $string = tr('email_verification_message');
                break;
            case 232:
                $string = tr('user_forgot_password_success_content' , $other_key);
                break;
            case 233:
                $string = tr('stripe_card_add_success');
                break;
            case 234:
                $string = tr('payment_mode_update_success');
                break;
            case 235:
                $string = tr('provider_service_removed_success');
                break;
            case 236:
                $string = tr('provider_gallery_updated_success');
                break;

            case 237:
                $string = tr('provider_gallery_deleted_success');
                break;
            case 238:
                $string = tr('add_no_of_times_used_coupon');
                break;
            case 239:
                $string = tr('create_a_new_coupon_row');
                break;
            case 240:
                $string = tr('fav_provider_removed_success');
                break;   

            case 30301: 
                $string = tr('provider_subscription_payment_success');
                break;
            case 30302: 
                $string = tr('user_notification_update_success');
                break;
            case 30303: 
                $string = tr('provider_notification_update_success');
                break;
                
            case 50001:
                    $string = tr('referral_code_valid');
                    break;
            default:
                $string = "";
        }
        return $string;
    
    }

    public static function send_notification($id,$title,$message) {

        if(Setting::get('push_notification')) {

            Log::info("Send Push Started");

            // Check the user type whether "USER" or "PROVIDER"
            if($id == "all") {
                $users = User::where('push_status' , 1)->get();
            } else {
                $users = User::find($id);
            }

            $push_data = array();

            $push_message = array('success' => true,'message' => $message,'data' => array());

            $push_notification = 1; // Check the push notifictaion is enabled

            if ($push_notification == 1) {

                Log::info('Admin enabled the push ');

                if($users){

                    Log::info('Check users variable');

                    foreach ($users as $key => $user) {

                        Log::info('Individual User');

                        if ($user->device_type == 'ios') {

                            Log::info("iOS push Started");

                            require_once app_path().'/ios/apns.php';

                            $msg = array("alert" => $message,
                                "status" => "success",
                                "title" => $title,
                                "message" => $push_message,
                                "badge" => 1,
                                "sound" => "default",
                                "status" => "",
                                "rid" => "",
                                );

                            if (!isset($user->device_token) || empty($user->device_token)) {
                                $deviceTokens = array();
                            } else {
                                $deviceTokens = $user->device_token;
                            }

                            $apns = new \Apns();
                            $apns->send_notification($deviceTokens, $msg);

                            Log::info("iOS push end");

                        } else {

                            Log::info("Andriod push Started");

                            require_once app_path().'/gcm/GCM_1.php';
                            require_once app_path().'/gcm/const.php';

                            if (!isset($user->device_token) || empty($user->device_token)) {
                                $registatoin_ids = "0";
                            } else {
                                $registatoin_ids = trim($user->device_token);
                            }
                            if (!isset($push_message) || empty($push_message)) {
                                $msg = "Message not set";
                            } else {
                                $msg = $push_message;
                            }
                            if (!isset($title) || empty($title)) {
                                $title1 = "Message not set";
                            } else {
                                $title1 = trim($title);
                            }

                            $message = array(TEAM => $title1, MESSAGE => $msg);

                            $gcm = new \GCM();
                            $registatoin_ids = array($registatoin_ids);
                            $gcm->send_notification($registatoin_ids, $message);

                            Log::info("Andriod push end");

                        }

                    }

                }

            } else {
                Log::info('Push notifictaion is not enabled. Please contact admin');
            }

            Log::info("*************************************");

        }

    }

    public static function send_email($page,$subject,$email,$email_data) {

        \Log::info("Mail Username ".envfile('MAIL_USERNAME'));

        \Log::info("Mail Username ".envfile('MAIL_PASSWORD'));

        // Log::info("send_email -".print_r($email_data, true));
        
        if(Setting::get('email_notification')) {

            if( envfile('MAIL_USERNAME') &&  envfile('MAIL_PASSWORD')) {

                try {

                    $site_url= url('/');
                    
                    $mail = Mail::queue($page, array('email_data' => $email_data), function ($message) use ($email, $subject) {
                        
                            $message->to($email)->subject($subject);
                    });

                } catch(\Exception $e) {

                    // \Log::info(print_r($e , true));

                    return Helper::error_message(108);
                }
                return Helper::success_message(105);

            } else {
                return Helper::error_message(108);
            }
        }
   
    }

    public static function delete_picture($picture) {
        File::delete( public_path() . "/uploads/" . basename($picture));
        return true;
    }

    public static function time_diff($start,$end) {
        $start_date = new \DateTime($start);
        $end_date = new \DateTime($end);

        $time_interval = date_diff($start_date,$end_date);
        return $time_interval;

    }

    public static function formatHour($date) {
        $hour_time  = date("H:i:s",strtotime($date));
        return $hour_time;
    }

    public static function formatDate($date) {
        $newdate  = date("Y-m-d",strtotime($date));
        return $newdate;
    }

    public static function add_date($date,$no_of_days) {

        $change_date = new \DateTime($date);
        $change_date->modify($no_of_days." hours");
        $change_date = $change_date->format("Y-m-d H:i:s");
        return $change_date;
    }


    // Usage : provider Incoming request and cron function

    public static function assign_next_provider($request_id,$provider_id) {

        if($requests = Requests::find($request_id)) {

            //Check the request is offered to the current provider
            
            if($provider_id) {

                $current_offered_provider = RequestsMeta::where('provider_id',$provider_id)
                                ->where('request_id',$request_id)
                                ->where('status', REQUEST_META_OFFERED)
                                ->first();

                // Change waiting to respond state
                if($current_offered_provider) {

                    $get_offered_provider = Provider::where('id',$current_offered_provider->provider_id)->first();
                    
                    $get_offered_provider->is_available = PROVIDER_AVAILABLE;
                    $get_offered_provider->waiting_to_respond = WAITING_TO_RESPOND_NORMAL;
                    $get_offered_provider->save();

                    // TimeOut the current assigned provider
                    $current_offered_provider->status = REQUEST_META_TIMEDOUT;
                    $current_offered_provider->save();
                }
            }

            //Select the new provider who is in the next position.
            $next_request_meta = RequestsMeta::where('request_id', '=', $request_id)->where('requests_meta.status', REQUEST_META_NONE)
                                ->leftJoin('providers', 'providers.id', '=', 'requests_meta.provider_id')
                                ->where('providers.is_available',true)
                                ->where('providers.is_approved',PROVIDER_APPROVED)
                                ->where('providers.waiting_to_respond',WAITING_TO_RESPOND_NORMAL)
                                ->select('requests_meta.id','requests_meta.status','requests_meta.provider_id')
                                ->orderBy('requests_meta.created_at')
                                ->first();

            //Check the next provider exist or not.
            if($next_request_meta){

                // change waiting to respond state
                $provider_detail = Provider::find($next_request_meta->provider_id);
                $provider_detail->waiting_to_respond = WAITING_TO_RESPOND;
                $provider_detail->save();

                //Assign the next provider.
                $next_request_meta->status = REQUEST_META_OFFERED;
                $next_request_meta->save();

                $time = date("Y-m-d H:i:s");

                //Update the request start time in request table
                Requests::where('id', '=', $request_id)->update( array('request_start_time' => date("Y-m-d H:i:s")) );

                Log::info('assign_next_provider_cron assigned provider to request_id:'.$request_id.' at '.$time);


                // Send notifications to the provider

                $title = Helper::push_message(8010);

                $message = Helper::push_message(8011);

                dispatch(new sendPushNotification($next_request_meta->provider_id,PROVIDER,PUSH_PROVIDER_HOME,$title,$message , 1));


            } else {
                Log::info("No provider available this time - cron");
                //End the request
                //Update the request status to no provider available
                Requests::where('id', '=', $request_id)->update( array('status' => REQUEST_NO_PROVIDER_AVAILABLE) );

                // No longer need request specific rows from RequestMeta
                RequestsMeta::where('request_id', '=', $request_id)->forceDelete();

                // Send Push Notification to User
                // $title = Helper::tr('cron_no_provider_title');
                // $message = Helper::tr('cron_no_provider_message');

                // dispatch(new NormalPushNotification($requests->user_id,USER,$title,$message));

                // Send notifications to the provider

                $title = Helper::push_message(8012);

                $message = Helper::push_message(8013);

                dispatch(new sendPushNotification($requests->user_id,USER,PUSH_USER_HOME,$title,$message));
            
            }
        }
    
    }

    public static function registerUserOnBrainTree($first_name,$last_name,$email,$phone) {
        
        $result = Braintree_Customer::create(array(
            'firstName' => $first_name,
            'lastName' => $last_name,
            'email' => $email,
            'phone' => $phone
        ));

        if ($result->success) {
            return $result->customer->id;
        } else {
            $errorFound = '';
            foreach ($result->errors->deepAll() as $error) {
                $errorFound .= $error->message . "<br />";
            }

            echo $errorFound ;
        }
    }

    public static function getCardToken($customer_id,$card_number,$card_expiry,$cvv) {

        $card_result = Braintree_CreditCard::create(array(
            'number' => $card_number,
            'expirationDate' => trim($card_expiry),
            'customerId' => $customer_id,
            'cvv' => $cvv
        ));
        
        if($card_result->success) {
            return $card_result->creditCard->token;
        } else {
            return $card_result;
        }
    
    }

    /**
     * User payment with brain tree
     *
     * @param $customer_id = user card customer id
     *
     * @param $request_id  For which request user paying
     * 
     * @param $total = Request total 
     *
     */

    public static function createTransaction($customer_id,$request_id,$total) {

        $result = Braintree_Transaction::sale([
                    'customerId' => $customer_id,
                    'amount' => $total,
                    'merchantAccountId'=> envfile('PAYMENT_MERCHANT_ACCOUNTID'),
                    'orderId' => $request_id,
                    'options' => [
                            'submitForSettlement' => True
                    ]
                    ]);
        
       // Log::info('Payment details'.print_r($result->transaction->id,true));

        if ($result->success) {
                return $result->transaction->id;
        } else {
                //return 0;
            $errorFound = '';
            
            foreach ($result->errors->deepAll() as $error1) {
                $errorFound .= $error1->message . "<br />";

            }
            return $errorFound;
        }
    
    }

    /**
     * Used save total earnings of the provider
     * 
     * @param $provider_id , $amount , $paymentt_mode
     *
     */

    public static function provider_payment($provider_id, $total = 0, $payment_mode, $admin_commission = 0, $provider_amount = 0) {

        Log::info("Provider Payment UPDATE - "."Payment MODE : ". $payment_mode ."TOTAL AMOUNT ".$total ." Admin Amount :". $admin_commission."PROVIDER AMOUNT : ".$provider_amount);

        $provider_payment = ProviderPayment::where('provider_id' , $provider_id)->first();

        if(!$provider_payment) {

            $provider_payment = new ProviderPayment;

            $provider_payment->provider_id = $provider_id;

            $provider_payment->pending_amount = $provider_payment->total = 0;

            $provider_payment->status = 1;

        }

        // Update provider and admin commission percentage

        $provider_payment->admin_commission += $admin_commission;

        $provider_payment->provider_amount += $provider_amount;

        if($payment_mode == COD) {

            $provider_payment->cod += $total;

            $provider_payment->pending_amount += $admin_commission;

        } else {

            $provider_payment->card += $total; 

            // Check the pending amount is less than the total amount

            $provider_pending_amount_to_admin_commission = $provider_payment->pending_amount;

            if(floatval($total) >= floatval($provider_pending_amount_to_admin_commission)) {

                // minus the pending amount and add remaing amount to wallet
                
                $wallet = $total - $provider_pending_amount_to_admin_commission - $admin_commission;

                $provider_payment->wallet += $wallet;

                $provider_payment->pending_amount = 0;

            } else {

                $provider_payment->pending_amount -= $total;

            }
        
        }

        $provider_payment->total += $total;

        $provider_payment->save();

    }

  
    public static function upload_language_file($folder,$picture,$filename) {

        $ext = $picture->getClientOriginalExtension();
            
        $picture->move(base_path() . "/resources/lang/".$folder ."/", $filename);

    }

    public static function delete_language_files($folder, $boolean, $filename = '') {
       
        if ($boolean) {
            
            $path = base_path() . "/resources/lang/" .$folder;
            File::cleanDirectory($path);
            Storage::deleteDirectory( $path );
            rmdir( $path );

        } else {
            
            File::delete( base_path() . "/resources/lang/" . $folder ."/".$filename);

            return true;
        }

    }

    public static function push_message($code) {

        switch($code) {
            case 8000:
                $string = tr('push_assign_request_provider_title');
                break;
            case 8001:
                $string = tr('push_assign_request_provider_message');
                break;
            case 8002:
                $string = tr('push_user_waiting_request_cancel_title');
                break;
            case 8003:
                $string = tr('push_user_waiting_request_cancel_message');
                break;
            case 8004:
                $string = tr('push_user_reject_bid_title');
                break;
            case 8005:
                $string = tr('push_user_reject_bid_message');
                break;
            case 8006:
                $string = tr('push_user_cancel_job_title');
                break;
            case 8007:
                $string = tr('push_user_cancel_job_message');
                break;
            case 8008:
                $string = tr('push_user_paid_title');
                break;
            case 8009:
                $string = tr('push_user_paid_message');
                break;
            case 8010:
                $string = tr('push_user_new_request_title');
                break;
            case 8011:
                $string = tr('push_user_new_request_message');
                break;
            case 8012:
                $string = tr('push_cron_request_timeout_title');
                break;
            case 8013:
                $string = tr('push_cron_request_timeout_message');
                break;
            case 8014:
                $string = tr('push_provider_accept_request_title');
                break;
            case 8015:
                $string = tr('push_provider_accept_request_message');
                break;
            case 8016:
                $string = tr('push_provider_send_bid_title');
                break;
            case 8017:
                $string = tr('push_provider_send_bid_message');
                break;
            case 8018:
                $string = tr('push_provider_started_location_title');
                break;
            case 8019:
                $string = tr('push_provider_started_location_message');
                break;

            case 8020:
                $string = tr('push_provider_arrived_location_title');
                break;
            case 8021:
                $string = tr('push_provider_arrived_location_message');
                break;

            case 8022:
                $string = tr('push_provider_service_started_title');
                break;
            case 8023:
                $string = tr('push_provider_service_started_message');
                break;

            case 8024:
                $string = tr('push_provider_service_completed_title');
                break;
            case 8025:
                $string = tr('push_provider_service_completed_message');
                break;

            case 8026:
                $string = tr('push_provider_payment_confirm_title');
                break;
            case 8027:
                $string = tr('push_provider_payment_confirm_message');
                break; 

            case 8028:
                $string = tr('push_provider_cancel_title');
                break;
            case 8029:
                $string = tr('push_provider_cancel_message');
                break;

            case 8030:
                $string = tr('push_provider_confirm_request_title');
                break;
            case 8031:
                $string = tr('push_provider_confirm_request_message');
                break;

            default:
                $string = tr('unknown_error');
        }

        return $string;

    }

    public static function getEnvValues() {
        $data =  array();

        $path = base_path('.env');

        if(file_exists($path)) {

            $values = file_get_contents($path);

            $values = explode("\n", $values);

            foreach ($values as $key => $value) {

                $var = explode('=',$value);

                if(count($var) == 2 ) {
                    if($var[0] != "")
                        $data[$var[0]] = $var[1] ? $var[1] : null;
                } else if(count($var) > 2 ) {
                    $keyvalue = "";
                    foreach ($var as $i => $imp) {
                        if ($i != 0) {
                            $keyvalue = ($keyvalue) ? $keyvalue.'='.$imp : $imp;
                        }
                    }
                    $data[$var[0]] = $var[1] ? $keyvalue : null;
                }else {
                    if($var[0] != "")
                        $data[$var[0]] = null;
                }
            }

            array_filter($data);
        
        }

        return $data;

    }

    /**
     * @method check_user_delete_account() 
     * 
     * @created Vidhya R
     *
     * @updated 
     *
     * @param int $user_id
     *
     * @return boolean response
     *
     */

    public static function check_user_delete_account($user_id) {

        // Check ongoing requests and pending payments

        $check_request = Requests::where('user_id' , $user_id)->whereIn('status' , [REQUEST_ONGOING ,REQUEST_COMPLETE_PENDING , WAITING_PAYMENT_CONFIRM])->count();

        return $check_request ? 0 : 1;

    }

    /**
     *
     * @method youtube_vimeo_link_convert() 
     *
     * @uses to convert youtube and vimeo link convert to iframe link
     * 
     * @created Vidhya R
     *
     * @updated 
     *
     * @param string $url
     *
     * @return string $converted_url
     *
     */

    public static function youtube_vimeo_link_convert($url) {

        $shortUrlRegex = '/youtu.be\/([a-zA-Z0-9_]+)\??/i';
        
        $longUrlRegex = '/youtube.com\/((?:embed)|(?:watch))((?:\?v\=)|(?:\/))(\w+)/i';

        if (preg_match($longUrlRegex, $url, $matches)) {
            $youtube_id = $matches[count($matches) - 1];
        }

        if (preg_match($shortUrlRegex, $url, $matches)) {
            $youtube_id = $matches[count($matches) - 1];
        }
        return 'https://www.youtube.com/embed/' . $youtube_id ;

    }

}
