<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Repositories\ProviderRepository as ProviderRepo;

use App\Repositories\CommonRepository as CommonRepo;

use App\Repositories\AdminRepository as AdminRepo;

use App\Requests;

use App\User;

use App\Provider;

use App\Admin;

use App\Language;

use App\Card;

use App\UserHistory;

use App\RequestPayment;

use App\ProviderPayment;

use App\ProviderDocument;

use App\ConfigValue;

use App\Settings;

use App\Page;

use App\Category;

use App\SubCategory;

use App\ProviderService;

use App\Document;

use App\BannerImage;

use App\Coupon;

use App\ProviderRating;

use App\UserRating;

use App\AbuseProvider;

use App\Helpers\Helper;

use Validator;

use Hash;

use Mail;

use DB;

use Auth;

use Redirect;

use Setting;

use Log;

use Enveditor;

use Artisan;

use Exception;

use App\Redeem;

use App\RedeemRequest;

use App\ProviderGallery;

use App\UserFavProvider;

use App\ProviderSubscription;

use App\ProviderSubscriptionPayment;

use App\Referral;

use App\UserReferrer;

use App\EmailTemplate;

class AdminController extends Controller {

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('admin');  
    }

    /**
     *
     * @method check_role()
     *
     * @uses To check the role of logged in user (admin/subadmin) 
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param $request - request details 
     *
     * @return success/failure message
     */
    public function check_role(Request $request) {
        
        if(Auth::guard('admin')->check()) {
            
            $admin_details = Auth::guard('admin')->user();

            if($admin_details->role == ADMIN) {

                return redirect()->route('admin.dashboard');
            }

            if($admin_details->role == SUBADMIN) {

                return redirect()->route('subadmin.dashboard');
            }

        } else {

            return redirect()->route('admin.login');
        }

    }

    public function login() {
       
            return view('admin.login')
                    ->with('page','admin-login')
                    ->with('sub_page','');
    }

    /**
     * @method dashboard()
     *
     * @uses overall website analytics
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return view page
     */
    public function dashboard() {
        
        $user_count = User::where('login_status','=','user')->count();

        $provider_count = Provider::where('login_status','=','provider')->count();

        $requests_count = Requests::count();

        $recent_providers = Provider::orderBy('created_at' , 'DESC')->get();

        $get_registers = get_register_count();

        $recent_users = get_recent_users();
        
        $total_revenue = total_revenue();

        $view = last_days(10);

        return view('admin.dashboard')
                    ->with('page','dashboard')
                    ->with('sub_page','')
                    ->with('user_count' , $user_count)
                    ->with('provider_count' , $provider_count)
                    ->with('requests_count' , $requests_count)
                    ->with('get_registers' , $get_registers)
                    ->with('view' , $view)
                    ->with('total_revenue' , $total_revenue)
                    ->with('recent_users' , $recent_users)
                    ->with('recent_providers' , $recent_providers);
    
    }

    /**
     * @method profile()
     *
     * @uses display profile page
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return view page
     */
    public function profile() {

        $admin_details = Admin::where('id' , Auth::guard('admin')->user()->id)->first();

        return view('admin.profile')
                ->with('page','profile')
                ->with('sub_page','')
                ->with('admin_details' , $admin_details);
    }

    /**
     * @method profile_save()
     *
     * @uses To store/update the admin details
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - (request) details
     *
     * @return success/error message
     */
    public function profile_save(Request $request) {

        try {

            DB::begintransaction();

            if(!CommonRepo::profile_validation($request->all(), $errors)) {

                throw new Exception($errors, 101);
            }
                
            $admin_details = Admin::find($request->id);
            
            $admin_details->name = $request->has('name') ? $request->name : $admin_details->name;

            $admin_details->email = $request->has('email') ? $request->email : $admin_details->email;

            $admin_details->mobile = $request->has('mobile') ? $request->mobile : $admin_details->mobile;

            $admin_details->gender = $request->has('gender') ? $request->gender : $admin_details->gender;

            $admin_details->address = $request->has('address') ? $request->address : $admin_details->address;

            if($request->hasFile('picture')) {
                
                Helper::delete_avatar('uploads/admin', $admin_details->picture);

                $admin_details->picture = Helper::upload_avatar('uploads/admin',$request->file('picture'));
            }
                
            $admin_details->remember_token = Helper::generate_token();

            if( $admin_details->save()) {

                DB::commit();

                return back()->with('flash_success', tr('admin_not_profile'));
            }

            throw new Exception(tr('admin_not_error'), 101);                    
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }
     
    }

    /**
     * @method change_password()
     *
     * @uses To change password for the logged in admin
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - (request) details
     *
     * @return success/error message
     */
    public function change_password(Request $request) {

        try {

            DB::begintransaction();

            $validator = Validator::make($request->all(), [              
                'password' => 'required|confirmed|min:6',
                'old_password' => 'required',
                'id' => 'required|exists:admins,id'
            ]);

            if($validator->fails()) {

                $errors = implode(',',$validator->messages()->all());

                throw new Exception($errors, 101);
            } 

            $admin = Admin::find($request->id);

            if(Hash::check($request->old_password,$admin->password)) {

                $admin->password = Hash::make($request->password);

                if( $admin->save()) {

                    DB::commit();

                    return back()->with('flash_success', tr('admin_password_change_success'));
                }

                throw new Exception(tr('admin_password_save_error'), 101);
            }

            throw new Exception( tr('admin_password_mismatch'), 101);

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error',$error);
        }       
    }
    
    /**
     * @method users_index()
     *
     * @uses To list out users details
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return View page
     */
    public function users_index(Request $request) {

        try {

            $base_query = User::orderBy('created_at','desc')->where('login_status','user');
                    
            if ($request->user_id && $request->user_referrer_id) {
               
                $refered_users = Referral::where('parent_user_id', '=' , $request->user_id)->where('user_referrer_id', '=' , $request->user_referrer_id)->select('user_id')->get();
               
                if(!$refered_users) { 
                    
                    throw new Exception(tr('admin_user_refered_accounts_not_found'), 101);                    
                }

                $refered_users_ids = array_column($refered_users->toArray(), 'user_id');

                $base_query->whereIn('id', $refered_users_ids);                
                // $base_query->where('id', $refered_users_ids);                
            }

            $users = $base_query->paginate(10);

            return view('admin.users.index')
                        ->withPage('users')
                        ->with('sub_page','users-view')
                        ->with('users' , $users);
           
        } catch (Exception $e) {

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error',$error);            
        }

    }

    /**
     * @method users_create()
     *
     * @uses To create a user details
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return View page
     */
    public function users_create() {

        $user_details = new User;

        return view('admin.users.create')
                    ->with('page' , 'users')
                    ->with('sub_page','users-create')
                    ->with('user_details',$user_details);
    }

    /**
     * @method users_edit()
     *
     * @uses To display and update user details based on the user id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $user_id
     *
     * @return View page
     */
    public function users_edit(Request $request) {

        try {
          
            $user_details = User::find($request->user_id);

            if(!$user_details) {

                throw new Exception( tr('user_not_found'), 101);
            } 

            return view('admin.users.edit')
                    ->with('page' , 'users')
                    ->with('sub_page','users-view')
                    ->with('user_details',$user_details);
                      
        } catch( Exception $e) {
            
            $error = $e->getMessage();

            return redirect()->route('admin.users.index')->with('flash_error',$error);
        }

    }

    /**
     * @method users_save()
     *
     * @uses To save the user object details of new/existing based on details
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $user_id , (request) user details
     *
     * @return success/error message
     */
    public function users_save(Request $request) {

        try {
        
            DB::beginTransaction();

            if($request->user_id != '') {

                $validator = CommonRepo::admin_edit_validation($request->all(), $errors, USER, $request->user_id);

            } else {

                $validator = CommonRepo::admin_add_validation($request->all(), $errors, USER);
            }
   
            if(!$validator) {

                throw new Exception($errors, 101);
            } 

            if($request->user_id != '') {

                $user_details = User::find($request->user_id);

                $message = tr('admin_user_update_success');

                if ($request->hasFile('picture')) {

                    Helper::delete_avatar('uploads/users',$user_details->picture); // Delete the old pic
                }

            } else {

                // Add New User

                $user_details = new User;

                $message = tr('admin_user_add_success');

                $name = $request->has('name') ? routefreestring($request->name) : "";
                
                $user_details->password = \Hash::make($request->password);
                
                $user_details->payment_mode = COD;

                $user_details->login_by = LOGIN_BY_MANUAL;

                $user_details->login_status = USER;

                $user_details->device_type = $user_details->register_type = DEVICE_WEB;

                $user_details->picture = asset('placeholder.png');

                $user_details->is_activated = $user_details->status = $user_details->is_verified = USER_APPROVED; 

                $user_details->token = Helper::generate_token();

                $user_details->token_expiry = Helper::generate_token_expiry();

            } 

            if($request->hasFile('picture')) {

                $user_details->picture = Helper::upload_avatar('uploads/users',$request->file('picture'));
            } 

            $user_details->name = $request->has('name') ? $request->name : '';

            $user_details->email = $request->has('email') ? $request->email: '';

            $user_details->mobile = $request->has('mobile') ? $request->mobile : '';

            $user_details->description = $request->has('description') ? $request->description : '';
            
            if($request->user_id == '') {

                $email_details['name'] = $user_details->name;
                
                $email_details['password'] = $request->password;
                
                $email_details['email'] = $user_details->email;


                $subject = tr('user_welcome_title');

                $page = "emails.welcome";

                $email = $user_details->email;

                if(envfile('MAIL_USERNAME')  && envfile('MAIL_PASSWORD')) {

                    Helper::send_email($page,$subject,$email,$email_details);
                }

                register_mobile('web');   
            }

            if( $user_details->save() ) {

                DB::commit();

                CommonRepo::become_a_provider($user_details);

                return redirect()->route('admin.users.view' ,['user_id' => $user_details->id] )->with('flash_success', $message);

            } 

            throw new Exception( tr('admin_user_save_error') , 101);
            
         } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error', $error);
        }   
    
    }

    /**
     * @method users_view()
     *
     * @uses To display the particular user details single view page
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $user_id 
     *
     * @return redirect to view user page
     */
    public function users_view(Request $request) {

        try {
               
            $user_details = User::find($request->user_id) ;

            if( count($user_details) == 0 ) {

                throw new Exception(tr('user_not_found'), 101);
            } 
            // @todo Wallet modeule to be done hence hidden referral_settings
            $users_referral_details = UserReferrer::where('user_id', $request->user_id)                            
                ->withCount('getReferral')
                ->first();
          
            return view('admin.users.view')
                    ->with('page','users')
                    ->with('sub_page','users-view')
                    ->with('user_details' , $user_details)
                    ->with('users_referral_details' , $users_referral_details);
            
        } catch( Exception $e) {
            
            $error = $e->getMessage();

            return redirect()->route('admin.users.index')->with('flash_error',$error);
        }

    }

    /**
     * @method users_delete()
     *
     * @uses To delete perticular user object based on user id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $user_id
     *
     * @return success/error message
     */
    public function users_delete(Request $request) {

        try {

            DB::begintransaction();

            if( $user_details = User::where('id',$request->user_id)->first() ) {

                // check the user having any ongoing requests ( on decline only we need to check the below)

                $check_status = [REQUEST_NONE,REQUEST_NO_PROVIDER_AVAILABLE,REQUEST_TIME_EXCEED_CANCELLED,REQUEST_CANCELLED,REQUEST_COMPLETED];

                $check_requests = Requests::where('user_id' , $request->user_id)->whereNotIn('status' , $check_status)->count();

                if($check_requests) {

                    throw new Exception( tr('admin_user_decline_ongoing_restrict'), 101);
                }

                // Delete the become provider details as well
                $provider_details = $user_details->provider_id ? Provider::find($user_details->provider_id) : "";

                if( $provider_details ) {

                    if( $provider_details->forceDelete()) {

                        DB::commit();

                    } else {

                       throw new Exception(tr('admin_provider_delete_error'), 101);
                    }
                }

                if( $user_details->forceDelete()) {

                    DB::commit();

                    return redirect()->route('admin.users.index')->with('flash_success',tr('user_delete_success'));

                }

                throw new Exception( tr('admin_user_delete_error'), 101);
            } 

            throw new Exception(tr('user_not_found'), 101);            
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.users.index')->with('flash_error', $error);
        }
 
    }

    /**
     * @method users_status_change()
     *
     * @uses To update user approve/decline status based on user id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $user_id
     *
     * @return success/error message
     */
    public function users_status_change(Request $request) {

        try {

            DB::beginTransaction();
       
            $user_details = User::find($request->user_id);

            if(!$user_details) {
                
                throw new Exception(tr('user_not_found'), 101);
            } 

            // check the user having any ongoing requests ( on decline only we need to check the below)

            if($user_details->status == USER_APPROVED) {

                $check_status = [REQUEST_NONE,REQUEST_NO_PROVIDER_AVAILABLE,REQUEST_TIME_EXCEED_CANCELLED,REQUEST_CANCELLED,REQUEST_COMPLETED];

                $check_requests = Requests::where('user_id' , $request->user_id)->whereNotIn('status' , $check_status)->count();

                if($check_requests) {

                    throw new Exception( tr('admin_user_decline_ongoing_restrict'), 101);
                }
            }

            $user_details->status = $user_details->status == USER_APPROVED ? USER_DECLINED : USER_APPROVED;

            if( $user_details->save() ) {

                DB::commit();

                $message = $user_details->status == USER_APPROVED ? tr('user_approve_success') : tr('user_decline_success');
        
                return back()->with('flash_success', $message);

            } 

            throw new Exception(tr('admin_user_status_error'), 101);
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.users.index')->with('flash_error' , $error);
        }

    }


    /**
     * @method users_verify_status()
     *
     * @uses To update user email verification status based on user id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $user_id
     *
     * @return success/error message
     */
    public function users_verify_status(Request $request) {

        try {

            DB::beginTransaction();

            $user_details = User::find($request->user_id);

            if(!$user_details) {

                throw new Exception(tr('user_not_found'), 101);                
            }

            $user_details->is_verified  = $user_details->is_verified == USER_EMAIL_VERIFIED ? USER_EMAIL_NOT_VERIFIED : USER_EMAIL_VERIFIED;

            if( $user_details->save() ) {

                DB::commit();
                
                $message = $user_details->is_verified == USER_EMAIL_VERIFIED ? tr('user_verify_success') : tr('user_unverify_success'); 

                return back()->with('flash_success' , $message);

            } 

            throw new Exception(tr('admin_user_verify_status_error'), 101);
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.users.index')->withInput()->with('flash_error' , $error);
        }

    }

    /**
     * @method users_history()
     *
     * @uses To list out the particular user history based on user id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $user_id
     *
     * @return success/error message
     */

    public function users_history(Request $request) {

        try {

            $user_details = User::find($request->user_id);

            if(!$user_details) {

                throw new Exception( tr('user_not_found'), 101);
            }

            $history = $requests = Requests::where('user_id' , $request->user_id)
                    ->paginate(10);

            return view('admin.users.history')
                        ->with('page','users')
                        ->with('sub_page','users')
                        ->with('history' , $history)
                        ->with('user_details' , $user_details);
            
        } catch ( Exception $e ){

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }    
    }   


    /**
     * @method users_requests()
     *
     * @uses To list out the particular user requests based on user id
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param Integer (request) $user_id
     *
     * @return success/error message
     */

    public function users_requests(Request $request) {

        try {

            $user_details = User::find($request->user_id);

            if(!$user_details) {

                throw new Exception( tr('user_not_found'), 101);
            } 

            $requests = $requests = Requests::where('user_id' , $request->user_id)
                    ->paginate(10);

            return view('admin.users.requests')
                        ->with('page','users')
                        ->with('sub_page','users')
                        ->with('requests' , $requests)
                        ->with('user_details' , $user_details);

        } catch ( Exception $e ){

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }    
    }   


    /**
     * @method users_fav_providers()
     *
     * @uses list the favorite providers based on the selected user
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $user_id
     *
     * @return View page
     */
    public function users_fav_providers(Request $request) {
        try {

            $user_details = User::find($request->user_id);

            if(!$user_details) {

                throw new Exception(tr('user_not_found'), 101);
            }

            $fav_providers = UserFavProvider::where('user_fav_providers.user_id' , $request->user_id)
                        ->leftJoin('requests' ,'requests.id' , "=" , 'user_fav_providers.request_id')
                        ->leftJoin('providers' ,'providers.id' , "=" , 'user_fav_providers.provider_id')
                        ->select('providers.name as provider_name' , 'providers.picture as provider_picture', 'providers.email as provider_email' , 'user_fav_providers.id as user_fav_provider_id', 'user_fav_providers.user_id','user_fav_providers.request_id','user_fav_providers.provider_id','requests.name as request_name','requests.unique_id as request_unique_id' , 'user_fav_providers.created_at')
                        ->get();
           
            return view('admin.users.fav_providers')
                        ->with('page','users')
                        ->with('sub_page','users')
                        ->with('user_details' , $user_details)
                        ->with('fav_providers' , $fav_providers);
            

        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->route('admin.users.index')->with('flash_error',$error);
        }   
    }

    /**
     * @method users_fav_providers_remove()
     *
     * @uses list the favorite providers based on the selected user
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $user_fav_provider_id
     *
     * @return View page
     */

    public function users_fav_providers_remove(Request $request) {

        try {

            DB::begintransaction();

            // check the fav provider record exists

            $fav_provider_details = UserFavProvider::find($request->user_fav_provider_id);

            if(!$fav_provider_details) {

                throw new Exception(tr('user_fav_provider_not_found'), 101);
            } 

            if( $fav_provider_details->delete() ) {

                DB::commit();

                return back()->with('flash_success' , tr('user_fav_provider_delete_success'));
            } 

            throw new Exception(tr('user_fav_provider_delete_error'), 101);
            
        } catch(Exception $e){

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.users.index')->with('flash_error', $error);
        }
    }

    // PROVIDER METHODS

    /**
     * @method providers_index()
     *
     * @uses to list out providers details
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     *
     * @param -
     * 
     * @return view page
     */
    public function providers_index() {

        $providers = Provider::where('login_status','provider')
                        ->orderBy('updated_at','desc')
                        ->paginate(10);

        return view('admin.providers.index')
                    ->with('page','providers')
                    ->with('sub_page','providers-view')
                    ->with('providers' , $providers);
    }

    /**
     * @method providers_create()
     *
     * @uses To create provider object
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     * 
     * @param -
     * 
     * @return view page
     */
    public function providers_create() {

        $provider_details = new Provider;

        $categories = getCategories();

        return view('admin.providers.create')
                    ->with('page' , 'providers')
                    ->with('sub_page','providers-create')
                    ->with('categories',$categories)
                    ->with('provider_details',$provider_details);    
    }

    /**  
     * @method providers_edit()
     *
     * @uses update the provider object based on provider_id
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id
     *
     * @return Success message
     */

    public function providers_edit(Request $request) {

        try {

            DB::begintransaction();

            $provider_details = Provider::find($request->provider_id);

            if(!$provider_details){

                throw new Exception( tr('provider_not_found'), 101);
            } 

            $categories = getCategories();
          
            $provider_services = ProviderService::where('provider_id' , $request->provider_id)
                                            ->orderBy('updated_at','desc')
                                            ->first();

            // Requested the provider values not found -> Added the provider values in default

            if($provider_services =='') {

                $provider_services = new ProviderService;

                $provider_services->provider_id = $request->provider_id ; 

                $provider_services->category_id = $request->category_id ? $request->category_id : DEFAULT_TRUE;

                $provider_services->sub_category_id = $request->sub_category_id ? $request->sub_category_id : DEFAULT_TRUE;

                $provider_services->price_per_hour = $request->price_per_hour ? $request->price_per_hour : DEFAULT_TRUE;

                if( $provider_services->save() ){

                    DB::commit();
                
                } else {

                    throw new Exception( tr('admin_provider_save_error'), 101);
                }
            }

            $request->request->add(['category_id' => $provider_services->category_id]);
           
            $sub_categories = getSubCategories($request);
           
            return view('admin.providers.edit')
                        ->with('page' , 'providers')
                        ->with('sub_page','providers-view')
                        ->with('provider_details',$provider_details)
                        ->with('provider_services' , $provider_services)
                        ->with('sub_categories' , $sub_categories)
                        ->with('categories',$categories);    

        } catch(Exception $e){

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.providers.index')->with('flash_error', $error);
        }    
    }

    /**
     * @method providers_save()
     *
     * @uses to store/update the provider object based on details
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id , (request) provider details
     *
     * @return view page
     */
    public function providers_save(Request $request) {

        try {

            DB::begintransaction();

            if($request->provider_id != '') {

                $validator = CommonRepo::admin_edit_validation($request->all(), $errors, PROVIDER, $request->provider_id);

            } else {

                $validator = CommonRepo::admin_add_validation($request->all(), $errors, PROVIDER);                
            }
         
            if(!$validator) {

                throw new Exception($errors, 101);
            }

            if($request->provider_id != '') {

                $provider_details = Provider::find($request->provider_id);

                $message = tr('admin_not_provider');

                if($request->hasFile('picture')) {

                    Helper::delete_avatar('uploads/providers',$provider_details->picture);
                }

                if($request->hasFile('promo_video')) {

                    Helper::delete_avatar('uploads/providers',$provider_details->promo_video);
                }

            } else {

                //Add New provider

                $provider_details = new Provider;

                $message = tr('admin_add_provider');

                $provider_details->login_by = 'manual';

                $provider_details->login_status = 'provider';

                $provider_details->register_type = $provider_details->device_type = 'web';

                $provider_details->password = \Hash::make($request->password);
                
                $provider_details->picture = asset('placeholder.png');

                $provider_details->token = Helper::generate_token();

                $provider_details->token_expiry = Helper::generate_token_expiry();

            }    

            if($request->hasFile('picture')) {

                $provider_details->picture = Helper::upload_avatar('uploads/providers' , $request->file('picture'));
            } 

            if($request->hasFile('promo_video')) {

                $provider_details->promo_video = Helper::upload_avatar('uploads/providers' , $request->file('promo_video'));
            }   

            $provider_details->name = $request->has('name') ? $request->name : '';

            $provider_details->email = $request->has('email') ? $request->email: '';

            $provider_details->description = $request->has('description') ? $request->description: '';

            $provider_details->mobile = $request->has('mobile') ? $request->mobile : '';

            if( $provider_details->save() ){

                DB::commit();

            } else {

                throw new Exception(tr('admin_provider_save_error'), 101);
            }

            // check the new register

            if($request->provider_id == '') {
                
                $email_details['name'] = $provider_details->name;

                $email_details['password'] = $request->password;

                $email_details['email'] = $provider_details->email;

                $subject = tr('provider_welcome_title'); $page = "emails.welcome"; $email = $provider_details->email;
                
                if(envfile('MAIL_USERNAME')  && envfile('MAIL_PASSWORD')) 
                {
                    Helper::send_email($page,$subject,$email,$email_details);
                }

                register_mobile('web');

                $provider_details->is_approved = PROVIDER_APPROVED;

                $provider_details->is_available = PROVIDER_AVAILABLE;

                $provider_details->is_verified = PROVIDER_EMAIL_VERIFIED;

                $provider_details->is_activated = $provider_details->status = 1;

                if( $provider_details->save() ){
                
                    DB::commit();

                } else {

                    throw new Exception(tr('admin_provider_save_error'), 101);
                }                
            }
         
            if($provider_details) {

                CommonRepo::become_a_user($provider_details);

                return redirect()->route('admin.providers.view', ['provider_id' => $provider_details->id])->with('flash_success', $message);

            } else {

                throw new Exception( tr('admin_not_error'), 101 );
            }

        } catch(Exception $e){

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error', $error);
        }  

    }

    /**
     * @method providers_view()
     *
     * @uses To display the provider details single view page based on provider id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id
     *
     * @return view page
     */
    public function providers_view(Request $request) {

        try {
            
            $provider_details = Provider::find($request->provider_id);
           
            if(!$provider_details) {
                
                throw new Exception(tr('provider_not_found'), 101);
            }

            $amount_details = get_provider_amount($request->provider_id);

            return view('admin.providers.view')
                        ->with('page','providers')
                        ->with('sub_page','providers-view')
                        ->with('provider_details' , $provider_details)
                        ->with('amount_details', $amount_details);
       

        } catch(Exception $e){

            $error = $e->getMessage();

            return redirect()->route('admin.providers.index')->with('flash_error', $error);
        }
   
    }

    /**
     * @method providers_delete()
     *
     * @uses To delete the provider based on provider id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id
     *
     * @return view page
     */
    public function providers_delete(Request $request) {

        try {

            DB::beginTransaction();

            $provider_details = Provider::find($request->provider_id);

            if(!$provider_details){

                throw new Exception( tr('provider_not_found'), 101);     

            }
              
            // check the provider having any ongoing requests ( on decline only we need to check the below)

            $check_status = [REQUEST_NONE,REQUEST_NO_PROVIDER_AVAILABLE,REQUEST_TIME_EXCEED_CANCELLED,REQUEST_CANCELLED,REQUEST_COMPLETED];

            $check_requests = Requests::where('provider_id' , $request->provider_id)->whereNotIn('status' , $check_status)->count();

            if($check_requests) {

                throw new Exception(tr('admin_provider_decline_ongoing_restrict'), 101);
            }

            // Delete the become user details as well

            $user_details = $provider_details->user_id ? User::find($provider_details->user_id) : "";

            if( $user_details != null ) {

                if( $user_details->forceDelete() ) {

                DB::commit();

                } else {

                    throw new Exception( tr('admin_user_delete_error'), 101);
                }
            }                    

            if( $provider_details->forceDelete() ) {
              
                DB::commit();

                return redirect()->route('admin.providers.index')->with('flash_success',tr('provider_delete_success'));

            } 
            
            return back()->with('flash_success', tr('provider_delete_success') );
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error' , $error);
        }
   
    }

    /**
     * @method providers_status_change()
     *
     * @uses To update provider approve/decline status based on provider id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id
     *
     * @return view page
     */
    public function providers_status_change(Request $request) {

        try {

            DB::beginTransaction();

            $provider_details = Provider::find($request->provider_id);

            if(!$provider_details) {

                throw new Exception(tr('provider_not_found'), 101);
            } 

            if($provider_details->status == PROVIDER_APPROVED) {

                $check_status = [REQUEST_NONE,REQUEST_NO_PROVIDER_AVAILABLE,REQUEST_TIME_EXCEED_CANCELLED,REQUEST_CANCELLED,REQUEST_COMPLETED];

                $check_requests = Requests::where('provider_id' , $request->provider_id)->whereNotIn('status' , $check_status)->count();

                if($check_requests) {

                    throw new Exception(tr('admin_provider_decline_ongoing_restrict'), 101);                            
                }
            }

            $provider_details->is_approved = in_array($provider_details->is_approved, [PROVIDER_STATUS_PENDING ,PROVIDER_DECLINED]) ? PROVIDER_APPROVED : PROVIDER_DECLINED;

            if( $provider_details->save() ) {

                DB::commit();

                $message = $provider_details->is_approved == COMMON_APPROVED ? tr('provider_approve_success') : tr('provider_decline_success');
                
                return back()->with('flash_success', $message);
            } 

            throw new Exception( tr('admin_provider_status_error'), 101);

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.providers.index')->withInput()->with('flash_error' , $error);

        }

    }

    /**
     * @method providers_verify_status()
     *
     * @uses To update user email verification status based on provider id
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id
     *
     * @return view page
     */

    public function providers_verify_status(Request $request) {

        try {

            DB::beginTransaction();
            
            $provider_details = Provider::find($request->provider_id);

            if(!$provider_details) {
                
                throw new Exception(tr('provider_not_found'), 101);

            } 

            $provider_details->is_verified  = $provider_details->is_verified ? PROVIDER_EMAIL_NOT_VERIFIED : PROVIDER_EMAIL_VERIFIED;

            if( $provider_details->save()) {

                DB::commit();

                $message = $provider_details->is_verified == PROVIDER_EMAIL_VERIFIED ? tr('provider_verify_success') : tr('provider_unverify_success');

                return back()->with('flash_success' , $message);

            } 

            throw new Exception(tr('admin_provider_verify_error'), 101);

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.providers.index')->withInput()->with('flash_error' , $error);
        }
    
    }

    /**
     * @method providers_pay()
     *
     * @uses checkout for the selected provider
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return view page
     */

    public function providers_pay(Request $request) {

        try {

            DB::begintransaction();

            $validator = Validator::make($request->all() , [
                'provider_id' => 'required|exists:providers,id',
                'amount' => 'required', 
                ]);

            if($validator->fails()) {

                $error = implode(",", $validator->messages()->all());

                throw new Exception($error, 101);
                
            } 

            $provider_payment = ProviderPayment::where('provider_id' , $request->provider_id)->first();

            if($provider_payment) {

                if($request->payout < 0) {
                    
                    $provider_payment->paid_amount = $provider_payment->paid_amount + $request->amount;

                } else {

                    $provider_payment->paid_amount = $provider_payment->paid_amount + $request->amount;
                }

                $provider_payment->save();

                return back()->with('flash_success' , tr('action_success'));

            }

            throw new Exception(tr('admin_provider_payment_not_found'), 101);
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error' , $error);
        }

    }

    /**
     * @method providers_documents()
     *
     * @uses To get particular provider documents and particular document updated in provider list
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id & $document_id     
     *
     * @return document list page
     */
    public function providers_documents(Request $request) {

        try {

            $provider_details = $document_details = [];

            // get provider documents

            if($request->provider_id) {

                $provider_details = Provider::find($request->provider_id);

                if(count($provider_details) == 0) {

                    throw new Exception( tr('provider_not_found', 101));
                }

                $provider_documents = ProviderDocument::where('provider_id' , $request->provider_id)->paginate(10);

                $page = 'providers';

                $sub_page = 'providers-view';

            } else {

                $document_details = Document::find($request->document_id);

                if(!$document_details) {

                    throw new Exception(tr('document_not_found'), 101);
                }

                $provider_documents = ProviderDocument::where('document_id' , $request->document_id)->paginate(10);

                $page = 'documents';

                $sub_page = 'documents-view';
            }

            return view('admin.providers.documents')
                            ->with('page',$page)
                            ->with('sub_page',$sub_page)
                            ->with('provider_documents', $provider_documents)
                            ->with('provider_details', $provider_details)
                            ->with('document_details', $document_details);
        
        } catch(Exception $e){

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }
    }

    /**
     * @method providers_history()
     *
     * @uses To list out provider history for request completed based on provider id
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id  
     *
     * @return view page
     */

    public function providers_history(Request $request) {

        try {

            $provider_details = Provider::find($request->provider_id);
            
            if(!$provider_details) {

                throw new Exception(tr('provider_not_found'), 101);
                
            } 
           
            $requests = $requests = Requests::where('provider_id' , $request->provider_id)
                                            ->where('status' , REQUEST_COMPLETED)
                                            ->paginate(10);

            return view('admin.providers.history')
                        ->with('page','providers')
                        ->with('sub_page','providers-view')
                        ->with('requests' , $requests)
                        ->with('provider_details' , $provider_details);
        
        } catch(Exception $e){

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }    
    }

    /**
     * @method providers_gallery()
     *
     * @uses To list out provider gallery photos based on provider id
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id  
     *
     * @return view page
     */

    public function providers_gallery(Request $request) {

        try {

            $provider_details = Provider::find($request->provider_id);

            if(!$provider_details) {

                throw new Exception(tr('provider_not_found'), 101);
            }

            $galleries = ProviderGallery::where('provider_id' , $request->provider_id)->get();

            return view('admin.providers.galleries')
                        ->with('page' , 'providers')
                        ->with('sub_page' , 'providers-view')
                        ->with('provider_details' , $provider_details)
                        ->with('galleries' , $galleries);

        } catch(Exception $e){

            $error = $e->getMessage();

            return redirect()->route('admin.providers.index')->with('flash_error', $error);
        }    
    }

    /**
     * @method providers_gallery_save()
     *
     * @uses To save the gallery image based on provider id
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id  
     *
     * @return view page
     */

    public function providers_gallery_save(Request $request) {

        try {

            DB::begintransaction();

            $validator = Validator::make($request->all() , [
                        'picture' => 'required|mimes:jpg,jpeg,png',
                        'provider_id' => 'required|exists:providers,id'
                    ]);

            if($validator->fails()) {

                $error = implode(',', $validator->messages()->all());

                throw new Exception($error, 101);             

            }

            $gallery = new ProviderGallery;

            $gallery->provider_id = $request->provider_id;

            $gallery->picture = Helper::upload_avatar('uploads/providers/' , $request->file('picture'));

            $gallery->status = APPROVED;

            if( $gallery->save() ) {

                DB::commit();

                return back()->with('flash_success' , tr('provider_gallery_success'));
            
            } 

            throw new Exception(tr('admin_provider_gallery_save_error'), 101);

        } catch(Exception $e){

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }
    }

    /**
     * @method providers_services()
     *
     * @uses To display provider services list based on provider id
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id  
     *
     * @return success/error message
     */

    public function providers_services(Request $request) {

        try {
        
            $provider_details = Provider::find($request->provider_id);

            if(!$provider_details) {

                throw new Exception(tr('user_not_found'), 101);
            } 

            $provider_services = ProviderService::where('provider_id', $request->provider_id)->where('status' , APPROVED)->paginate(10);

            if(!$provider_services) {

                throw new Exception(tr('provider_service_not_found'), 101);
            }

            return view('admin.providers.services')
                        ->with('page','providers')
                        ->with('sub_page','providers-view')
                        ->with('provider_details',$provider_details)
                        ->with('provider_services',$provider_services);
        

        } catch( Exception $e){
            
            $error = $e->getMessage();

            return redirect()->back()->with('flash_error',$error);
        }

    }

    /**
     * Function: categories_index()
     *
     * @uses To list out categories.
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param -
     *
     * @return view page
     */
    public function categories_index() {

        $categories = Category::orderBy('created_at','desc')->paginate(10);

        return view('admin.categories.index')
                        ->with('page','categories')
                        ->with('sub_page','categories-view')
                        ->with('categories' , $categories);
    }

    /**
     * @method categories_create()
     *
     * @uses To create category object
     *
     * @created vidhya
     *
     * @updated Anjana H
     *
     * @param 
     * 
     * @return view page
     *
     */

    public function categories_create() {

        $category_details = new Category;

        return view('admin.categories.create')
                    ->with('page' , 'categories')
                    ->with('sub_page','categories-create')
                    ->with('category_details',$category_details);
    }

    /**
     * @method categories_edit()
     *
     * @uses To display the catgeory details based on category id 
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $category_id  
     * 
     * @return view page
     */

    public function categories_edit(Request $request) {

        try {

            $category_details = Category::find($request->category_id);

            if(!$category_details) {

                throw new Exception(tr('category_not_found'), 101);
            }

            return view('admin.categories.edit')
                    ->with('page' , 'categories')
                    ->with('sub_page','categories-view')
                    ->with('category_details',$category_details);

        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->route('admin.categories.index')->with('flash_error', $error);
        }

    }

    /**
     * @method categories_save()
     *
     * @@uses To save/update the category details based on category id
     *
     * @created Aravinth R
     *
     * @updated Anjana H
     * 
     * @param Integer (request) $category_id  , (request) details
     *
     * @return view page
     */
    public function categories_save(Request $request) {

        try {

            DB::beginTransaction();

            if($request->category_id != '') {

                $validator = CommonRepo::admin_category_edit_validation($request->all(), $errors, CATEGORY,$request->category_id);

            } else {

                $validator = CommonRepo::admin_category_add_validation($request->all(), $errors, CATEGORY);
            }
      
             if(!$validator) {

                throw new Exception($errors, 101);
                
            }

            $message = $request->category_id ? tr('category_update_success') : tr('category_add_success');

            if( $request->category_id != '') {

                $category_details = Category::find($request->category_id);

                if($request->hasFile('picture') && $category_details->picture) {

                    Helper::delete_avatar('uploads/category' , $category_details->picture);
                }

            } else{

                $category_details = new Category;

                $category_details->status = COMMON_APPROVED;
           
            }

            $message = $request->category_id ? tr('category_update_success') : tr('category_add_success');

            if( $request->hasFile('picture') ) {
               
                $category_details->picture = Helper::upload_avatar('uploads/category' ,$request->file('picture'));
            }

            $category_details->name = $request->has('name') ? $request->name : '';
            
            $category_details->description = $request->has('description') ? $request->description : '';
            
            if( $category_details->save() ) {

                DB::commit();

                return redirect()->route('admin.categories.index')->with('flash_success',$message);
            }

            throw new Exception(tr('admin_category_save_error'), 101);
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error' , $error);
        }
    }

    /**
     * @method categories_delete()
     *
     * @uses To delete the category based on category id
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     * 
     * @param Integer (request) $category_id
     * 
     * @return view page
     */    
    public function categories_delete(Request $request) {

        try {

            DB::beginTransaction();

            $category_details = Category::find($request->category_id);

            if(!$category_details) {

                throw new Exception(tr('category_not_found'), 101);
            } 

           if( $category_details->forceDelete() ) {

                DB::commit();

                return back()->with('flash_success',tr('category_delete_success'));
            } 

            throw new Exception(tr('admin_category_delete_error'), 101);
      
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error' , $error);
        }    
    }


    /**
     * @method categories_status_change()
     *
     * @uses To update category approved/decline status based on category id
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     * 
     * @param Integer (request) $category_id 
     *
     * @return view page
     */

    public function categories_status_change(Request $request) {

        try {

            DB::beginTransaction();

            $category_details = Category::find($request->category_id);

            if(!$category_details) {
                
                throw new Exception(tr('category_not_found'), 101);
            } 

            $category_details->status = in_array($category_details->status, [COMMON_PENDING ,COMMON_DECLINED]) ? COMMON_APPROVED : COMMON_DECLINED;

            if( $category_details->save()) {

                DB::commit();

                // Change the 
                if ($category_details->status != COMMON_APPROVED) {

                    foreach($category_details->subCategory as $sub_category) {

                        $sub_category->status = COMMON_DECLINED;

                        if( $sub_category->save() ) {

                            DB::commit();

                        } else {

                            throw new Exception(tr('admin_sub_category_status_error'), 101);
                        }
                    } 
                }

                $message = $category_details->status == COMMON_APPROVED ? tr('category_approve_success') : tr('category_decline_success');
        
                return back()->with('flash_success', $message);

            } 

            throw new Exception(tr('admin_category_status_error'), 101);
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.categories.index')->withInput()->with('flash_error' , $error);
        }
    }

    /**
     * @method categories_sub_category()
     *
     * @uses To display the sub catgeories list based on the category id
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     * 
     * @param Integer (request) $category_id 
     *
     * @return view page
     */
    public function categories_sub_category(Request $request) {

        try {

            $category_details = Category::find($request->category_id);

            if(!$category_details) {
                
                throw new Exception(tr('category_not_found'), 101);
            } 

            $sub_categories = SubCategory::where('category_id' , $request->category_id)
                            ->select(
                                    'sub_categories.id as id',
                                    'sub_categories.name as sub_category_name',
                                    'sub_categories.description',
                                    'sub_categories.status',
                                    'sub_categories.picture',
                                    'sub_categories.price'
                                    )
                            ->orderBy('sub_categories.created_at', 'desc')
                            ->paginate(10);

            return view('admin.categories.sub-categories')
                            ->with('page','categories')
                            ->with('sub_page','categories-view')
                            ->with('category_details' , $category_details)
                            ->with('sub_categories' , $sub_categories);

        } catch(Exception $e){

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }  
    }

    /**
     * @method sub_categories_index()
     *
     * @uses to list out sub_categories details
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     *
     * @param -
     *  
     * @return view page
     */
    public function sub_categories_index(Request $request) {

        try {

            $category_details = [];

            $query =  SubCategory::orderBy('created_at','desc');

            if($request->category_id) {

                $category_details = Category::find( $request->category_id );

                $query = $query->where('category_id' , $request->category_id );
            }

            $sub_category = $query->paginate(10);

            return view('admin.sub_categories.index')
                            ->with('page','sub_categories')
                            ->with('sub_page','sub_categories-view')
                            ->with('sub_category' , $sub_category)
                            ->with('category_details' , $category_details);

        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->route('admin.sub_categories.index')->with('flash_error', $error);
        }
    }

    /**
     * @method sub_categories_create()
     *
     * @uses To create a sub_category details
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     *
     * @param -
     * 
     * @return view page
     */
    public function sub_categories_create() {

        $sub_category_details = new SubCategory;

        $category = Category::where('status' , COMMON_APPROVED)
                        ->select('categories.id as category_id' , 'categories.name as category_name' , 'categories.description as category_description' , 'categories.picture as category_picture' , 'categories.unique_id' , 'categories.status')
                        ->get();

        return view('admin.sub_categories.create')
                    ->with('page' , 'sub_categories')
                    ->with('sub_page','sub_categories-create')
                    ->with('categories',$category)
                    ->with('sub_category_details',$sub_category_details);

    }

    /**
     * @method sub_categories_edit()
     *
     * @uses To display and update user details based on the sub category id
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     * 
     * @param Integer (request) $sub_category_id 
     * 
     * @return view page
     */
    public function sub_categories_edit(Request $request) {

        try {

            $sub_category_details = SubCategory::find($request->sub_category_id);

            if(!$sub_category_details) {
                
                throw new Exception(tr('sub_category_not_found'), 101);
            } 

            $category = Category::where('status' , COMMON_APPROVED)
                    ->select('categories.id as category_id' , 'categories.name as category_name' , 'categories.description as category_description' , 'categories.picture as category_picture' , 'categories.unique_id' , 'categories.status')
                    ->get();

            return view('admin.sub_categories.edit')
                    ->with('page' , 'sub_categories')
                    ->with('sub_page','sub_categories-view')
                    ->with('sub_category_details',$sub_category_details)
                    ->with('categories',$category);

        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error',$error);
        }  
   
    }
    
    /**
     * Function: sub_categories_save()
     *
     * @use To save/update the subcategory object details of new/existing based on details
     * 
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $sub_category_id ,(request) details
     *
     * @return success/error message
     */
    public function sub_categories_save(Request $request) {

        try {

            DB::beginTransaction();

            if($request->sub_category_id != '') {

                $validator = CommonRepo::admin_category_edit_validation($request->all(), $errors, SUB_CATEGORY,$request->sub_category_id);

            } else {

                $validator = CommonRepo::admin_category_add_validation($request->all(), $errors, SUB_CATEGORY);
            }
       
            if(!$validator) {
                throw new Exception($errors, 101);
            }

            $message = $request->sub_category_id ? tr('sub_category_update_success') : tr('sub_category_add_success');

            if($request->sub_category_id != '') {

                $sub_category = SubCategory::find($request->sub_category_id);

                if($sub_category->picture && $request->hasFile('picture')) {

                    Helper::delete_avatar('uploads/sub-category' , $sub_category->picture);
                }

            } else {

                $sub_category = new SubCategory;

                $sub_category->status = COMMON_APPROVED;
            
            }

            if($request->hasFile('picture')) {
            
                $sub_category->picture = Helper::upload_avatar('uploads/sub-category' , $request->file('picture'));
            }

            $sub_category->category_id = $request->category_id;

            $sub_category->name = $request->has('name') ? $request->name : '';

            $sub_category->description = $request->has('description') ? $request->description : '';

            $sub_category->price = $request->price != 0 ? $request->price: 0.1;

            if($sub_category->save()) {

                DB::commit();

                return back()->with('flash_success',$message);

                $sub_category->status = COMMON_APPROVED;
            
            }

            if($request->hasFile('picture')) {
            
                $sub_category->picture = Helper::upload_avatar('uploads/sub-category' , $request->file('picture'));
            }

            $sub_category->category_id = $request->category_id;

            $sub_category->name = $request->has('name') ? $request->name : '';

            $sub_category->description = $request->has('description') ? $request->description : '';

            $sub_category->price = $request->price != 0 ? $request->price: 0.1;

            if($sub_category->save()) {

                DB::commit();

                return redirect()->route('admin.sub_categories.index')->with('flash_success',$message);
            } 
            
            throw new Exception(tr('admin_sub_category_save_error'), 101);
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error' , $error);
        }
   
    }

    /**
     * @method sub_categories_delete()
     *
     * @uses To delete the sub category based on sub category id
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     * 
     * @param Integer (request) $sub_category_id
     * 
     * @return view page
     */    
    public function sub_categories_delete(Request $request) {

        try {

            DB::beginTransaction();

            $sub_category_details = SubCategory::find($request->sub_category_id);

            if(!$sub_category_details) {

                throw new Exception(tr('sub_category_not_found'), 101);
            } 

           if( $sub_category_details->forceDelete() ) {

                DB::commit();

                return back()->with('flash_success',tr('sub_category_delete_success'));

            } 

            throw new Exception(tr('admin_sub_category_delete_error'), 101);
      
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error' , $error);
        }    
    }

    /**
     * @method sub_categories_status_change()
     *
     * @uses To update sub_category status to approve/decline status based on sub category id
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     * 
     * @param Integer (request) $sub_category_id 
     * 
     * @return view page
     */    
    public function sub_categories_status_change(Request $request) {

        try {

            DB::beginTransaction();
         
            $sub_category_details = SubCategory::find($request->sub_category_id);

            if(!$sub_category_details) {
                
                throw new Exception(tr('sub_category_not_found'), 101);
            }

            $sub_category_details->status = in_array($sub_category_details->status, [COMMON_PENDING ,COMMON_DECLINED]) ? COMMON_APPROVED : COMMON_DECLINED;

            if( $sub_category_details->save() ) { 
                
                DB::commit();

                $message = $sub_category_details->status == COMMON_APPROVED ? tr('sub_category_approve_success') : tr('sub_category_decline_success');
        
                return back()->with('flash_success', $message);

            } 

            throw new Exception(tr('admin_sub_category_status_error'), 101);
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.sub_categories.index')->withInput()->with('flash_error' , $error);
        }
    
    }

    // Requests

    public function requests_ondemand(Request $request) {

        $requests = Requests::where('job_type' , REQUEST_NOW)
                        ->orderBy('updated_at' , 'desc')
                        ->paginate(10);

        return view('admin.requests.ondemand')
                        ->with('page','requests')
                        ->with('sub_page','ondemand-requests')
                        ->withRequests($requests);

    }

    public function requests_bidding(Request $request) {

        $requests = Requests::where('job_type' , REQUEST_LATER)
                        ->orderBy('requests.updated_at' , 'desc')
                        ->paginate(10);

        $requests->is_cancelled = YES;
       
        return view('admin.requests.bidding')
                    ->with('page','requests')
                    ->with('sub_page','bidding-requests')
                    ->withRequests($requests);
    }

    /**
     * @method requests_payments()
     *
     * @uses list out request payments 
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param 
     *
     * @return view page
     */
    public function requests_payments(Request $request) {
        
        $base_query = RequestPayment::latest()
                                ;
        if($request->provider_id) {
            
            $provider_details = Provider::find($request->provider_id);

            if(!$provider_details){

                throw new Exception( tr('provider_not_found'), 101);
            } 

            $base_query = $base_query->where('provider_id',$request->provider_id);
        }

        $request_payments = $base_query->paginate(20);

        return view('admin.revenues.requests_payments')
                        ->with('page','payments')
                        ->with('sub_page','request-payments')
                        ->with('request_payments' , $request_payments); 
    
    }

    /**
     * @method requests_view()
     *
     * @uses To display single request details
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return View page
     */
    public function requests_view(Request $request) {

        try {

            $request_details = Requests::where('unique_id' , $request->request_unique_id)->first();

            if(!$request_details) {

                throw new Exception(tr('request_not_found'), 101);
            }

            $request_payment_details = $request_details->payments ? $request_details->payments : new RequestPayment;

            return view('admin.requests.view')
                        ->with('page','requests')
                        ->with('sub_page','')
                        ->with('request_details' , $request_details)
                        ->with('request_payment_details' , $request_payment_details);

        } catch(Exception $e){

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }
      
    }

    /**
     * @method requests_fav_providers()
     *
     * @uses list the favorite providers
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return View page
     */
    public function requests_fav_providers() {

        $fav_providers = UserFavProvider::leftJoin('requests' ,'requests.id' , "=" , 'user_fav_providers.request_id')
                            ->leftJoin('providers' ,'providers.id' , "=" , 'user_fav_providers.provider_id')
                            ->select('providers.name as provider_name' , 'providers.picture as provider_picture','providers.mobile as provider_mobile' ,'providers.email as provider_email' , 'user_fav_providers.id as user_fav_provider_id', 'user_fav_providers.user_id','user_fav_providers.request_id','user_fav_providers.provider_id','requests.name as request_name','requests.unique_id as request_unique_id' , 'user_fav_providers.created_at')
                            ->get();

        return view('admin.users.fav_providers')
                        ->with('page','requests')
                        ->with('sub_page','fav-providers')
                        ->with('user_details' , [])
                        ->with('fav_providers' , $fav_providers);
    }

    /**
     * @method provider_reviews()
     *
     * @uses Provider reviews for the requests
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param -
     *
     * @return view page
     */
    public function provider_reviews() {

        $provider_reviews = ProviderRating::leftJoin('providers', 'provider_ratings.provider_id', '=', 'providers.id')
            ->leftJoin('users', 'provider_ratings.user_id', '=', 'users.id')
            ->select('provider_ratings.*','provider_ratings.id as review_id', 'provider_ratings.rating', 'provider_ratings.comment', 'users.name as user_name', 'providers.name as provider_name', 'users.id as user_id', 'providers.id as provider_id', 'provider_ratings.created_at')
            ->orderBy('provider_ratings.created_at', 'desc')
            ->paginate(10);

        return view('admin.reviews')
                    ->with('page','rating-review')
                    ->with('sub_page','providers-review')
                    ->with('reviews', $provider_reviews);
    }

    /**
     * @method provider_reviews_delete()
     *
     * @uses delete the selected provider review
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - Integer (request) $provider_rating_id
     *
     * @return view page
     */
    public function provider_reviews_delete(Request $request) {

        try {

            DB::begintransaction();

            $provider_review_details = ProviderRating::find($request->provider_rating_id);

            if(!$provider_review_details) {
                
                throw new Exception(tr('admin_provider_reviews_not_found'), 101);
            } 
               
            if($provider_review_details->delete()) {

                DB::commit();

                return back()->with('flash_success', tr('provider_review_delete_success')); 

            } 

            throw new Exception( tr('admin_provider_reviews_delete_error'), 101);            

        } catch(Exception $e){

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }
    }


    /**
     *
     * @method user_reviews()
     *
     * @uses user reviews for the requests
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param -
     *
     * @return view page
     */
    public function user_reviews() {

        $user_reviews = UserRating::leftJoin('providers', 'user_ratings.provider_id', '=', 'providers.id')
                ->leftJoin('users', 'user_ratings.user_id', '=', 'users.id')
                ->select('user_ratings.*','user_ratings.id as review_id', 'user_ratings.rating', 'user_ratings.comment', 'users.name as user_name', 'providers.name as provider_name','users.id as user_id', 'providers.id as provider_id', 'user_ratings.created_at')
                ->orderBy('user_ratings.updated_at', 'desc')
                ->paginate(10);

        return view('admin.reviews')
                    ->with('page','rating-review')
                    ->with('sub_page','users-review')
                    ->with('name', 'User')
                    ->with('reviews', $user_reviews);
    
    }

    /**
     * @method user_reviews_delete()
     *
     * @uses To delete user review based on user rating id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - Integer (request) $user_rating_id
     *
     * @return view page
     */

    public function user_reviews_delete(Request $request) {

        try {

            DB::begintransaction();

            $user_review_details = UserRating::find($request->user_rating_id);

            if(!$user_review_details) {

                throw new Exception(tr('user_review_not_found'), 101);
            } 
           
            if( $user_review_details->delete() ){

                DB::commit();

                return back()->with('flash_success', tr('user_review_delete_success'));
            }  
      
            throw new Exception(tr('admin_user_reviews_delete_error'), 101);
      
        } catch(Exception $e){

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }

    }

    /**
     * @method settings()
     *
     * @uses settings view page
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return view page
     */

    public function settings() {

        $settings = array();

        //  NOT USING NOW

        $symbol = $currency = "";

        $languages = Language::where('status', DEFAULT_TRUE)->get();

        return view('admin.settings.settings')
                    ->with('page','settings')
                    ->with('sub_page','')
                    ->with('symbol',$symbol)
                    ->with('currency',$currency)
                    ->with('languages',$languages)
                    ->with('settings' , $settings); 
   
    }

    /**
     * @method settings_save()
     *
     * @uses update the site settings
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return view page
     */

    public function settings_save(Request $request) {

        try {

            DB::begintransaction();

            $settings = Settings::all();

            $refresh = "";
            
            foreach ($settings as $setting) {

                $key = $setting->key;
               
                $temp_setting = Settings::find($setting->id);

                if($temp_setting->key == 'site_icon') {

                    $site_icon = $request->file('site_icon');

                    if($site_icon == null) {
                        $icon = $temp_setting->value;
                    } else {

                        if($temp_setting->value) {
                            Helper::delete_avatar( 'uploads/site' , $temp_setting->value);
                        }

                        $icon = Helper::upload_avatar('uploads/site',$site_icon);
                    }


                    $temp_setting->value = $icon;
                    
                } else if($temp_setting->key == 'site_logo') {

                    $picture = $request->file('site_logo');

                    if($picture == null){
                        $logo = $temp_setting->value;
                    } else {
                        if($temp_setting->value) {
                            Helper::delete_avatar( 'uploads/site' , $temp_setting->value);

                        }
                        $logo = Helper::upload_avatar('uploads/site',$picture);
                    }

                    $temp_setting->value = $logo;

                }  else if($temp_setting->key == 'site_name') {

                    $temp_setting->value = $request->$key ? $request->$key : $temp_setting->value;

                    \Enveditor::set('SITENAME' , $request->$key);

                } else if($setting->key == 'default_lang') {

                    if ($request->default_lang != $temp_setting->value) {

                        $refresh = $request->default_lang;

                    }

                    $temp_setting->value = $request->$key;

                } else if($request->$key != "") {

                    $temp_setting->value = $request->$key ? $request->$key : "";
                }

                if ( $temp_setting->save() ){

                    DB::commit();

                } else {

                    throw new Exception(tr('admin_settings_save_error'), 101);
                }                
            }

            if ($refresh) {

                $fp = fopen(base_path() .'/config/new_config.php' , 'w');
                fwrite($fp, "<?php return array( 'locale' => '".$refresh."', 'fallback_locale' => '".$refresh."');?>");
                fclose($fp);
                \Log::info("Key : ".config('app.locale'));
                
            }

            return redirect(route('clear-cache'));    

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }

    }

    /**
     * @method common_settings_save()
     *
     * @uses save the values in env file
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - object $request Post Attribute values
     *
     * @return settings values
     */
    
    public function common_settings_save(Request $request) {

        try {

            $admin_id = \Auth::guard('admin')->user()->id;

            foreach ($request->all() as $key => $data) {

                $temp_setting = Settings::where('key', $key)->first();

                if ($temp_setting) {

                     $temp_setting->value = $data;

                     if( $temp_setting->save() ) {

                        DB::commit();

                     } else {

                        throw new Exception(tr('admin_settings_save_error'), 101);
                        
                     }

                } else {

                    if($request->has($key)) {
                        \Enveditor::set($key,$data);
                    }
                }
            }

            $result = Helper::getEnvValues();

            return redirect(route('clear-cache'))
                    ->with('result' , $result)
                    ->with('flash_success' , tr('common_settings_success'));

        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }

    }

    /**
     * @method help()
     *
     * @uses help page
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return view page
     */

    public function help() {

        return view('admin.static.help')
                ->with('page','help')
                ->with('sub_page' , "");
    }

    /**
     * @method static_pages_index()
     *
     * @uses list out static pages details
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return view page
     */

    public function static_pages_index() {

        $pages = Page::orderBy('updated_at' , 'desc')->paginate(10);

        return view('admin.pages.index')
                ->with('page','static-pages')
                ->with('sub_page',"static-pages-view")
                ->with('pages',$pages);
    }

    /**
     * @method static_pages_create()
     *
     * @uses To create a static page details
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return view page
     */

    public function static_pages_create() {

        $page_details = new Page;

        $all_pages = Page::all();

        $static_keys = ['about' , 'contact' , 'privacy' , 'terms' , 'help', 'others','how_it_works_user','faq'];

        $pages = [];

        foreach ($all_pages as $key => $page) {

            if ($page->type != 'others') {

                $pages[] = $page->type;

            }

        }

        return view('admin.pages.create')
                ->with('page','static-pages')
                ->with('sub_page',"static-pages-create")
                ->with('view_pages',$pages)
                ->with('static_keys', $static_keys)     
                ->with('page_details', $page_details);
    
    }

    /**
     * @method static_pages_create()
     *
     * @uses To display and update page details based on the static page id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return view page
     */

    public function static_pages_edit(Request $request) {

        try {

            $page_details = Page::find($request->static_page_id);

            if(!$page_details){

                throw new Exception(tr('admin_not_error'), 101);
            } 

            return view('admin.pages.edit')
                    ->with('page','static-pages')
                    ->with('sub_page','static-pages-view')
                    ->with('page_details', $page_details);

        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->route('admin.static_pages.index')->with('flash_error', $error);
        }
    }

    /**
     * @method static_pages_save()
     *
     * @uses store/update the page details
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return  success/error message
     */

    public function static_pages_save(Request $request) {
        
        try {

            DB::begintransaction();

            $validator = Validator::make($request->all(),[
                    'heading' => 'required|max:100',
                    'description' => 'required'
                ]
            );

            if($validator->fails()) {

                $error = implode(',',$validator->messages()->all());

                throw new Exception($error, 101);               

            }

            // Check the page type 

            $unique_id = preg_replace("/[^A-Za-z0-9]/", "-", $request->heading);

            $check_page_exists = Page::where('unique_id' , $unique_id)->count();

            if( $check_page_exists ) {

                $unique_id = $unique_id."-".rand();
            }

            if($request->static_page_id) {

                $page_details = Page::find($request->static_page_id);

                $page_details->heading = $request->heading ? $request->heading : $page_details->heading;

                $page_details->unique_id = $unique_id;

                $page_details->description = $request->description;

                $message = tr('page_edit_success');

            }  else {

                $check_page = "";

                if ($request->type != 'others') {

                    $check_page = Page::where('type',$request->type)->first();
                }
                
                if(!$check_page) {

                    $page_details = new Page;

                    $page_details->unique_id = $unique_id;

                    $page_details->type = $request->type ? $request->type : "others";

                    $page_details->heading = $request->heading ? $request->heading : "";

                    $page_details->description = $request->description ? $request->description : "";

                    $page_details->status = APPROVED;

                    $message = tr('page_create_success');

                } else {

                    throw new Exception(tr('page_already_alert'), 101);
                }
            }
            
            if( $page_details->save() ) {

                DB::commit();

                return back()->with('flash_success',$message);

            } else{

                throw new Exception(tr('admin_page_save_error'), 101);             
            }

           
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error', $error);

        }
    
    }


    /**
     * @method static_pages_delete()
     *
     * @uses To delete the page details based on static page id
     *
     * @created Vidhya R 
     *
     * @updated Anjana H
     *
     * @param - Integer (request) $static_page_id
     *
     * @return success/error message
     */
    public function static_pages_delete(Request $request) {

        try {

            DB::begintransaction();

            $page_details = Page::find($request->static_page_id);

            if(!$page_details) {

                throw new Exception(tr('page_not_found'), 101);
            } 

            if( $page_details->delete() ) {

                DB::commit();

                return redirect()->route('admin.static_pages.index')->with('flash_success',tr('page_delete_success'));

            } 

            throw new Exception(tr('admin_page_delete_error'), 101);
            
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.static_pages.index')->with('flash_error', $error);
        }

    }

    /**
     * @method documents_index()
     *
     * @uses list out documents details 
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return redirect to view page
     */

    public function documents_index() {

        $documents = Document::orderBy('updated_at' , 'desc')->paginate(10);

        return view('admin.documents.index')
                        ->with('page','documents')
                        ->with('sub_page','documents-view')
                        ->with('documents',$documents);
    }

    /**
     * @method documents_create()
     *
     * @uses To display the create document page 
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return redirect to view page
     */

    public function documents_create() {
        
        $document_details = new Document;

        return view('admin.documents.create')
                    ->with('page','documents')
                    ->with('sub_page','documents-create')
                    ->with('document_details',$document_details);
    }

    /**
     * @method documents_edit()
     *
     * @uses To display and update document details based on document id
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $document_id
     * 
     * @return redirect to view page
     */

    public function documents_edit(Request $request) {

        try {
        
            $document_details = Document::find($request->document_id);

            if(!$document_details){


                throw new Exception(tr('document_not_found'));
            } 

            return view('admin.documents.edit')
                ->with('page','documents')
                ->with('sub_page','documents-view')
                ->with('name', tr('edit_document'))
                ->with('document_details',$document_details);
     
        } catch( Exception $e) {
            
            $error=$e->getMessage();

            return redirect()->route('admin.documents.index')->with('flash_error',$error);
        }
    }

    /**
     * @method documents_save()
     *
     * @uses To save the documents object details of new/existing based on details
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $document_id,(request) details
     * 
     * @return redirect to view page
     */

    public function documents_save(Request $request) {

        try {

            DB::beginTransaction();

            $validator = Validator::make(
                $request->all(),
                array(
                    'name' => 'required|max:255'
                )            
            );

            if($validator->fails()) {

                $error = implode(',',$validator->messages()->all());

                throw new Exception($error, 101);

            } 

            if($request->document_id != '') {

                $document_details = Document::find($request->document_id);
               
                $message = tr('documents_update_success');

            } else {

                $document_details = new Document;

                $message = tr('documents_create_success');

                $document_details->status = APPROVED;
            }

            $document_details->name = $request->name;

            if( $document_details->save() ) {

                DB::commit();

                return back()->with('flash_success',$message);

            }

            throw new Exception(tr('admin_document_save_error'), 101);
           
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error' , $error);

        }
    }

    /**
     * @method document_edit()
     *
     * @uses To delete document object based on documeny id
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $document_id
     *  
     * @return redirect to view page
     */
    public function documents_delete(Request $request) {

        try {

            DB::begintransaction();

            $document_details = Document::find($request->document_id);

            if(!$document_details) {

                throw new Exception(tr('document_not_found'), 101);
            } 

            if( $document_details->delete() ){

                DB::commit();

                return back()->with('flash_success',tr('documents_delete_success'));

            } 

            throw new Exception(tr('admin_document_delete_error'), 101);

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.documents.index')->with('flash_error', $error);
        }       
    }

    /**
     * @method documents_status_change()
     *
     * @uses used for change the status of the selected document 
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     *
     * @param request document_id
     *  
     * @return view page
     */
    
    public function documents_status_change(Request $request) {

        try {

            DB::beginTransaction();

            $document_details = Document::find($request->document_id);

            if(!$document_details) {

                throw new Exception(tr('document_not_found'), 101);
            }

            $document_details->status = in_array($document_details->status, [COMMON_PENDING ,COMMON_DECLINED]) ? COMMON_APPROVED : COMMON_DECLINED;

            if( $document_details->save() ){

                DB::commit();

                $message = $document_details->status == COMMON_APPROVED ? tr('document_approve_success') : tr('document_decline_success');
        
                return back()->with('flash_success', $message); 
            
            }

            throw new Exception(tr('admin_document_status_error'), 101);         

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.documents.index')->withInput()->with('flash_error' , $error);

        }    
    }

    /**
     * @method banners_index()
     *
     * @uses to list out banners details
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     *
     * @param - 
     *  
     * @return view page
     */

    public function banners_index() {

        $banners = BannerImage::orderBy('created_at' , 'desc')->paginate(10);

        return view('admin.banners.index')
                    ->with('page','banners')
                    ->with('sub_page','banners-view')
                    ->with('banners',$banners);
    }

    /**
     * @method banners_create()
     *
     * @uses to create a banner object
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     *
     * @param -
     *  
     * @return redirect to view page
     */
    public function banners_create() {
        
        $bannerimage_details = new BannerImage;

        return view('admin.banners.create')
                ->with('page','banners')
                ->with('sub_page','banners-create')
                ->with('bannerimage_details', $bannerimage_details);
    }

    /**
     * @method banners_edit()
     *
     * @uses to display and update the banner details based on banner id
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     * 
     * @param Integer (request) $banner_id
     * 
     * @return view page
     */
    public function banners_edit(Request $request) {

        try {
            
            $bannerimage_details = BannerImage::find($request->banner_id);

            if(!$bannerimage_details){

                throw new Exception(tr('bannerimage_not_found'), 101);
            } 
    
            return view('admin.banners.edit')
                    ->with('page','banners')
                    ->with('sub_page','banners-view')
                    ->with('name', 'Edit bannerimage')
                    ->with('bannerimage_details',$bannerimage_details);

        } catch( Exception $e) {
            
            $error=$e->getMessage();

            return redirect()->route('admin.banners.index')->with('flash_error',$error);
        }
    }

    /**
     * @method banners_save()
     *
     * @uses to save/update banner object based on details
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     *
     * @param Integer (request) $banner_id, (request) details
     * 
     * @return view page
     */

    public function banners_save(Request $request) {

        try {

            DB::beginTransaction();

            $validator = Validator::make(
                $request->all(),
                array(
                    'picture' => 'required|mimes:jpeg,jpg,bmp,png',
                    'banner_title' => 'string|max:100',
                    'banner_text' => 'string|max:150',
                ), 
                [
                    'banner_title.max' => 'Sorry!,Maximum 100 letters allowed',
                    'banner_text.max' => 'Sorry!,Maximum 150 letters allowed'
                ]            
            );

            if($validator->fails()) {

                $error = implode(',',$validator->messages()->all());

                throw new Exception($error, 101);
            } 

            $message = $request->banner_id ? tr('banner_update_success') : tr('banner_create_success');

            if($request->banner_id != '') {

                $banner_details = BannerImage::find($request->banner_id);

                $banner_details->banner_text = $request->banner_text ? $request->banner_text :"";

                $banner_details->banner_title = $request->banner_title ? $request->banner_title : "";

                if($request->hasFile('picture')) {

                    if($banner_details->picture) {
                        
                        Helper::delete_avatar( 'uploads/bannerimage' , $banner_details->value);
                    }

                    $banner_details->picture = Helper::upload_avatar('uploads/bannerimage' ,$request->file('picture'));
                }

            } else {

                $banners_limit = Setting::get('banners_limit');

                if(BannerImage::count() <= $banners_limit) {

                    $banner_details = new BannerImage;

                    $banner_details->banner_title = $request->banner_title ? $request->banner_title : "";

                    $banner_details->banner_text = $request->banner_text ? $request->banner_text : "";

                    $banner_details->status = COMMON_APPROVED;

                    $banner_details->picture = $request->hasFile('picture') ? Helper::upload_avatar('uploads/bannerimage' , $request->file('picture')) : Setting::get('site_logo');

                } else {

                    throw new Exception(tr('maximum_limit_reached'), 101);            
                }                    
            }

            if( $banner_details->save() ) {

                DB::commit();

                return back()->with('flash_success',$message);

            } 

            throw new Exception(tr('admin_banner_save_error'), 101);

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error' , $error);
        }    
    }

    /**
     * @method banners_delete()
     *
     * @uses to delete the banner object based on banner id
     *
     * @created Vidhy R
     *
     * @updated Anjana
     * 
     * @param integer banner_id
     * 
     * @return view page
     */
    
    public function banners_delete(Request $request) {

        try {

            DB::beginTransaction();
        
            $banner_details = BannerImage::find($request->banner_id);

            if( count($banner_details) == 0 ) {

                throw new Exception(tr('admin_banner_delete_error'), 101);

            } 

            if ( $banner_details->delete() ) {
                
                DB::commit();

                return back()->with('flash_success',tr('banner_delete_success'));
            
            } 

            throw new Exception(tr('admin_banner_delete_error'), 101);
               
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.banners.index')->withInput()->with('flash_error' , $error);
        }
    
    }

    /**
     * @method banners_status_change()
     *
     * @uses used for change the status of the banner based on banner id
     *
     * @created Vidhya R
     *
     * @updated Anjana H
     * 
     * @param integer (request) $banner_id
     * 
     * @return view page
     */    
    public function banners_status_change(Request $request) {

        try {
           
            $banner_details = BannerImage::find($request->banner_id);

            if(!$banner_details) {

                throw new Exception(tr('bannerimage_not_found'), 101);  
              
            } 

            $banner_details->status = in_array($banner_details->status, [COMMON_PENDING ,COMMON_DECLINED]) ? COMMON_APPROVED : COMMON_DECLINED;

            if( $banner_details->save()) {

                DB::commit();

                $message = $banner_details->status == COMMON_APPROVED ? tr('banner_approve_success') : tr('banner_decline_success');
        
                return back()->with('flash_success', $message);

            }

            throw new Exception(tr('admin_banner_status_error'), 101);
            
          
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.banners.index')->withInput()->with('flash_error' , $error);
        }    
    }

    /**
     * @method abuse_providers_index()
     *
     * @uses used for get the list abused providers
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     * 
     * @param integer redeem_request_id 
     *
     * @param integer paid_amount
     * 
     * @return redirect to view page
     */

    public function abuse_providers_index(Request $request) {

        $abuse_providers = AbuseProvider::where('status' , 1)
                            ->orderBy('created_at' , 'desc')
                            ->paginate(10);

        return view('admin.abuse-providers.index')
                    ->with('page','requests')
                    ->with('sub_page','abuse-providers')
                    ->with('abuse_providers' , $abuse_providers);
    }

    /**
     * @method redeems_requests()
     *
     * @uses used redeem requests for the selected provider
     *
     * @created Vidhy R
     *
     * @updated Anjana
     * 
     * @param integer redeem_request_id 
     *
     * @param integer paid_amount
     * 
     * @return redirect view page
     */

    public function redeems_requests(Request $request) {

        try {
            
            $base_query = RedeemRequest::orderBy('status' , 'asc');

            $provider_details = [];

            if($request->provider_id) {

                $base_query = $base_query->where('provider_id' , $request->provider_id);

                $provider_details = Provider::find($request->provider_id);
            } 

            $redeems_requests = $base_query->paginate(10);

            return view('admin.revenues.redeems')
                    ->with('page','payments')
                    ->with('sub_page' , 'redeems')
                    ->with('redeems_requests' , $redeems_requests)
                    ->with('provider' , $provider_details);

        } catch( Exception $e) {
            
            $error = $e->getMessage();

            return redirect()->back()->with('flash_error',$error);
        }
    
    }

    /**
     * Function: providers_redeems_payout_invoice()
     * 
     * @uses to list providers redeems payout invoice
     *
     * @created vidhya R
     *
     * @updated Anjana
     *
     * @param - 
     *
     * @return redirect to view page with success/failure message
     */

    public function providers_redeems_payout_invoice(Request $request) {

        try {
        
            $validator = Validator::make($request->all() , [
                'redeem_request_id' => 'required|exists:redeem_requests,id',
                'paid_amount' => 'required', 
                'provider_id' => 'required'
                ]);

            if($validator->fails()) {

                $error = implode(',',$validator->messages()->all());

                throw new Exception($error, 101);                

            } 

            $redeem_request_details = RedeemRequest::find($request->redeem_request_id);

            if($redeem_request_details) {

                if($redeem_request_details->status == REDEEM_REQUEST_PAID ) {

                    throw new Exception( tr('redeem_request_status_mismatch'), 101);

                } else {

                    $invoice_data['provider_details'] = $provider_details = Provider::find($request->provider_id);

                    $invoice_data['redeem_request_id'] = $request->redeem_request_id;

                    $invoice_data['redeem_request_status'] = $redeem_request_details->status;

                    $invoice_data['provider_id'] = $request->provider_id;

                    $invoice_data['item_name'] = Setting::get('site_name')." - Checkout to"."$provider_details ? $provider_details->name : -";

                    $invoice_data['payout_amount'] = $request->paid_amount;

                    $invoice_details = json_decode(json_encode($invoice_data));

                    return view('admin.providers.payout')
                                ->with('page','providers')
                                ->with('sub_page' , 'providers')
                                ->with('invoice_details' , $invoice_details);
                }
            
            }

            throw new Exception(tr('redeem_not_found'), 101);

        } catch( Exception $e) {
            
            $error = $e->getMessage();

            return redirect()->route('admin.providers.redeems')->with('flash_error' , tr('something_error'));        
        }

    }

    /**
     * @method provider_redeems_payout_direct()
     *
     * @uses used for the payout provider
     *
     * @created Vidhy R
     *
     * @updated Anjana H
     * 
     * @param integer redeem_request_id 
     *
     * @param integer paid_amount
     * 
     */

    public function providers_redeems_payout_direct(Request $request) {

        try {

            DB::beginTransaction();

            $validator = Validator::make($request->all() , [
                'redeem_request_id' => 'required|exists:redeem_requests,id',
                'paid_amount' => 'required', 
                ]);

            if($validator->fails()) {

                $error = implode(',', $validator->messages()->all());

                throw new Exception($error, 101);

            } 

            $redeem_request_details = RedeemRequest::find($request->redeem_request_id);

            if($redeem_request_details) {

                if($redeem_request_details->status == REDEEM_REQUEST_PAID ) {

                    throw new Exception(tr('redeem_request_status_mismatch'), 101);
                }

                $message = tr('action_success');

                $is_add_bal_redeem_to_provider = NO;

                $redeem_amount = $request->paid_amount ? $request->paid_amount : 0;

                // Check the requested and admin paid amount is equal 

                if($request->paid_amount == $redeem_request_details->request_amount) {

                    $redeem_request_details->paid_amount = $redeem_request_details->paid_amount + $request->paid_amount;

                    $redeem_request_details->status = REDEEM_REQUEST_PAID;

                    $redeem_request_details->save();

                } else if($request->paid_amount > $redeem_request_details->request_amount) {

                    $redeem_request_details->paid_amount = $redeem_request_details->paid_amount + $redeem_request_details->request_amount;

                    $redeem_request_details->status = REDEEM_REQUEST_PAID;

                    $redeem_request_details->save();

                    $redeem_amount = $redeem_request_details->request_amount;

                } else {

                    $message = tr('redeems_request_admin_less_amount');

                    $redeem_amount = 0; // To restrict the redeeem paid amount update

                }

                $provider_payment_details = ProviderPayment::where('provider_id' , $redeem_request_details->provider_id)->first();

                if( count($provider_payment_details) > 0 ) {

                    $provider_payment_details->paid_amount = $provider_payment_details->paid_amount + $redeem_amount;

                    if ( $provider_payment_details->save() ) {

                        DB::commit();

                        return redirect()->route('admin.providers.redeems')->with('flash_success' , $message);

                    } else {

                        throw new Exception(tr('admin_not_error'), 101);

                    }
                }

            } else {

                return redirect()->route('admin.providers.redeems')->with('flash_error' , tr('something_error'));
            }

        } catch (Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return back()->with('flash_error',$error);
        }

    }

    /**
     * Function: providers_redeems_payout_response()
     * 
     * @uses used to get the response from paypal checkout
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param - 
     *
     * @return redirect to view page with success/failure message
     */

    public function providers_redeems_payout_response(Request $request) {

        try {

            DB::beginTransaction();

            $validator = Validator::make($request->all() , [
                'redeem_request_id' => 'required|exists:redeem_requests,id',
                ]);

            if($validator->fails()) {

                $error = implode(',',$validator->messages()->all());

                throw new Exception($error, 101);
            } 

            if($request->success == false) {
               
                throw new Exception(tr('redeem_paypal_cancelled'),101);
            }

            $redeem_request_details = RedeemRequest::find($request->redeem_request_id);

            if($redeem_request_details) {

                if($redeem_request_details->status == REDEEM_REQUEST_PAID ) {

                    throw new Exception( tr('redeem_request_status_mismatch'), 101 );

                } else {

                    $message = tr('action_success');

                    $redeem_amount = $request->paid_amount ? $request->paid_amount : 0;

                    // Check the requested and admin paid amount is equal 

                    if($request->paid_amount == $redeem_request_details->request_amount) {

                        $redeem_request_details->paid_amount = $redeem_request_details->paid_amount + $request->paid_amount;

                        $redeem_request_details->status = REDEEM_REQUEST_PAID;

                        if( $redeem_request_details->save()) {

                            DB::commit();

                        } else {

                            throw new Exception(tr('admin_redeem_request_save_error'), 101);
                        }

                    } else if($request->paid_amount > $redeem_request_details->request_amount) {

                        $redeem_request_details->paid_amount = $redeem_request_details->paid_amount + $redeem_request_details->request_amount;

                        $redeem_request_details->status = REDEEM_REQUEST_PAID;

                        if ( $redeem_request_details->save() ){

                            DB::commit();

                        } else {

                            throw new Exception(tr('admin_redeem_request_save_error'), 101);
                        }

                        $redeem_amount = $redeem_request_details->request_amount;

                    } else {

                        $message = tr('redeems_request_admin_less_amount');

                        $redeem_amount = 0; // To restrict the redeeem paid amount update

                    }

                    $provider_payment_details = ProviderPayment::where('provider_id' , $redeem_request_details->provider_id)->first();

                    if( count($provider_payment_details) > 0 ) {

                        $provider_payment_details->paid_amount = $provider_payment_details->paid_amount + $redeem_amount;

                        if ($provider_payment_details->save()){
                            
                            DB::commit();

                            return redirect()->route('admin.providers.redeems')->with('flash_success' , $message);

                        } else{

                            throw new Exception(tr('admin_provider_payment_save_error'), 101);
                        }
                    }
                }
            
            } 

            throw new Exception(tr('redeem_not_found'), 101);
            
        } catch (Exception $e) {

            DB::rollback();

            $error =  $e->getMessage();

            return redirect()->route('admin.providers.redeems')>withInput()->with('flash_error', $error );

        }

    }


    /**
     * @method revenues()
     *
     * @uses revenue dashboard for the website
     *
     * @created: vidhya R
     *
     * @updated: Vidhya R
     *
     * @param -
     *
     * @return html page
     */

    public function revenues(Request $request) {

        $provider_subscription_payments = ProviderSubscriptionPayment::where('status', PAID)->sum('paid_amount');

        $completed_requests = Requests::where('status' , REQUEST_COMPLETED)->with('payments')->get();

        $provider_amount = $completed_requests->sum('payments.provider_amount');

        $admin_amount = $completed_requests->sum('payments.admin_amount');

        $total = ( $completed_requests->sum('payments.total') +  $provider_subscription_payments ) ?: 0.0;
    
        $data = last_x_days_revenue(9);

        return view('admin.revenues.index')
                    ->with('page' , 'payments')
                    ->with('sub_page' , 'revenues')
                    ->with('total_revenue' , $total)
                    ->with('data' , $data)
                    ->with('provider_amount' , $provider_amount)
                    ->with('provider_subscription_payments' , $provider_subscription_payments)
                    ->with('admin_amount' , $admin_amount);
    }

    /**
     * @method controls_create()
     *
     * @uses get the admin control form
     *
     * @created: Maheswari
     *
     * @updated: Maheswari
     *
     * @param auth admin
     *
     * @return html page
     */
    public function controls_create() {

        return view('admin.settings.control')
                ->with('page', 'admin_control');
    }

    /**
     * @method controls_save()
     *
     * @uses Admin enable and disable the controls for edit,delete,add,submit option
     *
     * @created: Maheswari
     *
     * @updated: Maheswari
     *
     * @param Request the control values
     *
     * @return Success message
     */
    public function controls_save(Request $request){
      
        $model = Settings::get();

        foreach ($model as $key => $value) {
           
            $current_key = "";

            $current_key = $value->key;

            if($request->has($current_key)) {

                $value->value = $request->$current_key;

            }
            $value->save();

        }
        return back()->with('flash_success' , tr('admin_control_success'));
    }

    // Coupons

    /**
     * @method coupons_index()
     *
     * @uses list out coupon details
     *
     * @created Maheswari
     *
     * @updated Anjana H
     *
     * @param -
     *
     * @return Html table from coupon list page
     */
    public function coupons_index() {

        $coupons = Coupon::orderBy('updated_at','desc')->paginate(10);

        return view('admin.coupons.index')
                    ->with('page','coupons')
                    ->with('sub_page','coupons-view')
                    ->with('coupons',$coupons);    
    }

    /**
     * @method coupons_create()
     *
     * @uses To create a counpon details
     *
     * @created Maheswari
     *
     * @updated Anjana H
     *
     * @param -
     *
     * @return view page
     */
    public function coupons_create() {

        $coupon_details = new Coupon;

        $coupon_details->expiry_date = date('d-m-Y');

        return view('admin.coupons.create')
                ->with('page','coupons')
                ->with('sub_page','coupons-create')
                ->with('coupon_details', $coupon_details);
    }

    /**
     * @method coupons_edit() 
     *
     * @uses Edit the coupon details and get the coupon edit form for 
     *
     * @created Maheswari
     *
     * @updated Anjana
     *
     * @param Coupon id
     *
     * @return Get the html form
     */
    public function coupons_edit(Request $request){

        try {

            $coupon_details = Coupon::find($request->coupon_id);

            if(!$coupon_details) {

                throw new Exception(tr('coupon_not_found_error'), 101);
            } 

            return view('admin.coupons.edit')
                    ->with('page','coupons')
                    ->with('sub_page','coupons-view')
                    ->with('coupon_details',$coupon_details);

        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error',$error);
        }   
    }

    /**
     * @method coupons_save()
     *
     * @uses To save the coupon object details of new/existing based on details
     *
     * @created Maheswari
     *
     * @updated Anjana H
     *
     * @param request coupon_id, details
     *
     * @return success/error message
     */
    public function coupons_save(Request $request){

        try {

            $validator = Validator::make($request->all(),[
                'id'=>'exists:coupons,id',
                'title'=>'required',
                'coupon_code'=>$request->coupon_id ? 'required|max:10|min:1|unique:coupons,coupon_code,'.$request->coupon_id : 'required|unique:coupons,coupon_code|min:1|max:10',
                'amount'=>'required|numeric|min:1|max:5000',
                'amount_type'=>'required',
                'expiry_date'=>'required|date_format:d-m-Y|after:today',
                'no_of_users_limit'=>'required|numeric|min:1|max:1000',
                'per_users_limit'=>'required|numeric|min:1|max:100',
            ]);

            if($validator->fails()){

                $error = implode(',',$validator->messages()->all());

                throw new Exception( $error, 101);                
            }

            if($request->coupon_id !='') {
                        
                    $coupon_detail = Coupon::find($request->coupon_id); 

                    $message=tr('coupon_update_success');

            } else {

                $coupon_detail = new Coupon;

                $coupon_detail->status = COUPON_APPROVED;

                $message = tr('coupon_add_success');
            }

            // Check the condition amount type equal zero mean percentage
            
            if($request->amount_type == PERCENTAGE){

                // Amount type zero must should be amount less than or equal 100 only
                if($request->amount <= 100){

                    $coupon_detail->amount_type = $request->has('amount_type') ? $request->amount_type :DEFAULT_FALSE;
     
                    $coupon_detail->amount = $request->has('amount') ?  $request->amount : '';

                } else{

                    throw new Exception(tr('coupon_amount_lessthan_100'), 101);
                }

            } else {

                // This else condition is absoulte amount 

                // Amount type one must should be amount less than or equal 5000 only
                if($request->amount <= 5000){

                    $coupon_detail->amount_type=$request->has('amount_type') ? $request->amount_type : DEFAULT_TRUE;

                    $coupon_detail->amount=$request->has('amount') ?  $request->amount : '';

                } else {

                    throw new Exception(tr('coupon_amount_lessthan_5000'), 101);

                }
            }
            $coupon_detail->title=ucfirst($request->title);

            // Remove the string space and special characters
            $coupon_code_format  = preg_replace("/[^A-Za-z0-9\-]+/", "", $request->coupon_code);

            // Replace the string uppercase format
            $coupon_detail->coupon_code = strtoupper($coupon_code_format);

            // Convert date format year,month,date purpose of database storing
            $coupon_detail->expiry_date = date('Y-m-d',strtotime($request->expiry_date));
          
            $coupon_detail->description = $request->has('description')? $request->description : '' ;
             // Based no users limit need to apply coupons
            $coupon_detail->no_of_users_limit = $request->no_of_users_limit;

            $coupon_detail->per_users_limit = $request->per_users_limit;
            
            if($coupon_detail) {

                if( $coupon_detail->save() ) {

                    DB::commit();

                    return back()->with('flash_success',$message);

                }

                throw new Exception(tr('admin_coupon_save_error'), 101);
                
            }

            return redirect()->route('admin.coupons.view',['coupon_id' => $coupon_detail->id])->with('flash_success',$message);           
                       
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error', $error);
        }
        
    }


    /**
     * @method coupons_delete()
     *
     * @uses Delete the particular coupon detail
     *
     * @created Maheswari
     *
     * @updated Anjana H
     *
     * @param integer $id
     *
     * @return Deleted Success message
     */
    public function coupons_delete(Request $request) {

        try {
        
            $coupon_details = Coupon::find($request->coupon_id);

            if(!$coupon_details){

                throw new Exception(tr('coupon_not_found_error'), 101);
            } 

            if( $coupon_details->delete()) {

                DB::commit();

                return redirect()->route('admin.coupons.index')->with('flash_success',tr('coupon_delete_success'));

            } 

            throw new Exception(tr('admin_coupon_delete_error'), 101);
        
        } catch( Exception $e) {
            
            $error = $e->getMessage();

            return redirect()->route('admin.coupons.index')->with('flash_error',$error);
        }
        
    }

    /**
     * @method coupons_status_change()
     * 
     * @uses Coupon status for active and inactive update the status function
     *
     * @created Maheswari
     *
     * @updated Anjana H
     *
     * @param integer $coupon_id
     *
     * @return Success message for active/inactive
     */
    public function coupons_status_change(Request $request) {

        try {

            $coupon_details = Coupon::find($request->coupon_id);

            if(!$coupon_details){

                throw new Exception(tr('coupon_not_found_error'), 101); 
            } 

            $coupon_details->status = $coupon_details->status == COUPON_APPROVED ? COUPON_DECLINED : COUPON_APPROVED;

            if( $coupon_details->save()) { 

                DB::commit();

                $message = $coupon_details->status == COUPON_DECLINED ? tr('coupon_decline_success') : tr('coupon_approve_success');

                return back()->with('flash_success',$message);

            } 

            throw new Exception(tr('admin_coupon_status_error'), 101);

        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error',$error);
        } 
        
    }

    /**
     * @method coupons_view()
     *
     * @uses To display coupon details based on coupon_id
     *
     * @created Maheswari
     *
     * @updated Anjana H
     *
     * @param Integer (request() $coupon_id
     *
     * @return view page
     */
    public function coupons_view(Request $request) {

        try {

            $coupon_details = Coupon::find($request->coupon_id);

            if(!$coupon_details){

                throw new Exception(tr('coupon_not_found_error'), 101);
            }

            $user_coupon = "0";

            return view('admin.coupons.view')
                    ->with('page','coupons')
                    ->with('sub_page','coupons-view')
                    ->with('coupon_details',$coupon_details)
                    ->with('user_coupon', $user_coupon);
        
      
        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }  
    }

      /**
    * @method ios_control()
    *
    * @uses To update the ios payment subscription status
    *
    * @param settings key value
    *
    * @created Maheswari
    *
    * @updated Maheswari
    *
    * @return response of success / failure message.
    */
    public function ios_control(){

        if(Auth::guard('admin')->check()) {

            return view('admin.settings.ios-control')->with('page','ios-control');

        }

        return back();
        
    }

    /**
    * @method ios_control()
    *
    * @uses To update the ios settings value
    *
    * @param settings key value
    *
    * @created Maheswari
    *
    * @updated Maheswari
    *
    * @return response of success / failure message.
    */
    public function ios_control_save(Request $request){

        if(Auth::guard('admin')->check()) {

            $settings = Settings::get();

            foreach ($settings as $key => $setting_details) {

                # code...

                $current_key = "";

                $current_key = $setting_details->key;
                
                    if($request->has($current_key)) {

                        $setting_details->value = $request->$current_key;
                    }

                $setting_details->save();
            }

            return back()->with('flash_success',tr('admin_settings_success'));
        } 

        return back();
        
    }


    

    /**
     * @method provider_subscriptions_index()
     *
     * @uses To list out all the provider_subscriptions
     *
     * @created Anjana
     *
     * @updated Anjana
     *
     * @param -
     *
     * @return success/failure message
     */

    public function provider_subscriptions_index(Request $request) {
   
        $base_query = ProviderSubscription::orderBy('created_at', 'desc');
        
        $category_details = [];

        if($request->category_id) {

            $category_details = Category::find($request->category_id);

            if(!$category_details) {

                throw new Exception(tr('category_not_found'), 101);
            }

            $base_query = $base_query->where('category_id',$request->category_id);
        }

        $provider_subscriptions = $base_query->get();

        return view('admin.provider_subscriptions.index')
                ->with('page', 'provider_subscriptions')
                ->with('sub_page', 'provider_subscriptions-view')
                ->with('category_details', $category_details)
                ->with('provider_subscriptions', $provider_subscriptions);
    }

    /**
     * @method provider_subscriptions_create()
     *
     * @uses to get provider_subscriptions create form page 
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param -
     *
     * @return view page
     */
    public function provider_subscriptions_create() {

        $provider_subscription_details = new ProviderSubscription;
        
        $provider_subscription_details->expiry_date = date('Y-m-d');

        $categories = getCategories();

        return view('admin.provider_subscriptions.create')
                ->with('page', 'provider_subscriptions')
                ->with('sub_page', 'provider_subscriptions-create')
                ->with('categories', $categories)
                ->with('provider_subscription_details', $provider_subscription_details);
    
    }

    /**
     * @method provider_subscriptions_edit
     *
     * @uses To edit the provider_subscriptions based on id
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param request provider_subscription_id
     *
     * @return success/failure message
     */
    public function provider_subscriptions_edit(Request $request) {

        try {

            $provider_subscription_details = ProviderSubscription::where('id', $request->provider_subscription_id)->first();
            
            if(!$provider_subscription_details) {

                throw new Exception(tr('admin_provider_subscription_not_found'), 101);
            }
            
            $categories = getCategories();

            return view('admin.provider_subscriptions.edit')
                ->with('page', 'provider_subscriptions')
                ->with('sub_page', 'provider_subscriptions-view')
                ->with('categories', $categories)
                ->with('provider_subscription_details', $provider_subscription_details);
       
        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->route('admin.provider_subscriptions.index')->with('flash_error', $error);

        } 
    
    }

    /**
     * @method provider_subscriptions_save
     *
     * @uses To save the subscription based on id
     *
     * @created Anjana H
     *
     * @updated Anjana
     *
     * @param object $request - provider_subscription Id
     *
     * @return response of success/failure response
     */
    public function provider_subscriptions_save(Request $request) {

        try {

            DB::begintransaction();

            $validator = Validator::make($request->all(), [
                'title'  => 'required|max:255',
                'amount' => 'required|numeric|min:0.1|max:10000000',
                'description' => 'max:255',
                'plan' => 'required|numeric|min:1',
                'image'  => 'mimes:jpeg,png,jpg',
                'category_id' => $request->category_id ? 'required|exists:categories,id' : '',
            ]);

            if($validator->fails()){

                $error = implode(',', $validator->messages()->all());

                throw new Exception($error, 101);
            }

            $provider_subscription_details = $request->provider_subscription_id ? ProviderSubscription::find($request->provider_subscription_id) : new ProviderSubscription;

            if(!$provider_subscription_details) {

                throw new Exception(tr('admin_provider_subscription_not_found'), 101);
            }

            if($request->provider_subscription_id != '') {
               
                if($request->hasFile('image')) {

                    Helper::delete_file('/uploads/provider_subscriptions/', $provider_subscription_details->image);
                }

            } else {
                
                $provider_subscription_details->status = APPROVED;

                $provider_subscription_details->picture = asset('placeholder.png');

            }

            if($request->hasFile('image')) {

                $provider_subscription_details->picture = Helper::upload_avatar('uploads/provider_subscriptions', $request->file('image'));
            }

            $provider_subscription_details->title = $request->title;

            $provider_subscription_details->description = $request->description;

            $provider_subscription_details->amount = $request->amount;

            $provider_subscription_details->plan = $request->plan;

            $provider_subscription_details->category_id = $request->category_id;

            $provider_subscription_details->expiry_date = date('Y-m-d', strtotime($request->expiry_date));

            if($provider_subscription_details->save()) {

                DB::commit();

                $message = $request->provider_subscription_id ? tr('admin_provider_subscription_update_success') : tr('admin_provider_subscription_create_success');

                return redirect()->route('admin.provider_subscriptions.view', ['provider_subscription_id' => $provider_subscription_details->id])->with('flash_success', $message);
            } 

            throw new Exception(tr('admin_provider_subscription_saved_error') , 101);

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error', $error);
        } 
    
    }

    /**
     * @method provider_subscriptions_view
     *
     * @uses To display subscription based on provider_subscription_id
     *
     * @created Anjana H
     *
     * @param request provider_subscription_id
     *
     * @updated Anjana H
     *
     * @param request provider_subscription_id
     *
     * @return success/failure message
     */
    public function provider_subscriptions_view(Request $request) {

        try {

           $provider_subscription_details = ProviderSubscription::where('id', $request->provider_subscription_id)->first();

            if(!$provider_subscription_details) {

                throw new Exception(tr('admin_provider_subscription_not_found'), 101);
            }

            $provider_subscription_details->total_subscriptions = ProviderSubscriptionPayment::where('provider_subscription_id',  $request->provider_subscription_id)->where('status', PAID)->count();

            $total_revenue = ProviderSubscriptionPayment::where('provider_subscription_id', $request->credit_package_id)->sum('amount');

            $category_details = Category::find($provider_subscription_details->category_id);

            $provider_subscription_details->category_name = 'category not avail';

            if($category_details) {
                $provider_subscription_details->category_name = $category_details->name;
            }

            $provider_subscription_details->total_revenue = $total_revenue;

            return view('admin.provider_subscriptions.view')
                    ->with('page', 'provider_subscriptions')
                    ->with('sub_page', 'provider_subscriptions-view')
                    ->with('provider_subscription_details', $provider_subscription_details);
        
        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);

        }     
    
    }

    /**
     * @method provider_subscriptions_delete()
     *
     * @uses To delete the particular provider_subscription detail
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param provider_subscription id
     *
     * @return success/error message
     */
    public function provider_subscriptions_delete(Request $request) {

        try {

            DB::beginTransaction();

            $provider_subscription_details = ProviderSubscription::find($request->provider_subscription_id);

            if (!$provider_subscription_details) {

                throw new Exception(tr('admin_provider_subscription_not_found'), 101);
            } 

            if( $provider_subscription_details->delete()) {

                DB::commit();

                return redirect()->route('admin.provider_subscriptions.index')->with('flash_success', tr('admin_provider_subscription_delete_success'));
            }

            throw new Exception(tr('admin_provider_subscription_delete_error'), 101);
                
        } catch (Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.provider_subscriptions.index')->with('flash_error', $error);

        }

    }

    /**
     * @method provider_subscriptions_status_change()
     *
     * @uses to change status approve/decline update process
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param request sub category id
     *
     * @return success message
     */
    public function provider_subscriptions_status_change(Request $request) {

        try {

            DB::begintransaction();

            $provider_subscription_details = ProviderSubscription::find($request->provider_subscription_id);

            if(!$provider_subscription_details) {

                throw new Exception(tr('admin_provider_subscription_not_found'), 101);
            } 

            $provider_subscription_details->status = $provider_subscription_details->status == APPROVED ? DECLINED : APPROVED;

            if( $provider_subscription_details->save() ) {

                DB::commit();

                $message = $provider_subscription_details->status == APPROVED ? tr('admin_provider_subscription_approved_success') : tr('admin_provider_subscription_declined_success');

                return back()->with('flash_success', $message);
            }

            throw new Exception(tr('admin_provider_subscription_status_error'), 101);

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.provider_subscriptions.index')->with('flash_error', $error);
        }  
        
    }

    /**
     * @method provider_subscriptions_is_popular()
     *
     * @uses Make the provider_subscriptions as popular
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param request sub provider_subscription id
     *
     * @return success/error message
     */

    public function provider_subscriptions_is_popular(Request $request) {

        try {

            DB::begintransaction();

            $provider_subscription_details = ProviderSubscription::find($request->provider_subscription_id
                );

            if (!$provider_subscription_details) {

                throw new Exception(tr('admin_provider_subscription_not_found'), 101);
            }

            $provider_subscription_details->is_popular = $provider_subscription_details->is_popular == APPROVED ? DECLINED : APPROVED ;

            if( $provider_subscription_details->save() ) {

                DB::commit();

                $message = $provider_subscription_details->is_popular == YES ? tr('admin_provider_subscription_add_popular_success') : tr('admin_provider_subscription_remove_popular_success');
                
                return back()->with('flash_success',$message );
            } 

            throw new Exception(tr('admin_provider_subscription_is_popular_error'), 101);  

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.provider_subscriptions.index')->with('flash_error', $error);
        }

    }
    
    /**
     * @method provider_subscriptions_plans()
     *
     * @uses To display subscriptions based on provider_id
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id
     *
     * @return success/error message
     */
    public function provider_subscriptions_plans(Request $request) {

        try {

            $provider_subscriptions = ProviderSubscription::orderBy('created_at','desc')->where('status', APPROVED)->get();

            $payments = ProviderSubscriptionPayment::where('provider_id' , $request->provider_id)->orderBy('provider_subscription_payments.created_at' , 'desc')->get();
          
           return view('admin.provider_subscriptions.provider_plans')
                        ->withPage('providers')   
                        ->with('sub_page','providers-view')
                        ->with('provider_subscriptions' , $provider_subscriptions)
                        ->with('provider_id', $request->provider_id)
                        ->with('payments', $payments); 
        
        } catch (Exception $e) {
            
            $error = $e->getMessage();

            return redirect()->back()->with('flash_error',$error);
        }            
    }
    
    /**
     * @method provider_subscriptions_plans_save()
     *
     * @uses To save provider subscription based on subscription and provider_id
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_subscription_id, $provider_id
     *
     * @return success/error message
     */
    public function provider_subscriptions_plans_save(Request $request) {
          
        try {

            // validation

            DB::beginTransaction();

            // record check
            $provider_details = Provider::find($request->provider_id);

            if(!$provider_details ) {

                throw new Exception( tr('provider_not_found'), 101);
            } 
 
            $provider_subscription_details = ProviderSubscription::where('id', $request->provider_subscription_id)->first();
            
            if(!$provider_subscription_details) {

                throw new Exception(tr('admin_provider_subscription_not_found'), 101);
            } 

            $provider_subscription_payment = new ProviderSubscriptionPayment();
           
            $provider_subscription_payment->provider_id = $provider_details->id;

            $provider_subscription_payment->provider_subscription_id = $provider_subscription_details->id;

            $provider_subscription_payment->subscription_amount = $provider_subscription_payment->amount = $provider_subscription_details->amount;
             
            $provider_subscription_payment->payment_id = ($provider_subscription_payment->amount > 0) ? uniqid(str_replace(' ', '-', 'PAY')) : 'Free Plan'; 
            
            /** category subscriptions **/
            $category_subscriptions = ProviderSubscription::where('category_id',$provider_subscription_details->category_id)->pluck('id');

            /** get category based previous provider subscription payment **/
            $previous_payment_base_query = ProviderSubscriptionPayment::where('provider_id' , $provider_details->id)->where('status', PAID)->where('is_current_subscription', YES);            

            if($category_subscriptions) {
                
                $previous_payment_base_query = $previous_payment_base_query->whereIn('provider_subscription_id',$category_subscriptions);
            }
            
            $previous_subscription_payment_details = $previous_payment_base_query->first();

            if ($previous_subscription_payment_details) {
            
                $expiry_date = strtotime($previous_subscription_payment_details->expiry_date);
               
                $plan = $provider_subscription_details->plan;
                
                $provider_subscription_payment->expiry_date = $expiry_date >= strtotime(date('Y-m-d H:i:s')) ? date('Y-m-d H:i:s', strtotime("+{$plan} months", $expiry_date)) :  date('Y-m-d H:i:s',strtotime("+{$plan} months"));

                $previous_subscription_payment_details->is_current_subscription = NO;

            } else {

                $provider_subscription_payment->expiry_date = date('Y-m-d H:i:s',strtotime("+{$provider_subscription_details->plan} months"));
            }

            $provider_subscription_payment->is_current_subscription = YES;
          
            $provider_subscription_payment->status = DEFAULT_TRUE;
            
            $provider_subscription_payment->subscribed_by = ADMIN;

            if( $provider_subscription_payment->save())  {
                
                if( $previous_subscription_payment_details) { 
                    
                    $previous_subscription_payment_details->save();
                }
                
                $provider_details->provider_type = PAID_PROVIDER;

                $provider_details->save();

                DB::commit();

                return back()->with('flash_success', tr('admin_provider_subscription_applied_success'));
            } 
            
            throw new Exception(tr('admin_user_subascription_save_error'), 101);
                        
        } catch (Exception $e) {
            
            DB::rollback();
            
            $error = $e->getMessage();

            return back()->with('flash_error',$error);
        }

    }


    /**
     * @todo v4.0 
     *
     * @method provider_auto_subscription_disable()
     *
     * @uses To prevent automatic subscriptioon of user,user has option to cancel subscription
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param $request - User details & payment details
     *
     * @return success/failure message
     */
    public function provider_auto_subscription_disable(Request $request) {

       try {

            $provider_payment = ProviderSubscriptionPayment::where('provider_id', $request->provider_id)->where('status', PAID_STATUS)->orderBy('created_at', 'desc')->first();

            if(!$provider_payment) {

                throw new Exception(tr('admin_provider_payment_details_not_found'), 101);
            } 

            $provider_payment->is_cancelled = AUTORENEWAL_CANCELLED;

            $provider_payment->cancelled_reason = $request->cancel_reason;

            if ($provider_payment->save()) {
               
                DB::commit();

                return back()->with('flash_success', tr('admin_cancel_subscription_success'));            
            }

            throw new Exception(tr('admin_provider_auto_subscription_disable_error'), 101);                
          
        } catch (Exception $e) {
                  
            $error = $e->getMessage();

            return back()->with('flash_error',$error);
        }      

    }

    /**
     * @todo v4.0 
     *
     * @method provider_auto_subscription_enable()
     *
     * @uses To prevent automatic subscription, provider has option to cancel provider subscriptions
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param (request) - provider details & payment details
     *
     * @return success/failure message
     */
    public function provider_auto_subscription_enable(Request $request) {
        
        try {

            $provider_payment = ProviderSubscriptionPayment::where('provider_id', $request->provider_id)->where('status', PAID_STATUS)->orderBy('created_at', 'desc')
                ->where('is_cancelled', AUTORENEWAL_CANCELLED)
                ->first();

            if(!$provider_payment){

                throw new Exception(tr('admin_provider_payment_details_not_found'), 101);
            }  

            $provider_payment->is_cancelled = AUTORENEWAL_ENABLED;

            $provider_payment->save();

            return back()->with('flash_success', tr('admin_autorenewal_enable_success'));
        
        } catch (Exception $e) {
            
            $error = $e->getMessage();

            return back()->with('flash_error',$error);
        }     

    }  

    /**
     * @method provider_subscriptions_payments()
     *
     * @uses To list out all the subscription payments
     *
     * @created Shobana
     *
     * @updated vidhya
     *
     * @param object $request - As of now no attribute
     *
     * @return response of html page with payment details
     */

    public function provider_subscriptions_payments(Request $request) {
        
        $base_query = ProviderSubscriptionPayment::orderBy('updated_at', 'DESC');

        if($request->provider_subscription_id) {

            $base_query = $base_query->where('provider_subscription_id',$request->provider_subscription_id);
        }

        $provider_subscription_payments = $base_query->get();
    
        return view('admin.providers.subscription_payments')
                ->with('page', 'payments')
                ->with('sub_page', 'provider_subscription-payments')
                ->with('provider_subscription_payments', $provider_subscription_payments);
    }
    
    /**
     *
     * @method requests_cancel()
     *
     * @uses To get cancel reason for request
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param $request - request details 
     *
     * @return success/failure message
     */
    public function requests_cancel(Request $request) {
        
        try {

            $request_details = Requests::find($request->request_id);
           
            $allowed_cancel_status = [PROVIDER_NONE,PROVIDER_ASSIGNED,PROVIDER_ACCEPTED,PROVIDER_STARTED];

            // Check whether request cancelled previously

            if($request_details->status != REQUEST_CANCELLED) {

                if(in_array($request_details->provider_status, $allowed_cancel_status)) {

                    // Update status of the request to cancellation

                    $request_details->status = REQUEST_CANCELLED;

                    $request_details->cancelled_date = date('Y-m-d H:i:s');

                    $request_details->cancelled_by = ADMIN;

                    $request_details->cancelled_reason = $request->cancel_reason;
                    
                    $request_details->save();

                      // If request has confirmed provider then release him to available status

                    if($request_details->provider_id != false) {

                        $provider = Provider::find($request_details->provider_id);

                        if($request_details->job_type == REQUEST_NOW) {

                            $provider->is_available = PROVIDER_AVAILABLE;
                            
                            $provider->waiting_to_respond = WAITING_TO_RESPOND_NORMAL;
                        }

                        $provider->save();

                    }                        

                    return back()->with('flash_success', tr('request_cancel_success'));
                } 

                throw new Exception(tr('request_cancel_status_mismatch'), 101);
            }

            throw new Exception(tr('request_cancel_already'), 101);           

        } catch (Exception $e) {
                  
            $error = $e->getMessage();

            return back()->with('flash_error',$error);
        }      

    }


    /**
     * @method sub_admins_index()
     *
     * @uses To list out subadmins (only admin can access this option)
     * 
     * @created Anjana H
     *
     * @updated Anjana H  
     *
     * @param object $request
     *
     * @return view page
     */
    public function sub_admins_index() {

        $sub_admins = Admin::orderBy('created_at', 'desc')->where('role', SUBADMIN)->get();

        return view('admin.sub_admins.index')
                ->with('page', 'sub-admins')
                ->with('sub_page', 'sub-admins-view')
                ->with('sub_admins', $sub_admins);        
    }

    /**
     * @method sub_admins_create()
     *
     * To create a sub admin only admin can access this option
     * 
     * @created Anjana H
     *
     * @updated Anjana H  
     *
     * @param object $request - -
     *
     * @return response of html page with details
     */
    public function sub_admins_create() {

        $sub_admin_details = new Admin();

        return view('admin.sub_admins.create')
                ->with('page', 'sub-admins')
                ->with('sub_page', 'sub-admins-create')
                ->with('sub_admin_details', $sub_admin_details);
    }

    /**
     * @method sub_admins_edit()
     *
     * @uses To edit a sub admin based on subadmin id only  admin can access this option
     * 
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param object $request - sub Admin Id
     *
     * @return response of html page with details
     */
    public function sub_admins_edit(Request $request) {

       try {
          
            $sub_admin_details = Admin::find($request->sub_admin_id);

            if(!$sub_admin_details) {

                throw new Exception( tr('admin_sub_admin_not_found'), 101);
            }

            return view('admin.sub_admins.edit')
                        ->with('page', 'sub-admins')
                        ->with('sub_page', 'sub-admins-view')
                        ->with('sub_admin_details', $sub_admin_details);

        } catch( Exception $e) {
            
            $error = $e->getMessage();

            return redirect()->route('admin.sub_admins.index')->with('flash_error',$error);
        }
    }

    /**
     * @method sub_admins_view()
     *
     * @uses To view a sub admin based on sub admin id only admin can access this option
     * 
     * @created Anjana H
     *
     * @updated Anjana H  
     *
     * @param object $request - Sub Admin Id
     *
     * @return response of html page with details
     */
    public function sub_admins_view(Request $request) {

        try {
          
            $sub_admin_details = Admin::find($request->sub_admin_id);

            if(!$sub_admin_details) {

                throw new Exception( tr('admin_sub_admin_not_found'), 101);
            } 

            return view('admin.sub_admins.view')
                    ->with('page', 'sub-admins')
                    ->with('sub_page', 'sub-admins-view')
                    ->with('sub_admin_details', $sub_admin_details);
        
           } catch( Exception $e) {
            
            $error = $e->getMessage();

            return redirect()->route('admin.sub_admins.index')->with('flash_error',$error);
        }
    }

    /**
     * @method sub_admins_delete()
     *
     * @uses To delete a sub admin based on sub admin id. only admin can access this option
     * 
     * @created Anjana H
     *
     * @updated Anjana H  
     *
     * @param object $request - Sub Admin Id
     *
     * @return response of html page with details
     */
    public function sub_admins_delete(Request $request) {

        try {

            DB::beginTransaction();
            
            $sub_admin_details = Admin::where('id' , $request->sub_admin_id)->first();

            if(!$sub_admin_details) {  

                throw new Exception(tr('admin_sub_admin_not_found'), 101);
            }
            
            if( $sub_admin_details->delete() ) {

                DB::commit();

                return redirect()->route('admin.sub_admins.index')->with('flash_success',tr('admin_sub_admin_delete_success'));
            }

            throw new Exception(tr('admin_sub_admin_delete_error'), 101);
            
        } catch (Exception $e) {
            
            DB::rollback();

            $error = $e->getMessage();

            return back()->with('flash_error',$error);
        }
    }

    /**
     * @method sub_admins_save()
     *
     * @uses To save the sub admin details
     * 
     * @created Anjana H
     *
     * @updated Anjana H  
     *
     * @param object $request - Sub Admin Id
     *
     * @return response of html page with details
     */
    public function sub_admins_save(Request $request) {

        try {
            
            DB::beginTransaction();

            $validator = Validator::make( $request->all(),array(
                    'name' => 'required|max:100',
                    'email' => $request->sub_admin_id ? 'email|max:255|unique:admins,email,'.$request->sub_admin_id : 'required|email|max:255|unique:admins,email,NULL',
                    'mobile' => 'digits_between:4,16',
                    'address' => 'max:300',
                    'sub_admin_id' => 'exists:admins,id',
                    'picture' => 'mimes:jpeg,jpg,png',
                    'description'=>'required|max:255',
                    'password' => $request->sub_admin_id ? '' : 'required|min:6|confirmed',
                )
            );
            
            if($validator->fails()) {

                $error = implode(',', $validator->messages()->all());

                throw new Exception($error, 101);
            } 

            $sub_admin_details = $request->sub_admin_id ? Admin::find($request->sub_admin_id) : new Admin;

            if (!$sub_admin_details) {

                throw new Exception(tr('sub_admin_not_found'), 101);
            }

            $sub_admin_details->name = $request->name ?: $sub_admin_details->name;

            $sub_admin_details->email = $request->email ? $request->email : $sub_admin_details->email;

            $sub_admin_details->mobile = $request->has('mobile') ? $request->mobile : $sub_admin_details->mobile;

            $sub_admin_details->description = $request->description ? $request->description : '';

            $sub_admin_details->role = SUBADMIN;
                
            if (!$sub_admin_details->id) {

                $new_password = $request->password;
                
                $sub_admin_details->password = \Hash::make($new_password);

                $sub_admin_details->picture = asset('placeholder.png');

            }

            if($request->hasFile('picture')) {

                if($request->sub_admin_id) {

                    Helper::delete_picture($sub_admin_details->picture, "/uploads/sub_admins/");
                }

                $sub_admin_details->picture = Helper::upload_avatar('uploads/admins' ,$request->file('picture'));
            }

            $sub_admin_details->timezone = $request->timezone;

            $sub_admin_details->token = Helper::generate_token();

            $sub_admin_details->token_expiry = Helper::generate_token_expiry();

            $sub_admin_details->status = DEFAULT_TRUE;

            if($sub_admin_details->save()) {

                DB::commit();

                $message = $request->sub_admin_id ? tr('admin_sub_admin_update_success') : tr('admin_sub_admin_create_success');
                
                return redirect()->route('admin.sub_admins.view', ['sub_admin_id' =>$sub_admin_details->id ])->with('flash_success', $message);
            } 

            throw new Exception(tr('admin_sub_admin_save_error'), 101);
           
        } catch (Exception $e) {
            
            DB::rollback();
            
            $error = $e->getMessage();

            return back()->withInput()->with('flash_error',$error);
        }
    
    }

    /**
     * @method sub_admins_status()
     *
     * @uses To change the status of the sub admin, based on sub admin id. only admin can access this option
     * 
     * @created Anjana H
     *
     * @updated Anjana H  
     *
     * @param object $request - SubAdmin Id
     *
     * @return response of html page with details
     */
    public function sub_admins_status(Request $request) {

        try {

            DB::beginTransaction();
       
            $sub_admin_details = Admin::find($request->sub_admin_id);

            if(!$sub_admin_details) {
                
                throw new Exception(tr('admin_sub_admin_not_found'), 101);
            } 
            
            $sub_admin_details->status = $sub_admin_details->status == APPROVED ? DECLINED : APPROVED;

            $message = $sub_admin_details->status == APPROVED ? tr('admin_sub_admin_approve_success') : tr('admin_sub_admin_decline_success');

            if( $sub_admin_details->save() ) {

                DB::commit();

                return back()->with('flash_success', $message);
            } 

            throw new Exception(tr('admin_sub_admin_status_error'), 101);
            
        } catch( Exception $e) {

            DB::rollback();
            
            $error = $e->getMessage();

            return redirect()->route('admin.sub_admins.index')->with('flash_error',$error);
        }
    }


    /**
     * @method providers_dashboard()
     *
     * @uses to show revenue analitucs of providers
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param Integer (request) $provider_id  
     *
     * @return view page
     */
    public function providers_dashboard(Request $request) {

        try {
            
            $provider_details = [];

            if($request->provider_id) {
               
                $provider_details = Provider::find($request->provider_id);

                if(!$provider_details) {

                    throw new Exception(tr('provider_not_found'), 101);
                }
            }

            /** provider requests count **/
            $provider_requests_count = Requests::count();

            /** provider requests ongoing count **/
            $provider_requests_ongoing = Requests::where('status',REQUEST_ONGOING);

            /** provider requests upcoming count **/
            $provider_requests_upcoming = Requests::where('job_type' , REQUEST_LATER)->whereIn('requests.status' , [REQUEST_ONGOING])->whereIn('provider_status' , [PROVIDER_ACCEPTED])->leftJoin('requests_meta','requests_meta.request_id', '=','requests.id');
            
            /** provider requests completed count **/
            $provider_requests_completed = Requests::whereIn('status' , [REQUEST_RATING , REQUEST_COMPLETED]);

            /** total_cancelled_jobs **/
            $provider_total_cancelled_jobs = Requests::whereIn('status' , [REQUEST_CANCELLED , REQUEST_TIME_EXCEED_CANCELLED]);

            /** provider total_ernings **/
            $total_ernings = RequestPayment::where('is_paid' , YES);

            if($request->provider_id) {

                $provider_requests_count = Requests::where('provider_id',$request->provider_id)->count();
          
                $provider_requests_ongoing = $provider_requests_ongoing->where('provider_id',$request->provider_id);
            
                $provider_requests_upcoming = $provider_requests_upcoming->where('requests_meta.provider_id',$request->provider_id);
           
                $provider_requests_completed = $provider_requests_completed->where('provider_id',$request->provider_id);
            
                $total_ernings = $total_ernings->where('provider_id',$request->provider_id);

                $provider_total_cancelled_jobs = $provider_total_cancelled_jobs->where('provider_id' , $request->id);
          
            }

            $data['provider_requests_count'] = $provider_requests_count;

            $data['provider_requests_ongoing'] = $provider_requests_ongoing->count();

            $data['provider_requests_upcoming'] = $provider_requests_upcoming->count();

            $data['provider_requests_completed'] = $provider_requests_completed->count();

            $data['total_ernings'] = $total_ernings->sum('total') ?: 0; 

            $data['provider_total_cancelled_jobs'] = $provider_total_cancelled_jobs->count();
                       
            $data['provider_amount'] = $total_ernings->sum('request_payments.provider_amount') ?: 0;

            $data['admin_amount'] = $total_ernings->sum('request_payments.admin_amount') ?: 0;
            
            return view('admin.providers.dashboard')
                        ->with('page' , 'providers')
                        ->with('sub_page' , 'providers-dashboard')
                        ->with('provider_details' , $provider_details)
                        ->with('data' , $data);

        } catch(Exception $e){

            $error = $e->getMessage();

            return redirect()->route('admin.providers.index')->with('flash_error', $error);
        }    
    }

}





















