<?php

namespace App\Http\Controllers\Auth;

use App\User;
use Validator;
use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\ThrottlesLogins;
use Illuminate\Foundation\Auth\AuthenticatesAndRegistersUsers;
use Illuminate\Http\Request;

use App\Repositories\UserRepository as UserRepo;

use Illuminate\Support\Facades\Auth;

use App\Helpers\Helper;

use Setting;

class AuthController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Registration & Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users, as well as the
    | authentication of existing users. By default, this controller uses
    | a simple trait to add these behaviors. Why don't you explore it?
    |
    */

    use AuthenticatesAndRegistersUsers, ThrottlesLogins;

    /**
     * Where to redirect users after login / registration.
     *
     * @var string
     */

    protected $redirectTo = '/home';

    protected $redirectAfterLogout = '/login';


    /**
     * The Login form view that should be used.
     *
     * @var string
     */

    protected $loginView = 'user.auth.login';

    /**
     * The Register form view that should be used.
     *
     * @var string
     */

    protected $registerView = 'user.auth.register';

    protected $form_request;


    /**
     * Create a new authentication controller instance.
     *
     * @return void
     */
    public function __construct(Request $request)
    {        
        $this->form_request = $request;
        $this->middleware($this->guestMiddleware(), ['except' => 'logout']);

    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */        
    protected function validator(array $data) {

        return Validator::make($data, [
            'name' => 'required|max:255',
            'email' => 'required|email|max:255|unique:users',
            'password' => 'required|min:6|confirmed',
            'picture' => 'mimes:jpeg,jpg,bmp,png',
            'referral' => 'exists:user_referrers,referral_code,status,'.DEFAULT_TRUE

        ]);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return User
     */
    protected function create(array $data) {

        $response = UserRepo::store($this->form_request);

        // We need use common function, So only based on the response 

        if($response)

            $User = User::find($response['user_id']);
            
        // \Session::flash('flash_success', tr('register_success'));
        
        return $User;
    }

   /**
     * Log the user out of the application.
     *
     * @return \Illuminate\Http\Response
     */
    public function logout()
    {
        Auth::logout();
        
        \Session::flash('flash_success',tr('logout_success'));

        // flash()->info('Bye', 'You have been successfully logged out!');
        return redirect(property_exists($this, 'redirectAfterLogout') ? $this->redirectAfterLogout : '/');
    }


    public function register(Request $request)
    {
        $validator = $this->validator($request->all());
        if ($validator->fails()) {
            $this->throwValidationException(
              $request, $validator
            );
        }

        $user = $this->create($request->all());
        // @todo Wallet modeule to be done hence hidden referral -->
        // if($request->referral) {

        //     UserRepo::referral_register($request->referral, $user);
        // }


        if(Setting::get('email_verify_control') == EMAIL_VERIFICATION_ON) {

            \Session::flash('flash_error', tr('email_verify_alert'));

            return redirect(route('user.login.form'))->with('flash_error', tr('email_verify_alert'));

        } else {

            return redirect(route('user.login.form'))->with('flash_success', tr('register_success'));
        }
    }


    /**
     * Function called after user logs in
     *
     *
     *
     * @return \Illuminate\Http\RedirectResponse
     *
     */
    
    protected function authenticated(Request $request, User $user) {

        if(\Auth::check()) {
            
            if($user = User::find(\Auth::user()->id)) {

                $user->login_by = 'manual';

                $user->timezone = $this->form_request->timezone ? $this->form_request->timezone : $user->timezone;

                $user->token = Helper::generate_token();

                $user->token_expiry = Helper::generate_token_expiry();

                $user->save();
                
            }   
        }
        
        \Session::flash('flash_success', tr('login_success'));
        
        return redirect()->intended($this->redirectPath());
    }
}
