<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\ThrottlesLogins;
use Illuminate\Foundation\Auth\AuthenticatesAndRegistersUsers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

use App\Repositories\ProviderRepository as ProviderRepo;

use App\Admin;
use App\Provider;
use App\ProviderService;
use App\Helpers\Helper;

use Setting;

use Validator;

class ProviderAuthController extends Controller {
    /*
    |--------------------------------------------------------------------------
    | Registration & Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users, as well as the
    | authentication of existing users. By default, this controller uses
    | a simple trait to add these behaviors. Why don't you explore it?
    |
    */

    use AuthenticatesAndRegistersUsers, ThrottlesLogins;

    /**
     * Where to redirect users after login / registration / logout.
     *
     * @var string
     */
    protected $redirectTo = 'provider';

    protected $redirectAfterLogout = '/provider/login';

    /**
     * The guard to be used for validation.
     *
     * @var string
     */

    protected $guard = 'provider';

    /**
     * The Login form view that should be used.
     *
     * @var string
     */

    protected $loginView = 'provider.auth.login';

    /**
     * The Register form view that should be used.
     *
     * @var string
     */

    protected $registerView = 'provider.auth.register';

    protected $form_request;

    /**
     * Create a new authentication controller instance.
     *
     * @return void
     */
    public function __construct(Request $request) {

        $this->form_request = $request;

        $this->middleware($this->guestMiddleware(), ['except' => 'logout']);
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {        
        return Validator::make($data, [
            'name' => 'required|max:255',
            'email' => 'required|email|max:255|unique:providers',
            'password' => 'required|min:6|confirmed',
            'picture' => 'mimes:jpeg,jpg,bmp,png',
        ]);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return User
     */
    protected function create(array $data) {

         $response = ProviderRepo::store($this->form_request);

        // We need use common function, So only based on the response 

        if($response)

            $Provider = Provider::find($response['provider_id']);

        return $Provider;

    }

   /**
     * Log the user out of the application.
     *
     * @return \Illuminate\Http\Response
     */
    public function logout()
    {
        Auth::guard('provider')->logout();
        
        \Session::flash('flash_success',tr('logout_success'));

        return redirect(property_exists($this, 'redirectAfterLogout') ? $this->redirectAfterLogout : '/');
    }

    public function register(Request $request)
    {
        $validator = $this->validator($request->all());
        if ($validator->fails()) {
            $this->throwValidationException(
              $request, $validator
            );
        }

        $user = $this->create($request->all());


        if(Setting::get('email_verify_control') == EMAIL_VERIFICATION_ON) {

            \Session::flash('flash_error', tr('email_verify_alert'));

            return redirect(route('provider.login.form'))->with('flash_error', tr('email_verify_alert'));

        } else {

            return redirect(route('provider.login.form'))->with('flash_success', tr('register_success'));
        }
    }

    /**
     * Function called after user logs in
     * @return \Illuminate\Http\RedirectResponse
     */
    protected function authenticated() {

       // \Session::flash('flash_success', tr('login_success'));

        if(\Auth::guard('provider')->check()) {

            if($provider = Provider::find(\Auth::guard('provider')->user()->id)) {

                if(Setting::get('email_verify_control') == EMAIL_VERIFICATION_ON) {

                    if(!$provider->is_verified) {

                        \Auth::guard('provider')->logout();

                        // Check the verification code expiry

                        Helper::check_email_verification("" , $provider, $error, PROVIDER);

                        return redirect(route('provider.login.form'))->with('flash_error', tr('email_verify_alert'));

                    }

                }
               
                $provider->login_by = 'manual';
                
                $provider->timezone = $this->form_request->timezone ? $this->form_request->timezone : $user->timezone;

                $provider->token = Helper::generate_token();
                
                $provider->token_expiry = Helper::generate_token_expiry();

                $provider->save();

                if($provider->timezone) {
                    // date_default_timezone_set($provider->timezone);
                }
            }   
        }
        return redirect(route('provider.dashboard'))->with('flash_success', tr('register_success'));
    }
}
