<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Http\Requests;

use App\Repositories\AdminRepository as AdminRepo;

use App\Helpers\Helper;

use App\Language;

use App\Settings;

use DB;

use Exception;

use Setting;

class LanguageController extends Controller
{
    
    /**
     * @method languages_index()
     *
     * @uses To list out languages details 
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param -
     *
     * @return view page
     */
    public function languages_index() {

        // Load Lanugages

        $languages = Language::paginate(10);

        return view('admin.languages.index')
                    ->with('page','languages')
                    ->with('sub_page', 'languages-view')
                    ->with('languages', $languages);

    }
   
    /**
     * @method languages_create()
     *
     * @uses To create a language object details
     *
     * @created Anjana H 
     *
     * @updated Anjana H
     *
     * @param 
     *
     * @return View page
     */
    public function languages_create(Request $request) {

        $language_details = new Language;

        return view('admin.languages.create')
                ->withPage('languages')
                ->with('sub_page','create_language')
                ->with('language_details', $language_details);
    }

    /**
     * @method languages_edit()
     *
     * @uses language edit form page
     *
     * @created vidhya R
     *
     * @updated Anjana H
     *
     * @param language id
     *
     * @return view page
     */
    public function languages_edit(Request $request) {

        try {
            
            $language_details = Language::where('id', $request->language_id)->first();

            if(!$language_details){

                throw new Exception( tr('admin_language_not_found'), 101);
            } 

            return view('admin.languages.edit')
                ->with('page','languages')
                ->with('sub_page', 'languages-view')
                ->with('language_details', $language_details);

        } catch (Exception $e) {
            
            $error = $e->getMessage();

            return redirect()->back()->with('flash_error',$error);

        }    
    }
    
    /**
     * @method languages_save()
     *
     * @uses To save the language object details of new/existing based on details
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param (request) language details
     *
     * @return success/error message
     */
    public function languages_save(Request $request) {
  
        try {
            
            $language_details = AdminRepo::languages_save($request);
            
            if ($language_details['success'] == true) {
                
                return redirect(route('admin.languages.index'))->with('flash_success', $language_details['message']);            
            } 
                
            throw new Exception($language_details['error'], $language_details['code']);  

        } catch( Exception $e) {
            
            $error = $e->getMessage();

            return redirect()->back()->withInput()->with('flash_error',$error);
        }    
    }

    /**
     * @method languages_delete()
     * 
     * @uses To delete the languages object based on language id
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param 
     *
     * @return success/failure message
     */
    public function languages_delete(Request $request) {

        try {
            
            DB::beginTransaction();

            // Check the language is declined

            $language_details = Language::where('id', $request->language_id)->first();

            if(!$language_details) {

                throw new Exception(tr('admin_language_not_found'), 101);
            }

            if($language_details->status == DECLINED) {

                return back()->with('flash_error', tr('admin_language_declined'));
            }

            $check_languages = Language::where('id', '!=', $request->language_id)->count();

            if($language_details->folder_name == "en" || $check_languages == 0 || $language_details->folder_name == Setting::get('default_lang'))
            {

                return back()->with('flash_error', tr('admin_language_delete_restrict'));
            }
            
            $folder_name = $language_details->folder_name;

            if ($language_details->delete()) {

                $setting_details = Settings::where('key', 'default_lang')->first();

                if($setting_details) {

                    $setting_details->value = 'en';

                    if($setting_details->save()) {
                                               
                        DB::commit();

                        Helper::delete_language_files($folder_name, DEFAULT_TRUE);
                        
                        return back()->with('flash_success', tr('admin_language_delete_success'));  
                    } 
                        
                    throw new Exception(tr('admin_language_delete_error'), 101);
                }  

                throw new Exception(tr('admin_default_language_not_found'), 101);
            
            }
            
            throw new Exception(tr('admin_language_delete_error'), 101);
                   
        } catch (Exception $e) {
            
            DB::rollback();

            $error = $e->getMessage();

            return redirect()->route('admin.languages.index')->with('flash_error',$error);
        }

    }

    /**
     * @method languages_status_change()
     *
     * @uses To update languages status to approve/decline based on language id
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param Integer (request) $language_id
     *
     * @return success/error message
     */
    public function languages_status_change(Request $request) {

        try {

            DB::beginTransaction();
       
            $language_details = Language::find($request->language_id);

            if(!$language_details) {
                
                throw new Exception(tr('admin_language_not_found'), 101);
            }

            $language_details->status = $language_details->status == APPROVED ? DECLINED : APPROVED ;
                        
            $message = $language_details->status == APPROVED ? tr('admin_language_activate_success') : tr('admin_language_deactivate_success');

            if( $language_details->save() ) {

                DB::commit();

                return back()->with('flash_success',$message);
            } 

            throw new Exception(tr('admin_language_status_error'), 101);
           
        } catch( Exception $e) {

            DB::rollback();
            
            $error = $e->getMessage();

            return redirect()->route('admin.languages.index')->with('flash_error',$error);
        }
    }


    public function languages_download(Request $request) {

        try {

            $folder_name = $request->folder_name ?: 'en';;

            $file_name = $request->file_name;

            //PDF file is stored under project/public/download/info.pdf
            $file_path = base_path(). "/resources/lang/".$folder_name.'/'.$file_name.'.php';

            $headers = array(
                      'Content-Type: application/x-php',
                    );

            return response()->download($file_path, $file_name.'.php', $headers);

        } catch(Exception $e) {

            $error = $e->getMessage();

            return redirect()->back()->with('flash_error', $error);
        }

    }
    
    public function set_default_language($name) {
        // Load Setting Table
        $model = Settings::where('key','default_lang')->first();

        if ($model) {

            $model->value = $name;

            $model->save();

            if($model) {
                $fp = fopen(base_path() .'/config/new_config.php' , 'w');
                fwrite($fp, "<?php return array( 'locale' => '".$name."', 'fallback_locale' => '".$name."');?>");
                fclose($fp);

                \Log::info("Key : ".config('app.locale'));

                return back()->with('flash_success' , tr('set_default_language_success'))->with('flash_language', true);
            }
        }  

        return back()->with('flash_error',tr('admin_not_error'));
    }
}
