<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Helpers\Helper;

use App\Repositories\CommonRepository as CommonRepo;

use App\Repositories\ProviderRepository as ProviderRepo;

use App\Repositories\RequestRepository as RequestRepo;

use App\Jobs\sendPushNotification;

use App\User;

use App\Provider;

use App\ProviderService;

use App\Requests;

use App\RequestsMeta;

use App\NotificationTemplate;

use App\RequestPayment;

use App\UserRating;

use App\ProviderRating;

use App\Category;

use App\SubCategory;

use App\SmsTemplate;

use App\ChatMessage;

use App\Redeem;

use Setting;

use Validator;

use Log;

use Hash;

use DB;

use Exception;

use App\EmailTemplate;

use App\UserNotification;

use App\ProviderNotification;

use App\RedeemRequest;

use App\ProviderPayment;

use App\ProviderGallery;

use App\ProviderSubscription;

use App\ProviderSubscriptionPayment;

use App\ProviderCard;

class ProviderApiController extends Controller {

    public $loginProvider, $skip, $take, $timezone;

    public function __construct(Request $request) {

        $this->middleware('ProviderApiVal' , array('except' => ['register' , 'login' , 'forgot_password']));

        $this->skip = $request->skip ?: 0;

        $this->take = $request->take ?: (Setting::get('admin_take_count', 6));

        if($request->id){

            $this->loginProvider = Provider::ProviderResponse($request->id)->first();

            $this->timezone = $this->loginProvider->timezone ?? "America/New_York";
        }

    }

    /** 
     * @method register()
     *
     * @uses to save the provider details
     *
     * @created Vithya R
     *
     * @edited Vithya R 
     *
     * @param Form data
     *
     * @return JSON Response 
     */

    public function register(Request $request) {

        \Log::info("Provider Register Start");

        $operation = true;

        // Check the device type , device token and login by

        if(!CommonRepo::basic_validation($request->all(), $errors)) {

            \Log::info("Provider basic_validation Failed");

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            \Log::info("Provider basic_validation SUCCESS");

            $login_by = $request->login_by;

            $allowedSocialLogin = ['facebook','google'];

            if(in_array($login_by,$allowedSocialLogin)) {

                \Log::info("Provider social_validation Check");

                // Check the social unique id and form fields

                if(!CommonRepo::social_validation($request->all(), $errors)) {

                    \Log::info("Provider social_validation FAILED");

                    $operation = false;
                    $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];
                }

            } else {

                \Log::info("Provider MAIL VALIDATION Check");

                $check_mail_action = true;

                if($request->has('email')) {

                    $check_email_exist = Provider::where('email' , $request->email)->withTrashed()->count();

                    if($check_email_exist) {

                        \Log::info("Provider MAIL Already Exists");

                        $operation = $check_mail_action = false;
                        $response_array = ['success' => false , 'error' => Helper::error_message(102) , 'error_code' => 102];
                    } 
                }

                // Validate manual registration fields

                if($check_mail_action) {
                    
                    \Log::info("Provider MAIL VALIDATION Check");

                    if(!CommonRepo::manual_validation($request->all(), $errors)) {

                        \Log::info("Provider MAIL VALIDATION Failed");

                        $operation = false;
                        $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];
                    } 
                }
            
            }

            if($operation) {

                Log::info("Provider Register".print_r($request->all() , true));

                $check_provider = Provider::where('email' , $request->email)->first();

                if($check_provider) {

                    \Log::info("Update Provider");

                    $request->request->add(['is_social_login' => 1]);

                    $provider_details = ProviderRepo::update($request , $check_provider->id);

                } else {
                    
                    \Log::info("Create Provider");

                    $provider_details = ProviderRepo::store($request);                    
                }

                $provider_details = Helper::null_safe($provider_details);

                if($provider_details['is_approved'] == PROVIDER_DECLINED) {
                    
                    $response = array('success' => false , 'error' => Helper::error_message(502) , 'error_code' => 502);
                    return response()->json($response, 200);
                }

                if($provider_details['is_verified'] == PROVIDER_EMAIL_NOT_VERIFIED && !in_array($login_by,$allowedSocialLogin)) {

                    $response_array = array('success' => false , 'error' => Helper::error_message(503) , 'error_code' => 503);

                } else {
                    
                    $provider_details['is_appstore_upload'] = Setting::get('is_appstore_upload') ?: 0;
                    
                    $response_array = ['success' => true , 'data' => $provider_details , 'message' => Helper::success_message(204) , 'code' => 204];
                }

                \Log::info('Registration completed');

            }

        }

        \Log::info("Provider Register End");

        return response()->json($response_array, 200);
    
    }

    /**
     * @method register_agree()
     * 
     * @uses Provider Registeration Flow - Agree the terms and conditions
     *
     * @created Vithya R
     *
     * @edited Vithya R
     *
     * @param Request provider id.
     * 
     * @return JSON Response
     */

    public function register_agree(Request $request) {

        Log::info('Register Agree Started');

        if($request->agree) {

            $provider = Provider::find($request->id);

            $provider->step = PROVIDER_STEP_AGREE;
            $provider->save();

            $response_array = ['success' => true , 'message' => Helper::success_message(223) , 'code' => 223];

        } else {

            Log::info('Select agree Button');

            $response_array = ['success' => false , 'error' => Helper::error_message(159) , 'error_code' => 159];
        }
        return response()->json($response_array, 200);

    }

    /**
     * @method login()
     *
     * @uses check manual login and allow the provider to access
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param string email
     *
     * @param string password
     * 
     * @return JSON Response
     */

    public function login(Request $request) {
    
        try {
               
            // Check the device type , device token and login by

            if(!CommonRepo::basic_validation($request->all(), $errors)) {

                throw new Exception($errors , 101);
            } 

            if(!CommonRepo::login_validation($request->all(), $errors , 'providers')) {

                throw new Exception($errors, 101);
            } 

            $response_array = ProviderRepo::login($request);
        
            return response()->json($response_array, 200);

        } catch (Exception $e) {
            
            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success' => false, 'error' => $error_message , 'error_code' => $error_code];

            return response()->json($response_array);
        }
    }

    /**
     * @method forgot_password()
     *
     * @uses based on the entered email address, validate and send email
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param string email
     *
     * @return JSON Response
    */
    public function forgot_password(Request $request) {

        if(!CommonRepo::email_validation($request->all(), $errors , 'providers')) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $response_array = ProviderRepo::forgot_password($request);
        }

        return response()->json($response_array, 200);
    
    }

    /**
     * @method change_password()
     *
     * @uses based on the entered valid password, validate and change password
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param password
     *
     * @return JSON Response
    */

    public function change_password(Request $request) {

        if(!CommonRepo::change_password_validation($request->all(), $errors)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $provider = Provider::find($request->id);

            if(\Hash::check($request->old_password,$provider->password)) {

                $provider->password = \Hash::make($request->password);
                $provider->save();

                $response_array = ['success' => true , 'message' => Helper::success_message(217) , 'code' => 217];
            } else {
                $response_array = ['success' => false , 'error' => Helper::error_message(107) , 'error_code' => 107];
            }
        }

        return response()->json($response_array,200);
    
    }

    /**
     * @method profile()
     *
     * @uses to display provider profile
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param Provider id
     *
     * @return JSON Response
    */

    public function profile(Request $request) {
        
        $provider = Provider::providerResponse($request->id)->find($request->id);

        if($provider) {
            $provider = Helper::null_safe($provider->toArray());
        }

        $response_array = ['success' => true , 'data' => $provider];

        return response()->json($response_array, 200);
    
    }

    /**
     * @method update_profile()
     *
     * @uses to update provider details
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param Request Object 
     * 
     * @return Upadte provider details with an array
     */

    public function update_profile(Request $request) {

        $validator = Validator::make(
            $request->all(),
            array(
                'name' => 'max:255',
                'email' => 'email|unique:providers,email,'.$request->id.'|max:255',
                'picture' => 'mimes:jpeg,bmp,png',
                'cover' => 'mimes:jpeg,bmp,png',
                'gender' => 'in:male,female,others',
                'device_token' => '',
                'promo_video' => 'url'
            ));

        if ($validator->fails()) {

            // Error messages added in response for debugging
            $errors = implode(',',$validator->messages()->all());

            $response_array = ['success' => false,'error' => $errors,'error_code' => 101];

        } else {
            
            $provider = ProviderRepo::update($request, $request->id);
            
            if($provider) {
                $provider = Helper::null_safe($provider);
            }

            $response_array = ['success' => true , 'message' => Helper::success_message(218) ,'data' => $provider , 'code' => 218];
        }

        return response()->json($response_array, 200);
    
    }

    /**
     * @method galleries_index()
     *
     * @uses to update the gallery image 
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param Request Object 
     * 
     * @return Upadte provider details with an array
     */

    public function galleries_index(Request $request) {

        try {

            $provider_galleries = ProviderGallery::where('provider_id' , $request->id)->select('id as provider_gallery_id' ,'provider_id' ,'picture' , 'about' , 'created_at')->get();

            $response_array = ['success' => true , 'data' => $provider_galleries];

            return response()->json($response_array, 200);

        } catch(Exception $e) {

            $error = $e->getMessage();

            $code = $e->getCode();

            $response_array = ['success'=>false, 'error'=> $error, 'error_code'=>$code];

            return response()->json($response_array);
        }

    }

    /**
     * @method galleries_save()
     *
     * @uses to update the gallery image 
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param Request Object 
     * 
     * @return Upadte provider details with an array
     */

    public function galleries_save(Request $request) {

        try {

            DB::beginTransaction();

            $validator = Validator::make(
                $request->all(),
                array(
                    'about' => 'max:255',
                    'picture' => 'required|mimes:jpeg,bmp,png',
                    'provider_gallery_id' => 'exists:provider_galleries,id'
                    
                ));

            if ($validator->fails()) {

                // Error messages added in response for debugging
                $errors = implode(',',$validator->messages()->all());

                throw new Exception($errors, 101);
                
            } else {
            
                if($request->provider_gallery_id) {

                    $check_provider_gallery = $provider_gallery = ProviderGallery::find($request->provider_gallery_id);

                    if(count($check_provider_gallery) == 0) {

                        throw new Exception(Helper::error_message(174), 101);
                        
                    }

                    if($request->hasFile('picture')) {

                        Helper::delete_avatar('uploads/provider-galleries',$provider_gallery->picture); 
                    }

                } else {

                    $provider_gallery = new ProviderGallery;

                    $provider_gallery->provider_id = $request->id;

                }

                if($request->hasFile('picture')) {

                    $provider_gallery->picture = Helper::upload_avatar('uploads/provider-galleries',$request->file('picture'));

                }

                $provider_gallery->about = $request->about ? $request->about : "";

                $provider_gallery->status = 1;

                $provider_gallery->save();

                DB::commit();

                $response_array = ['success' => true , 'message' => Helper::success_message(236), 'code' => 236];

                return response()->json($response_array, 200);

            }
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            $code = $e->getCode();

            $response_array = ['success'=>false, 'error'=> $error, 'error_code'=>$code];

            return response()->json($response_array);
        }

    }

    /**
     * @method galleries_delete()
     *
     * @uses to delete the gallery image 
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param Request Object 
     * 
     * @return JSON Response
     */

    public function galleries_delete(Request $request) {

        try {

            DB::beginTransaction();

            $validator = Validator::make(
                $request->all(),
                array(
                    'provider_gallery_id' => 'required|exists:provider_galleries,id'
                    
                ));

            if ($validator->fails()) {

                // Error messages added in response for debugging
                $errors = implode(',',$validator->messages()->all());

                throw new Exception($errors, 101);
                
            } else {
            

                $check_provider_gallery = $provider_gallery = ProviderGallery::find($request->provider_gallery_id);

                if(count($check_provider_gallery) == 0) {

                    throw new Exception(Helper::error_message(174), 174);
                    
                }

                if($request->hasFile('picture')) {

                    Helper::delete_avatar('uploads/provider-galleries',$provider_gallery->picture); 
                }

                $check_provider_gallery->delete();
                

                DB::commit();

                $response_array = ['success' => true , 'message' => Helper::success_message(236), 'code' => 236];

                return response()->json($response_array, 200);

            }
        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            $code = $e->getCode();

            $response_array = ['success'=>false, 'error'=> $error, 'error_code'=>$code];

            return response()->json($response_array);
        }

    }

    /**
     * @method delete_account()
     *
     * @uses to delete provider details If the provider from social login , no need password
     * 
     * @created Vidhya R
     * 
     * @edited Vidhya R
     *
     * @param Request Object 
     * 
     * @return Upadte provider details with an array
     */

    public function delete_account(Request $request) {

        $validator = Validator::make($request->all(),
            array(
                'password' => '',
            ));

        if ($validator->fails()) {

            $errors = implode(',',$validator->messages()->all());

            $response_array = ['success' => false, 'error' => $errors,'error_code' => 101];

        } else {

            $provider_details = Provider::find($request->id);

            $allow = 0;

            if($provider_details->login_by != 'manual') {
                
                $allow = 1;
                
            } else {

                if(Hash::check($request->password, $provider_details->password)) {

                    $allow = 1;

                } else {

                    $allow = 0 ;

                    $response_array = array('success' => false , 'error' => Helper::error_message(107) ,'error_code' => 107);

                }

            }

            if($allow) {

                $provider_details = Provider::where('id',$request->id)->first();

                if($provider_details) {

                    // Delete the become user details as well

                    $user_details = $provider_details->user_id ? User::where('id',$provider_details->user_id)->first() : "";

                    if($user_details) {
                        $user_details->forceDelete();
                    }

                    $provider_details->forceDelete();

                    $response_array = ['success' => true , 'message' => Helper::success_message(219) , 'code' => 219];
                } else {
                    $response_array = ['success' =>false , 'error' => Helper::error_message(F6), 'error_code' => 146];
                }

            } else {
                $response_array = array('success' => false , 'error' => Helper::error_message(107) ,'error_code' => 107);
            }

        }

        return response()->json($response_array,200);

    }
 
    /**
     * @method categories()
     *
     * @uses to get categoru details
     * 
     * @created Anjana H 
     *
     * @updated Anjana H
     *
     * @param object $request
     *
     * @return Json response with provider details
     */
    public function categories(Request $request) {
        
        $categories = Category::with('subCategory')->where('status' , APPROVED)->get();

        $data = [];

        if($categories) {

            foreach ($categories as $key => $category) {

                $sub_categories = $category->subCategory;

                if(count($sub_categories) > 0) {

                    $service_data = new \stdClass;

                    $service_data->category_id = $category->id;
                    
                    $service_data->category_name = $category->name;
                    
                    $service_data->category_description = $category->description;
                    
                    $service_data->category_picture = $category->picture;

                    /** check category has paid subscriptions **/

                    $service_data->is_subscription_based_category = is_category_subscription($category->id) ? YES : NO;
                    
                    /** Provider total services count **/

                    $service_data->total_services_count = ProviderService::where('provider_id' , $request->id)
                                        ->where('category_id', $category->id)
                                        ->where('status', APPROVED)
                                        ->count();

                    $service_data->is_provider_paid_for_subscription = NO;

                    // if category has paid subscriptions check for provider subscribed for it
                    if($service_data->is_subscription_based_category == YES) {

                        $service_data->is_provider_paid_for_subscription = check_provider_subscribed_category($request->id, $category->id) ? YES : NO;
                   
                    }

                    // Is provider updated the services or not
                    $service_data->is_service_updated = $service_data->total_services_count > 0 ? YES : NO;

                    array_push($data, $service_data);
                
                }
            
            }
        }

        $data = Helper::null_safe($data);

        $response_array = ['success' => true , 'data' => $data];

        return response()->json($response_array , 200);
    }
 
    /**
     * @method sub_categories()
     *
     * @uses to get categoru details
     * 
     * @created Anjana H 
     *
     * @updated Anjana H
     *
     * @param object $request
     *
     * @return Json response with provider details
     */
    public function sub_categories(Request $request) {
       
        try {
            
            $category_details = Category::where('status', APPROVED)->where('categories.id', $request->category_id)->first();

            $data = [];
            
            if(!$category_details) {

                throw new Exception(Helper::error_message(177), 177);                
            }

            if($category_details) {

                $data = new \stdClass;

                $data->category_id = $category_details->id;
                
                $data->category_name = $category_details->name;
                
                $data->category_description = $category_details->description;
                
                $data->category_picture = $category_details->picture;

                /** check category has paid subscriptions **/

                $data->is_subscription_based_category = is_category_subscription($category_details->id) ? YES : NO;
                
                /** Provider total services count **/

                $data->total_services_count = ProviderService::where('provider_id' , $request->id)
                                    ->where('category_id', $category_details->id)
                                    ->where('status', APPROVED)
                                    ->count();

                $data->is_provider_paid_for_subscription = NO;

                // if category has paid subscriptions check for provider subscribed for it
                if($data->is_subscription_based_category == YES) {

                    $data->is_provider_paid_for_subscription = check_provider_subscribed_category($request->id, $category_details->id) ? YES : NO;
               
                }

                // Is provider updated the services or not
                $data->is_service_updated = $data->total_services_count > 0 ? YES : NO; 

                
                $sub_categories_data = [];
              
                $sub_categories = SubCategory::where('category_id', $request->category_id)->where('status', APPROVED)->get();

                foreach($sub_categories as $sc => $sub_category) {
          
                    $provider_service = ProviderService::where('provider_id' , $request->id)->where('sub_category_id', $sub_category->id)->first();

                    $sub_category_data['provider_service_id'] = "";
                  
                    $sub_category_data['status'] = 0; $sub_category_data['price_per_hour'] = 0.00;

                    if($provider_service) {

                        $sub_category_data['provider_service_id'] = $provider_service->id;
                       
                        $sub_category_data['status'] = $provider_service->status;

                        $sub_category_data['price_per_hour'] = $provider_service->price_per_hour;
                    }

                    $sub_category_data['sub_category_id'] = $sub_category->id;
                   
                    $sub_category_data['sub_category_name'] = $sub_category->name;
                   
                    $sub_category_data['sub_category_description'] = $sub_category->description;
                   
                    $sub_category_data['sub_category_picture'] = $sub_category->picture;

                    array_push($sub_categories_data, $sub_category_data);
                
                }

                Helper::null_safe($data);

                $data->sub_categories = $sub_categories_data;

                $response_array = ['success' => true, 'data' => $data];

                return response()->json($response_array , 200);
            }
            
            throw new Exception("Error Processing Request", 1);

        } catch (Exception $e) {
            
            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success' => false, 'error' => $error_message , 'error_code' => $error_code];

            return response()->json($response_array);
        }
    }

    /**
     * @method services()
     *
     * @uses to get all services when selected by the provider
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     * 
     * @param Id And token
     *
     * @return json response of the services
     */
    public function services(Request $request) {

        $categories = Category::with('subCategory')->where('status' , 1)->get();

        $data = [];

        if($categories) {

            foreach ($categories as $key => $category) {

                $sub_categories = $category->subCategory;

                if(count($sub_categories) > 0) {

                    $category_selection_status = 0;

                    $service_data['category_id'] = $category->id;
                    
                    $service_data['category_name'] = $category->name;
                    
                    $service_data['category_description'] = $category->description;
                    
                    $service_data['category_picture'] = $category->picture;

                    /** check category has paid subscriptions **/
                    $service_data['is_paid_category'] = is_category_subscription($category->id) ? YES : NO;
                    
                    /** Provider total services count **/
                    $service_data['total_services_count'] = ProviderService::where('provider_id' , $request->id)
                        ->where('category_id', $category->id)
                        ->where('status',APPROVED)->count();

                    /** if category has paid subscriptions check for provider subscription for it **/
                    if($service_data['is_paid_category']) {

                        $service_data['is_subscribed_category'] = check_provider_subscribed_category($request->id, $category->id) ? YES : NO;
                   
                    } else { /** on category has no paid subscriptions check for provider subscription for it **/
                        
                        $service_data['is_subscribed_category'] = $service_data['total_services_count'] > 0 ? YES : NO;                      
                    }                 

                    $sub = [];

                    foreach($sub_categories as $sc => $sub_category) {

                        if($sub_category->status) {

                            $provider_service = ProviderService::where('provider_id' , $request->id)->where('sub_category_id', $sub_category->id)->first();

                            $sub_data['provider_service_id'] = "";
                          
                            $sub_data['status'] = 0;
                          
                            $sub_data['price_per_hour'] = 0.00;

                            if($provider_service) {

                                $sub_data['provider_service_id'] = $provider_service->id;
                               
                                $sub_data['status'] = $provider_service->status;

                                $sub_data['price_per_hour'] = $provider_service->price_per_hour;

                                if(!$category_selection_status)

                                    $category_selection_status = $provider_service->status ? 1 : 0;
                            }

                            $sub_data['sub_category_id'] = $sub_category->id;
                            $sub_data['sub_category_name'] = $sub_category->name;
                            $sub_data['sub_category_description'] = $sub_category->description;
                            $sub_data['sub_category_picture'] = $sub_category->picture;

                            array_push($sub, $sub_data);

                        }

                    }

                    $service_data['selection_status'] = $category_selection_status;

                    $service_data['sub_category'] = $sub;

                    array_push($data, $service_data);
                
                }

            }
        }

        $data = Helper::null_safe($data);

        $response_array = ['success' => true , 'data' => $data];

        return response()->json($response_array , 200);
        
    }


    /**
     * @method services_save()
     *
     * @uses to Save the provider selected service 
     *
     * @created vidhya R
     *
     * @updated vidhya R
     * 
     * @param id, token, sub_category_id and price_per_hour
     *
     * @return json response of the services
     */

    public function services_save(Request $request) {

        Log::info("services_save".print_r($request->all() ,true));

        // Get all the sub categories

        $sub_categories = SubCategory::where('status' , 1)->get();

        if($sub_categories) {

            foreach ($sub_categories as $key => $sub_category) {

                $sub_category_id = $sub_category->id;

                // check the whether provider selected the service or not

                if($request->$sub_category_id) {

                    // check the selected service record already available

                    $check_sub_category = $provider_service = ProviderService::where('provider_id' , $request->id)->where('sub_category_id' , $sub_category_id)->first();

                    if(!$check_sub_category) {
                        
                        $provider_service = new ProviderService;
                        $provider_service->provider_id = $request->id;
                        $provider_service->category_id = $sub_category->category->id;
                        $provider_service->sub_category_id = $sub_category->id;
                    }

                    $provider_service->status = 1;
                    $provider_service->price_per_hour = $request->$sub_category_id;
                    $provider_service->save();
                }
            
            }
        
        }

        $categories = Category::with('subCategory')->where('status' , 1)->get();

        $data = [];

        if($categories) {

            foreach ($categories as $key => $category) {

                $sub_categories = $category->subCategory;

                $service_data['category_id'] = $category->id;
                $service_data['category_name'] = $category->name;
                $service_data['category_description'] = $category->description;
                $service_data['category_picture'] = $category->picture;

                $sub = [];

                foreach($sub_categories as $sc => $sub_category) {

                    if($sub_category->status) {

                        $provider_service = ProviderService::where('provider_id' , $request->id)->where('sub_category_id', $sub_category->id)->first();

                        $sub_data['provider_service_id'] = 0; $sub_data['price_per_hour'] = 0; $sub_data['status'] = 0.00;

                        if($provider_service) {
                            $sub_data['provider_service_id'] = $provider_service->id;
                            $sub_data['status'] = $provider_service->status;
                            $sub_data['price_per_hour'] = $provider_service->price_per_hour;
                        }

                        $sub_data['sub_category_id'] = $sub_category->id;
                        $sub_data['sub_category_name'] = $sub_category->name;
                        $sub_data['sub_category_description'] = $sub_category->description;
                        $sub_data['sub_category_picture'] = $sub_category->picture;

                        array_push($sub, $sub_data);
                        
                    }

                }

                $service_data['sub_category'] = $sub;

                array_push($data, $service_data);

            }
        
        }

        $data = Helper::null_safe($data);

        $response_array = ['success' => true , 'message' => Helper::success_message(220) ,'data' => $data , 'code' => 220];

        return response()->json($response_array , 200);

    }

    /**
     * @method services_remove()
     *
     * @uses to remove the provider selected service 
     *
     * @created vidhya R
     *
     * @updated Anjana H
     * 
     * @param id, token, sub_category_id
     *
     * @return json response of the services
     */

    public function services_remove(Request $request) {

        Log::info("services_remove".print_r($request->all() ,true));

        try {

            DB::beginTransaction();

            // Validate the common and basic fields

            $validator = Validator::make($request->all(),
                [
                    'sub_category_id' => 'required|exists:sub_categories,id',
                ]);

            if($validator->fails()) {

                $error = implode(',', $validator->messages()->all());

                throw new Exception($error , 101);

            } else {

                $check_provider_service = ProviderService::where('provider_id' , $request->id)->where('sub_category_id' , $request->sub_category_id)->first();

                if($check_provider_service) {

                    if($check_provider_service->status == OFF) {

                        throw new Exception(Helper::error_message(167) , 167);
                    }

                    $check_provider_service->status = OFF;

                    $check_provider_service->save();

                    DB::commit();

                    $response_array = ['success' => true, 'message' => Helper::success_message(235) , 'code' => 235];

                    return response()->json($response_array , 200);


                } else {

                    throw new Exception(Helper::error_message(166), 166);
                    
                }

            }

        } catch( Exception $e) {

            DB::rollback();

            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

            return response()->json($response_array);
        }

    }

    /**
     * @method location_update()
     *
     * @uses to update the provider location
     * 
     * @created Vidhya R
     *
     * @edited Vidhya R
     *  
     * @param id, token , latitude and longitude
     * 
     * @return json 
     */

    public function location_update(Request $request) {

        $validator = Validator::make(
                $request->all(),
                array(
                    'latitude' => 'required',
                    'longitude' => 'required'
                ));
            
        if ($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false, 'error' => $errors,'error_code' => 101];

        } else {

            $provider = Provider::find($request->id);
            $provider->address = $request->address;
            $provider->latitude = $request->latitude;
            $provider->longitude = $request->longitude;
            $provider->save();

            $response_array = Helper::null_safe(['success' => true,'provider_id' => $provider->id,'latitude' => $provider->latitude,'longitude' => $provider->longitude , 'message' => Helper::success_message(223) , 'code' => 223]);
        }
        
        return response()->json($response_array, 200);
    
    }

    /**
     * @method check_available()
     *
     * @uses to check the provider availability
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     * 
     * @param id, token
     *
     * @return json 
     */

    public function check_available(Request $request) {

        $active = 0;

        $provider = Provider::find($request->id);

        if($provider)
            $active = $provider->is_available;

        $response_array = Helper::null_safe(['success' => true,'provider_id' => $request->id,'active' => $active
        ]);

        return response()->json($response_array, 200);
    }

    /**
     * @method available_update()
     *
     * @uses to update the provider availability
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param id, token and status
     *
     * @return json 
     */

    public function available_update(Request $request) {

        $validator = Validator::make(
                $request->all(),
                array(
                    'status' => 'required|in:'.DEFAULT_TRUE.','.DEFAULT_FALSE,
                ));
            
        if ($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false, 'error' => $errors ,'error_code' => 101];

        } else {

            $provider = Provider::find($request->id);

            $provider->is_available = $request->status;
            $provider->save();

            $response_array = Helper::null_safe(['success' => true,'provider_id' => $provider->id,'active' => $provider->is_available , 'message' => Helper::success_message(221) , 'code' => 221]);
        }

        return response()->json($response_array, 200);
    }

    /**
     * @method documents()
     *
     * @uses  added the provider document values
     *
     * @created Vidhaya 
     *
     * @edited Vidhya R
     *
     * @param id
     *
     * @return JSON Response
    */

    public function documents(Request $request) {

        $documentss = Document::all();
        
        $documents = array();
        
        foreach($documentss as $doc) {

            $pro_doc = DB::table('provider_documents')
                      ->where('provider_documents.provider_id','=',$request->id)
                      ->where('provider_documents.document_id','=',$doc->id)->first();
            $document['id'] = $doc->id;
            $document['name'] = $doc->name;
            
            if($pro_doc){
                $document['document_url'] = $pro_doc->document_url;
            } else {
                $document['document_url'] = "";
            }
            
            array_push($documents,$document);
        }

        $provider = Provider::find($request->id);

        $response_array = array('success' => true, 'documents' => $documents);
        $response_array = Helper::null_safe($response_array);

        $response = response()->json($response_array, 200);
        return $response;
    
    }

    /**
     * @method upload_documents()
     *
     * @uses  Upload the provider document values
     *
     * @created Vidhaya 
     *
     * @edited Vidhya R
     *
     * @param id, url
     *
     * @return json 
    */

    public function upload_documents(Request $request) {

        $document_url = $request->file('document_url');

        $validator = Validator::make(
         $request->all(),
         array(
           'document_url' => 'required',
           'document_id' => 'required|exists:documents,id',

         ),
         array(
             'exists' => 'Invalid Document ID',
           ));

        if ($validator->fails()) {
            
            $errors = implode(',', $validator->messages()->all());
            
            $response_array = ['success' => false,'error' => $errors,'error_code' => 101];
        
        } else {

            $documents = Document::all();

            $data = $result = [];

            foreach($documents as $doc) {

                $pro_doc = ProviderDocument::where('provider_id','=',$request->id)->where('document_id','=',$doc->id)->first();

                $data['id'] = $doc->id;
                $data['name'] = $doc->name;
                
                if($pro_doc)
                    $data['document_url'] = $pro_doc->document_url;
                else {
                    $data['document_url'] = "";
                }
                array_push($result,$data);
            
            }

            $document = ProviderDocument::where('provider_id',$request->id)->where('document_id',$request->document_id)->first();

            if($document) {
                if ($document_url != ""){
                    //deleting old image if exists
                    Helper::delete_picture($document->document_url);
                    $document->document_url = Helper::upload_picture($document_url);
                    $document->save();
                }
            } else {
                    $document = new ProviderDocument;
                    $document->document_url = Helper::upload_picture($document_url);
                    $document->provider_id = $request->id;
                    $document->document_id = $request->document_id;
                    $document->save();
            }
        
        }

        $response_array = array('success' => true, 'documents' => $documents);
        $response_array = Helper::null_safe($response_array);
        $response = response()->json($response_array, 200);
        return $response;
    
    }

    public function delete_document(Request $request) {

        $check_document = ProviderDocument::where('provider_id',$request->id)->where('document_id',$request->document_id)->first();

        if($check_document) {
           
            $check_document->delete();

            $response_array = array('success' => true);

        } else {
            $response_array = array('success'=> false,'error' => Helper::error_message(159) , 'error_code' => 159);
        }

        return response()->json($response_array, 200);
    
    }

    public function settings(Request $request) {

        DB::beginTransaction();

        try {

            $provider_details = Provider::find($request->id);

            $provider_details->push_notification_status = $provider_details->push_notification_status ? 0 : 1;

            $provider_details->save();

            DB::commit();

            $message = tr('push_notification_disable');

            if($provider_details->push_notification_status) {

                $message = tr('push_notification_enable');

            }

            $response_array = array('success' => true, 'message' => $message , 'push_notification_status' => $provider_details->push_notification_status);

            return response()->json($response_array, 200);

        } catch(Exception $e) {

            DB::rollback();

            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

            return response()->json($response_array);
        
        }

    }

    /**
     * @uses to get the request now details
     *
     * @param 
     *
     * @return
     */
    
    public function incoming_request(Request $request) {

        $provider = Provider::find($request->id);

        // Don't check availability

        $request_meta = RequestsMeta::where('requests_meta.provider_id',$provider->id)
                        ->leftJoin('requests', 'requests.id', '=', 'requests_meta.request_id')
                        ->where('requests_meta.status',REQUEST_META_OFFERED)
                        ->where('requests_meta.is_cancelled', false)
                        ->where('requests.job_type' , REQUEST_NOW)
                        ->get();
                        
        $provider_timeout = Setting::get('provider_timeout');

        $request_meta_data = array();

        if($request_meta) {

            foreach($request_meta as $each_request_meta) {

                $request_details = Requests::find($each_request_meta->request_id);

                $time_left_to_respond = $provider_timeout - (time() - strtotime($request_details->request_start_time));

                // Check the time is negative
                if($time_left_to_respond < 0) {
                    // Assign to the next provider
                    Helper::assign_next_provider($each_request_meta['request_id'],$request->id);
                }

                $data = RequestRepo::single_request_response($request_details, "" , $provider ? $provider->timezone : "");

                $data['time_left_to_respond'] = $time_left_to_respond;

                array_push($request_meta_data, $data);
            }
        
        }

        $response_array = ['success' => true,'data' => $request_meta_data];
    
        return response()->json($response_array, 200);
    
    }

    /**
     * @uses to get the ongoing request details
     *
     * @param 
     *
     * @return
     */

    public function request_status_check(Request $request) {

        $provider = Provider::find($request->id);

        $check_status = array(REQUEST_COMPLETED,REQUEST_CANCELLED,REQUEST_NO_PROVIDER_AVAILABLE,REQUEST_TIME_EXCEED_CANCELLED,REQUEST_PROVIDER_ASSIGNED,REQUEST_WAITING,REQUEST_CREATED,REQUEST_NONE);
        
        // *** NOTE *** Don't change the order By , based on that only aligning the requests response

        $requests = Requests::where('requests.provider_id', '=', $provider->id)
                            ->whereNotIn('requests.status', $check_status)
                            ->whereNotIn('requests.provider_status', array(PROVIDER_RATED))
                            ->orWhere(function($q) use ($provider) {
                                     $q->where('requests.provider_id', $provider->id)
                                        ->where('provider_status', PROVIDER_SERVICE_COMPLETED)      
                                       ->where('requests.status', REQUEST_COMPLETED);
                                 })
                            ->orderBy('job_type','ASC')
                            ->get();

        $requests_data = $invoice = array();

        if($requests) {

            $is_request_now = false;

            foreach($requests as $request_details) {

                
                $allow = true;

                // Check the request is now or later

                // If Now allow to send response

                // check the any normal request on the time start

                if($request_details->job_type == REQUEST_NOW) {
                    $is_request_now = DEFAULT_TRUE;
                }

                if($is_request_now != 0 && $request_details->job_type == REQUEST_LATER) {
                    $allow = DEFAULT_FALSE;
                }

                // check the any normal request on the time end

                // 1 Hour before only the provider can start the request
                $compare_time = Helper::add_date($request_details->request_date , '-1');

                if($request_details->job_type == REQUEST_LATER && $request_details->provider_status == PROVIDER_ACCEPTED && strtotime($compare_time) >= strtotime(date('Y-m-d H:i:s'))) {
                    $allow = DEFAULT_FALSE;
                }

                // Get the request Response

                if($allow) {

                    $requests_data = RequestRepo::single_request_response($request_details , "" , $provider ? $provider->timezone : "");

                    // This time is used for after service started => In app if the provider closed the app, while timer is running. 
                
                    $requests_data['service_time_diff'] = "00:00:00";

                    if($request_details->start_time != "0000-00-00 00:00:00") {

                        $time_diff = Helper::time_diff($request_details->start_time,date('Y-m-d H:i:s'));

                        $requests_data['service_time_diff'] = $time_diff->format('%h:%i:%s');

                    }

                    $allowed_status = [WAITING_PAYMENT_CONFIRM,REQUEST_COMPLETE_PENDING,REQUEST_COMPLETED,REQUEST_RATING];

                    if(in_array($request_details->status, $allowed_status)) {

                        $user = User::find($request_details->user_id);

                        $invoice_query = RequestPayment::where('request_id' , $request_details->id)
                                        ->leftJoin('requests' , 'request_payments.request_id' , '=' , 'requests.id')
                                        ->leftJoin('users' , 'requests.user_id' , '=' , 'users.id')
                                        ->leftJoin('cards' , 'users.card_id' , '=' , 'cards.id');
                        if($user->payment_mode == CARD) {
                            $invoice_query = $invoice_query->where('cards.is_default' , DEFAULT_TRUE) ;  
                        }

                        $invoice = $invoice_query->select('requests.provider_id as provider_id' , 'request_payments.total_time','requests.price_per_hour',
                                            'request_payments.payment_mode as payment_mode' , 'request_payments.base_price',
                                            'request_payments.time_price' , 'request_payments.tax_price' , 'request_payments.total',
                                            'cards.card_token','cards.customer_id','cards.last_four')
                                        ->get()->toArray();
                    }
                
                }
            
            }
        
        }
        
        $response_array = ['success' => true,'data' => $requests_data,'invoice' => $invoice];

        $response_array = Helper::null_safe($response_array);
    
        return response()->json($response_array, 200);

    }

    /**
     * @uses to accept the request now job 
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function request_accept(Request $request) {

        Log::info('Provider Accept Request Started');

        if(!RequestRepo::requests_exists($request->all(), $errors)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            Log::info('Provider Accept Request VALIDATION SUCCESS');

            $request_details = Requests::find($request->request_id);

            // Check the job type is REQUEST_NOW

            Log::info('Provider Accept Request ---  CHECK JOB TYPE');

            if($request_details->job_type != REQUEST_NOW) {

                Log::info('Provider Accept Request ---  CHECK JOB TYPE FAILED');

                $response_array = ['success' => false , 'error' => Helper::error_message(148) , 'error_code' => 148];

                return response()->json($response_array , 200);
            }

            $provider = Provider::find($request->id);

            //Check whether the request is cancelled by user.

            if($request_details->status == REQUEST_CANCELLED) {

                Log::info('Provider Accept Request --- REQUEST_CANCELLED FAILED');

                $response_array = ['success' => false,'error' => Helper::error_message(120),'error_code' => 120];
            
            } else {
                // Verify if request was indeed offered to this provider
                $request_meta = RequestsMeta::where('request_id', '=', $request->request_id)
                    ->where('provider_id', '=', $provider->id)
                    ->where('status', '=', REQUEST_META_OFFERED)->first();

                if (!$request_meta) {

                    Log::info('Provider Accept Request --- REQUEST_META_OFFERED FAILED');

                    // This request has not been offered to this provider. Abort.
                    $response_array = [ 'success' => false,'error' => Helper::error_message(125),'error_code' => 125];
                } else {
                    // Accept the offer
                    $request_details->provider_id = $request->id;
                    $request_details->status = REQUEST_ONGOING;
                    $request_details->provider_status = PROVIDER_ACCEPTED;
                    $request_details->save();

                    $price_per_hour = Setting::get('price_per_hour' , 1);

                    // Per Hour Price based on provider

                    if(Setting::get('per_hour_price_by') == 'provider') {

                        if($service_data = ProviderService::where('provider_id' , $request->id)->where('sub_category_id' , $request_details->sub_category_id)->first()) {

                            $price_per_hour = $service_data->price_per_hour ? $service_data->price_per_hour : "";
                        }

                    } else {
                        // Admin Based per hour price
                        $price_per_hour = $request_details->subCategory ? $request_details->subCategory->price : "";
                    }
                    
                    $request_details->price_per_hour = $price_per_hour ? $price_per_hour : Setting::get('price_per_hour' , 1);

                    $request_details->save();

                    // change waiting to respond state to normal state
                    $provider->waiting_to_respond = WAITING_TO_RESPOND_NORMAL;

                    // Check the request is later or not and update is available state 
                    if($request_details->job_type == REQUEST_NOW) {
                        $provider->is_available = PROVIDER_NOT_AVAILABLE;
                    }

                    $provider->save();

                    $user_details = $request_details->user ? $request_details->user : '';

                    $provider_details = $request_details->provider ? $request_details->provider : '';

                    if ($user_details) {

                        EmailTemplate::getRowValue(PROVIDER_ACCEPT_REQUEST, $user_details, $provider_details, $request_details);

                        $notification = NotificationTemplate::getRowValue(PROVIDER_ACCEPT_REQUEST, $user_details, $provider_details, $request_details);

                        $content = $notification ? $notification : PROVIDER_ACCEPT_REQUEST;

                        if (Setting::get('user_notification_control')) {

                            UserNotification::save_notification($user_details, $content, $request_details->provider_id, $request_details->id, PROVIDER_ACCEPT_REQUEST);

                        }

                        if (Setting::get('sms_notification')) {

                            SmsTemplate::getRowValue(PROVIDER_ACCEPT_REQUEST, $user_details);
                            
                        }

                    }

                    // Send Push Notification to User 

                    $title = Helper::push_message(8014);
                    $message = Helper::push_message(8015);

                    $this->dispatch(new sendPushNotification($request_details->user_id,USER,PUSH_USER_ONGOING_NOW,$title,$message));   

                    // No longer need request specific rows from RequestMeta
                    RequestsMeta::where('request_id', '=', $request->request_id)->forceDelete();
                    
                    $data = RequestRepo::single_request_response($request_details);

                    $response_array = ['success' => true,'message' => Helper::success_message(208),'data' => $data ,'code' => 208];
                
                }
            
            }
        }

        Log::info('Provider Accept Request Completed');
        
        return response()->json($response_array , 200);
    
    }

    /**
     * @uses to reject the request now job 
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function request_reject(Request $request) {

        Log::info('Provider Reqject Request Started');

        if(!RequestRepo::requests_exists($request->all(), $errors)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            Log::info('Provider Reqject Request ---- VALIDATION STARTED');

            $request_details = Requests::find($request->request_id);

            // Check the job type is REQUEST_NOW

            if($request_details->job_type != REQUEST_NOW) {

                Log::info('Provider Reqject Request ---- JOB TYPE FAILED');

                $response_array = ['success' => false , 'error' => Helper::error_message(148) , 'error_code' => 148];

                return response()->json($response_array , 200);
            }

            $provider = Provider::find($request->id);

            //Check whether the request is cancelled by user.

            if($request_details->status == REQUEST_CANCELLED) {

                Log::info('Provider Reqject Request ---- REQUEST_CANCELLED FAILED');

                $response_array = ['success' => false,'error' => Helper::error_message(120),'error_code' => 120];

            } else {

                // Verify if request was indeed offered to this provider

                $request_meta = RequestsMeta::where('request_id', '=', $request->request_id)
                    ->where('provider_id', '=', $provider->id)
                    ->where('status', '=', REQUEST_META_OFFERED)
                    ->first();

                if (!$request_meta) {
    
                    Log::info('Provider Reqject Request ---- REQUEST_META_OFFERED FAILED');

                    $response_array = ['success' => false, 'error' => Helper::error_message(125),'error_code' => 125];

                } else {

                    Helper::assign_next_provider($request->request_id,$request->id);

                    $response_array = Helper::null_safe(['success' => true,'id' => $request->id,'request_id' => $request->request_id,'message' => Helper::success_message(215) , 'code' => 215]);
                }
            
            }
       
        }

        Log::info('Provider Reject Request Completed');
 
        return response()->json($response_array , 200);
    
    }

    /**
     * @uses to send bid to the user for later request
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function send_bid_request(Request $request) {

        \Log::info($request->bid_amount);

        if(!(float) $request->bid_amount) {

            $response_array = ['success' => false , 'error' => Helper::error_message(163) , 'error_code' => 163];

            return response()->json($response_array , 200);
        }

        Log::info('Provider - SEND Bid  - Started');

        if(!RequestRepo::requests_exists($request->all(), $errors)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $request_details = Requests::find($request->request_id);

            // Check the job type is REQUEST_LATER

            if($request_details->job_type != REQUEST_LATER) {

                $response_array = ['success' => false , 'error' => Helper::error_message(149) , 'error_code' => 149];

                return response()->json($response_array , 200);
            }

            if($request_details->status == REQUEST_CANCELLED) {

                Log::info('Provider Reject Request ---- REQUEST_CANCELLED FAILED');

                $response_array = ['success' => false,'error' => Helper::error_message(120),'error_code' => 120];
                return response()->json($response_array, 200);
            }

            $provider = Provider::find($request->id);

            $request_meta = RequestsMeta::where('provider_id' , $request->id)->where('request_id' ,$request->request_id)->first();

            Log::info('Provider - SEND Bid  - CHECK The Request Type as DIRECT');

            // Check the request is directly assigned to this provider

            if(in_array($request_details->request_type, [REQUEST_TYPE_LOCATION , REQUEST_TYPE_CATEGORY])) {

                Log::info('Provider - SEND Bid  - Request Type *** REQUEST_TYPE_LOCATION , REQUEST_TYPE_CATEGORY');

                // category type - There won't be previously created requests meta record , we need to create a new record

                if($request_meta) {

                    // check the current status of the requests

                    if($request_meta->status != REQUEST_META_BID) {

                        $request_meta->bid_amount = $request->bid_amount;
                        $request_meta->status = REQUEST_META_BID;
                        $request_meta->save();

                        $response_array = ['success' => true , 'request_id' => $request_details->id, 'status' => $request_details->status , 'message' => Helper::success_message(222) , 'code' => 222];

                    } else {
                        $response_array = ['success' => false , 'error' => Helper::error_message(109) , 'error_code' => 109];
                    }
                
                } else {

                    $request_meta = new RequestsMeta;
                    $request_meta->provider_id = $request->id;
                    $request_meta->request_id = $request->request_id;
                    $request_meta->bid_amount = $request->bid_amount;
                    $request_meta->status = REQUEST_META_BID;
                    $request_meta->save();   

                    $response_array = ['success' => true , 'request_id' => $request_details->id , 'status' => $request_details->status , 'message' => Helper::success_message(222) , 'code' => 222];

                }

                $request_details->status = REQUEST_WAITING;

                $request_details->save();

                // Send Push Notification to User 

                $title = Helper::push_message(8016); $message = Helper::push_message(8017);

                $page_type = PUSH_USER_POSTED_TAB;

                $this->dispatch(new sendPushNotification($request_details->user_id,USER,PUSH_USER_SINGLE,$title,$message , "" , $request_details->id , $page_type));


                if ($request_details->user) {

                    $email = EmailTemplate::getRowValue(PROVIDER_BID_REQUEST, $request_details->user, $request_details->provider, $request_details);

                    $notification = NotificationTemplate::getRowValue(PROVIDER_BID_REQUEST, $request_details->user, $request_details->provider, $request_details);

                    $content = $notification ? $notification : PROVIDER_BID_REQUEST;

                   // dd($content);

                    if (Setting::get('user_notification_control')) {

                        UserNotification::save_notification($request_details->user, $content, $request_details->provider_id, $request_details->id, PROVIDER_BID_REQUEST);

                    }

                    if (Setting::get('sms_notification')) {

                        SmsTemplate::getRowValue(PROVIDER_BID_REQUEST, $request_details->user);
                            
                    }

                }

            } else {

                Log::info('Provider - SEND Bid  - Request Type *** DIRECT');

                $check_request_meta = RequestsMeta::where('provider_id' , $request->id)->where('request_id' , $request->request_id)->first();

                if($check_request_meta) {

                    if($check_request_meta->status == REQUEST_META_BID) {

                        $response_array = ['success' => false , 'error' => Helper::error_message(139) , 'error_code' => 139];

                        return response()->json($response_array , 200);
                    }
                    $request_meta = $check_request_meta;
                } else {
                    $request_meta = new RequestsMeta;      
                }

                $request_meta->provider_id = $request->id;
                $request_meta->request_id = $request_details->id;
                $request_meta->bid_amount = $request->bid_amount;
                $request_meta->status = REQUEST_META_BID;
                $request_meta->save();

                $request_details->status = REQUEST_WAITING;
                $request_details->save();

                // Send Push Notification to User 

                $title = Helper::push_message(8016); $message = Helper::push_message(8017);

                $page_type = PUSH_USER_POSTED_TAB;

                $this->dispatch(new sendPushNotification($request_details->user_id,USER,PUSH_USER_SINGLE,$title,$message , "" , $request_details->id , $page_type));


                 if ($request_details->user) {

                    $email = EmailTemplate::getRowValue(PROVIDER_BID_REQUEST, $request_details->user, $request_details->provider, $request_details);

                    $notification = NotificationTemplate::getRowValue(PROVIDER_BID_REQUEST, $request_details->user, $request_details->provider, $request_details);

                    $content = $notification ? $notification : PROVIDER_BID_REQUEST;

                    if (Setting::get('user_notification_control')) {

                        UserNotification::save_notification($request_details->user, $content,$request_details->provider_id, $request_details->id, PROVIDER_BID_REQUEST);

                    }

                    if (Setting::get('sms_notification')) {

                        SmsTemplate::getRowValue(PROVIDER_BID_REQUEST, $request_details->user);
                            
                    }

                }

                $response_array = ['success' => true , 'request_meta_id' => $request_meta->id , 'request_id' => $request_details->id , 'status' => $request_details->status, 'message' => Helper::success_message(222) , 'code' => 222];

            }
        }

        Log::info('Provider Send Bid ---- Completed');
        
        return response()->json($response_array, 200);

    }

    /**
     * Used for once the user accept the bid, provider has to confirm the request
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function confirm_request(Request $request) {

        if(!RequestRepo::requests_exists($request->all(), $errors)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $request_details = Requests::find($request->request_id);

            // Check the job type is REQUEST_LATER

            if($request_details->job_type != REQUEST_LATER) {

                $response_array = ['success' => false , 'error' => Helper::error_message(149) , 'error_code' => 149];

                return response()->json($response_array , 200);
            }

            // Check the request is already cancelled

            if($request_details->status == REQUEST_CANCELLED) {

                Log::info('Provider Reject Request ---- REQUEST_CANCELLED FAILED');

                $response_array = ['success' => false,'error' => Helper::error_message(120),'error_code' => 120];
                return response()->json($response_array, 200);
            }

            // Get Assigned provider request meta details

            $request_meta = RequestsMeta::where('request_id' , $request->request_id)->where('provider_id' , $request->id)->where('status' , REQUEST_META_ASSIGNED )->first();

            // Check the status of the request

            if($request_meta && $request_details->status != PROVIDER_ACCEPTED) {

                RequestsMeta::where('request_id' , $request->request_id)->forceDelete();

                $request_details->provider_id = $request->id;
                $request_details->status = REQUEST_ONGOING;
                $request_details->provider_status = PROVIDER_ACCEPTED;
                $request_details->confirm_time = date('Y-m-d H:i:s');
                $request_details->save();

                // Send Push Notification to User 

                $title = Helper::push_message(8030);
                $message = Helper::push_message(8031);

                $page_type = PUSH_USER_ONGOING_TAB;

                $this->dispatch(new sendPushNotification($request_details->user_id,USER,PUSH_USER_SINGLE,$title,$message , "" , $request_details->id , $page_type));

                if ($request_details->user) {

                    $email = EmailTemplate::getRowValue(PROVIDER_ACCEPT_REQUEST, $request_details->user, $request_details->provider, $request_details);

                    $notification = NotificationTemplate::getRowValue(PROVIDER_ACCEPT_REQUEST, $request_details->user, $request_details->provider, $request_details);

                    $content = $notification ? $notification : PROVIDER_ACCEPT_REQUEST;

                    if (Setting::get('user_notification_control')) {

                        UserNotification::save_notification($request_details->user, $content,$request_details->provider_id,$request_details->id, PROVIDER_ACCEPT_REQUEST);

                    }
                    if (Setting::get('sms_notification')) {

                        SmsTemplate::getRowValue(PROVIDER_ACCEPT_REQUEST, $request_details->user);
                            
                    }

                }

                $response_array = ['success' => true , 'request_id' => $request_details->id , 'message' => Helper::success_message(223) , 'code' => 223];

            } else {
                $response_array = array('success' => false, 'error' => Helper::error_message(140), 'error_code' => 140);
            }

        }

        return response()->json($response_array, 200);

    }

    /**
     * Used for request_date of the job
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function check_provider_started(Request $request) {

        $validator = Validator::make($request->all(),
            array(
                'request_id' => 'required|integer|exists:requests,id,provider_id,'.$request->id,
            ),
            array(
                'exists' => 'The :attribute doesn\'t belong to this provider:'
            )
        
        );
        
        if ($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false, 'error' => $errors ,'error_code' => 101];

        } else {

            if($request_details = Requests::find($request->request_id)) {

                if($request_details->job_type == REQUEST_NOW) {

                    $response_array = ['success' => true];

                } else {

                    if(strtotime(date('Y-m-d H:i:s')) >= strtotime($request_details->request_date)) {
                        $response_array = ['success' => true];
                    } else {

                        $response_array = ['success' => false , 'error' => Helper::error_message(127) , 'error_code' => 127];
                    }
                }

            } else {
                $response_array = ['success' => false , 'error' => Helper::error_message(124) , 'error_code' => 124];
            }

        }

        return response()->json($response_array , 200);

    }

    /**
     * @uses to start from current location of the provider
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function provider_started(Request $request) {

        $provider = Provider::find($request->id);

        if(!ProviderRepo::request_validation($request->all() , $errors , $provider)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $request_id = $request->request_id;

            $allow_action = DEFAULT_TRUE; // Check any ongoing request is there on the time

            $request_details = Requests::where('id', '=', $request_id)
                                ->where('provider_id', '=', $provider->id)
                                ->where('provider_status' , PROVIDER_ACCEPTED)
                                ->where('status', REQUEST_ONGOING)
                                ->first();

            // Current state being validated in order to prevent accidental change of state

            if ($request_details && intval($request_details->provider_status) != PROVIDER_STARTED)  {

                // Check the requested time and current time greater than or equal

                if($request_details->job_type == REQUEST_LATER) {

                    $check_normal_ongoing = Requests::where('provider_id' , $request->id)->where('status' , REQUEST_ONGOING)->where('job_type' , REQUEST_NOW)->count();

                    // Check any later ongoing requests

                    $check_provider_status = [PROVIDER_STARTED,PROVIDER_ARRIVED,PROVIDER_SERVICE_STARTED,PROVIDER_SERVICE_COMPLETED];

                    $check_later_ongoing = Requests::where('provider_id' , $request->id)
                                            ->where('status' , REQUEST_ONGOING)
                                            ->whereIn('provider_status' , $check_provider_status)
                                            ->where('job_type' , REQUEST_LATER)
                                            ->count();

                    if($check_normal_ongoing != 0) {
                        $allow_action = DEFAULT_FALSE;
                    }

                    if($check_later_ongoing) {
                        $allow_action = DEFAULT_FALSE;          
                    }

                }

                if($allow_action == DEFAULT_TRUE) {

                    // Current state being validated in order to prevent accidental change of state

                    if ($request_details && intval($request_details->provider_status) != PROVIDER_STARTED ) {

                        $request_details->status = REQUEST_ONGOING;
                        $request_details->provider_status = PROVIDER_STARTED;
                        $request_details->start_time = date("Y-m-d H:i:s");
                        $request_details->save();

                        if($request_details->job_type == REQUEST_LATER) {

                            $provider->is_available = false;

                            $provider->save();

                        } 

                        $user_details = $request_details->user ? $request_details->user : '';

                        $provider_details = $request_details->provider ? $request_details->provider : '';



                        if ($user_details) {

                            $email = EmailTemplate::getRowValue(PROVIDER_STARTED_REQUEST, $user_details, $provider_details, $request_details);

                            $notification = NotificationTemplate::getRowValue(PROVIDER_STARTED_REQUEST, $user_details, $provider_details, $request_details);

                            $content = $notification ? $notification : PROVIDER_STARTED_REQUEST;

                            //dd($content);

                            if (Setting::get('user_notification_control')) {

                                UserNotification::save_notification($user_details, $content,$request_details->provider_id, $request_details->id, PROVIDER_STARTED_REQUEST);

                            }

                            if (Setting::get('sms_notification')) {

                                SmsTemplate::getRowValue(PROVIDER_STARTED_REQUEST, $user_details);
                                    
                            }

                        }

                       // Send Push Notification to User 

                        $title = Helper::push_message(8018);
                        $message = Helper::push_message(8019);

                        $page_id = $request_details->job_type == REQUEST_NOW ? PUSH_USER_ONGOING_NOW : PUSH_USER_ONGOING_LATER;

                        $this->dispatch(new sendPushNotification($request_details->user_id,USER,$page_id,$title,$message,"",$request_details->id , ""));
                   
                        $response_array = ['success' => true,'status' => PROVIDER_STARTED,'message' => Helper::success_message(209) , 'code' => 209];
                    
                    } else {
                        $response_array = array('success' => false, 'error' => Helper::error_message(126), 'error_code' => 126);

                        Log::info('Provider status Error:: Old state='.$request_details->provider_status.' and current state='.$current_state);
                    }

                } else {
                    $response_array = array('success' => false , 'error' => Helper::error_message(128) , 'error_code' => 128);
                }

            } else {

                Log::info('already Some requests is ongoing');

                $response_array = array('success' => false, 'error' => Helper::error_message(126), 'error_code' => 126);
            }
        
        }

        return response()->json($response_array , 200);
    }

    /**
     * @uses to arrived the job location status update
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function provider_arrived(Request $request) {

        $provider = Provider::find($request->id);

        if(!ProviderRepo::request_validation($request->all() , $errors , $provider)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $request_details = Requests::where('id', '=', $request->request_id)
                                ->where('provider_id', '=', $provider->id)
                                ->where('provider_status' , PROVIDER_STARTED)
                                ->where('status', REQUEST_ONGOING)
                                ->first();

            // Current state being validated in order to prevent accidental change of state
            if ($request_details && intval($request_details->provider_status) != PROVIDER_ARRIVED) {

                $request_details->status = REQUEST_ONGOING;
                $request_details->provider_status = PROVIDER_ARRIVED;
                $request_details->save();


                $user_details = $request_details->user ? $request_details->user : '';

                $provider_details = $request_details->provider ? $request_details->provider : '';

                if ($user_details) {

                    $email = EmailTemplate::getRowValue(PROVIDER_ARRIVED_REQUEST, $user_details, $provider_details, $request_details);

                    $notification = NotificationTemplate::getRowValue(PROVIDER_ARRIVED_REQUEST, $user_details, $provider_details, $request_details);

                    $content = $notification ? $notification : PROVIDER_ARRIVED_REQUEST;

                    if (Setting::get('user_notification_control')) {

                        UserNotification::save_notification($user_details, $content,$request_details->provider_id, $request_details->id, PROVIDER_ARRIVED_REQUEST);
                    }

                    if (Setting::get('sms_notification')) {

                        SmsTemplate::getRowValue(PROVIDER_ARRIVED_REQUEST, $user_details);
                            
                    }

                }

                // Send Push Notification to User 

                $title = Helper::push_message(8020);
                $message = Helper::push_message(8021);

                $page_id = $request_details->job_type == REQUEST_NOW ? PUSH_USER_ONGOING_NOW : PUSH_USER_ONGOING_LATER;

                $this->dispatch(new sendPushNotification($request_details->user_id,USER,$page_id,$title,$message , "",$request_details->id , ""));
           
                $response_array =['success' => true,'status' => REQUEST_ONGOING,'message' => Helper::success_message(210) , 'code' => 210];
            } else {
                $response_array = ['success' => false, 'error' => Helper::error_message(129), 'error_code' => 129];
            }
        }
        return response()->json($response_array , 200);
    
    }

    /**
     * @uses to upadte request status
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function request_started(Request $request) {

        Log::info("REQUEST STARTED".print_r($request->all(), true));

        $provider = Provider::find($request->id);

        if(!ProviderRepo::request_validation($request->all() , $errors , $provider)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $request_details = Requests::where('id', '=', $request->request_id)
                                ->where('provider_id', '=', $provider->id)
                                ->where('provider_status' , PROVIDER_ARRIVED)
                                ->where('status', REQUEST_ONGOING)
                                ->first();

            // Current state being validated in order to prevent accidental change of state
            if ($request_details && intval($request_details->provider_status) != PROVIDER_SERVICE_STARTED) {

                if($request->hasFile('before_image')) {
                    $request_details->before_image = Helper::upload_avatar('uploads/jobs',$request->file('before_image'));
                }

                $request_details->start_time = date("Y-m-d H:i:s");
                $request_details->status = REQUEST_ONGOING;
                $request_details->provider_status = PROVIDER_SERVICE_STARTED;
                $request_details->save();


                $user_details = $request_details->user ? $request_details->user : '';

                $provider_details = $request_details->provider ? $request_details->provider : '';

                if ($user_details) {

                    $email = EmailTemplate::getRowValue(PROVIDER_SERVICE_STARTED_REQUEST, $user_details, $provider_details, $request_details);

                    $notification = NotificationTemplate::getRowValue(PROVIDER_SERVICE_STARTED_REQUEST, $user_details, $provider_details, $request_details);

                    $content = $notification ? $notification : PROVIDER_SERVICE_STARTED_REQUEST;

                    if (Setting::get('user_notification_control')) {

                        UserNotification::save_notification($user_details, $content,$request_details->provider_id, $request_details->id, PROVIDER_SERVICE_STARTED_REQUEST);

                    }

                    if (Setting::get('sms_notification')) {

                        SmsTemplate::getRowValue(PROVIDER_SERVICE_STARTED_REQUEST, $user_details);
                            
                    }

                }

                // Send Push Notification to User 

                $title = Helper::push_message(8022);
                $message = Helper::push_message(8023);

                $page_id = $request_details->job_type == REQUEST_NOW ? PUSH_USER_ONGOING_NOW : PUSH_USER_ONGOING_LATER;

                $this->dispatch(new sendPushNotification($request_details->user_id,USER,$page_id,$title,$message , "",$request_details->id , ""));
           
                $response_array = ['success' => true,'request_id' => $request->request_id,'status' => REQUEST_ONGOING,'message' => Helper::success_message(211) , 'code' => 211
                ];
           
            } else {

                $response_array = ['success' => false, 'error' => Helper::error_message(127), 'error_code' => 127];
                Log::info(Helper::error_message(127));
            }
        }
        return response()->json($response_array , 200);
    
    }

    /**
     * @method request_completed()
     *
     * @uses to upadte request status
     * 
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer - request_id
     *
     * @return json response
    */

    public function request_completed(Request $request) {

        $provider_details = Provider::find($request->id);

        if(!ProviderRepo::request_validation($request->all() , $errors , $provider_details)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $request_details = Requests::where('id', '=', $request->request_id)
                                ->where('provider_id', '=', $provider_details->id)
                                ->where('provider_status' , PROVIDER_SERVICE_STARTED)
                                ->where('status', REQUEST_ONGOING)
                                ->first();

            // Current state being validated in order to prevent accidental change of state

            if ($request_details && intval($request_details->provider_status) != PROVIDER_SERVICE_COMPLETED) {

                if($request->hasFile('after_image')) {
                    $request_details->after_image = Helper::upload_avatar('uploads/jobs' ,$request->file('after_image'));
                }

                //The availability will be change after the provider rated for user

                // Call Invoice function

                if(RequestRepo::invoice($request_details)) {

                    $request_details->status = REQUEST_COMPLETE_PENDING;

                    $request_details->provider_status = PROVIDER_SERVICE_COMPLETED;
                    
                    $request_details->save();
                }

                $user_details = $request_details->user ? $request_details->user : '';

                if ($user_details) {

                    $email = EmailTemplate::getRowValue(PROVIDER_SERVICE_COMPLETED_REQUEST, $user_details, $provider_details, $request_details);

                    $notification = NotificationTemplate::getRowValue(PROVIDER_SERVICE_COMPLETED_REQUEST, $user_details, $provider_details, $request_details);

                    $content = $notification ? $notification : PROVIDER_SERVICE_COMPLETED_REQUEST;

                    if (Setting::get('user_notification_control')) {

                        UserNotification::save_notification($user_details, $content,$request_details->provider_id,$request_details->id, PROVIDER_SERVICE_COMPLETED_REQUEST);
                    }

                    if (Setting::get('sms_notification')) {

                        SmsTemplate::getRowValue(PROVIDER_SERVICE_COMPLETED_REQUEST, $user_details);
                    }

                }

                // Send Push Notification to User 

                $title = Helper::push_message(8024);
                $message = Helper::push_message(8025);

                $page_id = $request_details->job_type == REQUEST_NOW ? PUSH_USER_ONGOING_NOW : PUSH_USER_ONGOING_LATER;

                $this->dispatch(new sendPushNotification($request_details->user_id,USER,$page_id,$title,$message,"",$request_details->id , ""));

                $requests_data = RequestRepo::single_request_response($request_details);

                // This time is used for after service started => In app if the provider closed the app, while timer is running. 
            
                $requests_data['service_time_diff'] = "00:00:00";

                if($request_details->start_time != "0000-00-00 00:00:00") {

                    $time_diff = Helper::time_diff($request_details->start_time,date('Y-m-d H:i:s'));

                    $requests_data['service_time_diff'] = $time_diff->format('%h:%i:%s');

                }

                $user = User::find($request_details->user_id);

                $invoice_query = RequestPayment::where('request_id' , $request_details->id)
                                ->leftJoin('requests' , 'request_payments.request_id' , '=' , 'requests.id')
                                ->leftJoin('users' , 'requests.user_id' , '=' , 'users.id')
                                ->leftJoin('cards' , 'users.card_id' , '=' , 'cards.id');

                if($user->payment_mode == CARD) {
                    $invoice_query = $invoice_query->where('cards.is_default' , DEFAULT_TRUE) ;  
                }

                $invoice = $invoice_query->select('requests.provider_id as provider_id' , 
                    'request_payments.total_time',
                    'requests.price_per_hour',
                                    'request_payments.payment_mode as payment_mode' , 'request_payments.base_price',
                                    'request_payments.time_price' , 'request_payments.tax_price' , 'request_payments.total',
                                    'cards.card_token','cards.customer_id','cards.last_four')
                                ->get()
                                ->toArray();

                $response_array = ['success' => true,
                    'request_id' => $request->request_id,
                    'status' => REQUEST_COMPLETE_PENDING,'data' => $requests_data , 
                    'invoice' => $invoice,
                    'message' => Helper::success_message(212) , 
                    'code' => 212
                ];

            } else {
                
                $response_array = ['success' => false, 'error' => Helper::error_message(131), 'error_code' => 131];

                Log::info(Helper::error_message(131));
            }
        
        }
        
        return response()->json($response_array , 200);
    
    }

    /**
     * If user paid as COD means the provider will confirm the payment
     *
     */

    public function request_payment_confirm(Request $request) {

        $validator = Validator::make(
            $request->all(),
            array(
                'request_id' => 'required|integer|exists:requests,id,provider_id,'.$request->id,
            ),
            array(
                'exists' => 'The :attribute doesn\'t belong to user:'.$request->id,
            )
        );
    
        if ($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false, 'error' => $errors, 'error_code' => 101];
        
        } else {

            $request_details = Requests::find($request->request_id);

            if($request_details->status == WAITING_PAYMENT_CONFIRM && $request_details->status != REQUEST_RATING) {

                $request_details->status = REQUEST_RATING;
                
                $request_details->is_paid = DEFAULT_TRUE;
                
                $request_details->save();

                if($request_payment = RequestPayment::where('request_id' , $request->request_id)->first()) {

                    $request_payment->is_paid = DEFAULT_TRUE;

                    $request_payment->paid_date = date('Y-m-d H:i:s');

                    $request_payment->save();

                    Log::info("PROVIDER TOTAL - provider_payment");

                    Helper::provider_payment($request->id, $request_payment->sub_total , COD , $request_payment->admin_amount , $request_payment->provider_amount);

                } else {

                    Log::info("PROVIDER NOT TOTAL - provider_payment");

                    Helper::provider_payment($request->id, $request_payment->sub_total, COD);
                }

                // Send Push Notification to User

                $title = Helper::push_message(8026);
              
                $message = Helper::push_message(8027);

                $page_id = $request_details->job_type == REQUEST_NOW ? PUSH_USER_ONGOING_NOW : PUSH_USER_ONGOING_LATER;

                $this->dispatch(new sendPushNotification($request_details->user_id,USER,$page_id,$title,$message ,"", $request_details->id , ""));

                $data = [];

                $data['request_id'] = $request_details->id;
                $data['user_id'] = $request_details->user_id;
                $data['user_name'] = $request_details->user->name;
                $data['user_picture'] = $request_details->user->picture;



                if ($request_details->user) {

                    $email = EmailTemplate::getRowValue(PROVIDER_AMOUNT_CONFIRMATION, $request_details->user, $request_details->provider, $request_details);

                    $notification = NotificationTemplate::getRowValue(PROVIDER_AMOUNT_CONFIRMATION, $request_details->user, $request_details->provider, $request_details);

                    $content = $notification ? $notification : PROVIDER_AMOUNT_CONFIRMATION;

                    if (Setting::get('user_notification_control')) {

                        UserNotification::save_notification($request_details->user, $content,$request_details->provider_id,$request_details->id, PROVIDER_AMOUNT_CONFIRMATION);

                    }

                    if (Setting::get('sms_notification')) {

                        SmsTemplate::getRowValue(PROVIDER_AMOUNT_CONFIRMATION, $request_details->user);
                            
                    }

                }
                
                $response_array = array('success' => true , 'message' => Helper::success_message(229) , 'data' => $data , 'code' => 229);
            } else {
                $response_array = ['success' => false , 'error' => Helper::error_message(136) ,'error_code' =>136];
            }
        }
        return response()->json($response_array, 200);
    }

    /**
     * @uses to upadte request status
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function request_rating(Request $request) {

        $provider = Provider::find($request->id);

        $validator = Validator::make(
            $request->all(),
            array(
                'request_id' => 'required|integer|exists:requests,id,provider_id,'.$provider->id.'|unique:provider_ratings,request_id',
                'rating' => 'integer',
                'comments' => 'max:255'
            ),
            array(
                'exists' => 'The :attribute doesn\'t belong to provider:'.$provider->id,
                'unique' => 'The :attribute already rated.'
            )
        );
    
        if ($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false, 'error' => $errors ,'error_code' => 101];

        } else {

            $request_details = Requests::where('id' ,$request->request_id)
                    ->whereIn('status' , array(REQUEST_COMPLETE_PENDING,REQUEST_RATING,REQUEST_COMPLETED))
                    ->where('provider_status' , PROVIDER_SERVICE_COMPLETED)
                    ->first();

            if ($request_details && intval($request_details->provider_status) != PROVIDER_RATED) { 
                
                if($request->has('rating')) {
                    //Save Rating
                    $rev_user = new ProviderRating();
                    $rev_user->provider_id = $request_details->provider_id;
                    $rev_user->user_id = $request_details->user_id;
                    $rev_user->request_id = $request_details->id;
                    $rev_user->rating = $request->rating;
                    $rev_user->comment = $request->comment ?: '';
                    $rev_user->save();
                }

                $request_details->provider_status = PROVIDER_RATED;
                $request_details->save();

                //Update provider availability
                $provider = Provider::find($request_details->provider_id);
                $provider->is_available = PROVIDER_AVAILABLE;
                $provider->save();

                if ($request_details->user) {

                    $email = EmailTemplate::getRowValue(JOB_REQUEST_COMPLETED, $request_details->user, $request_details->provider, $request_details);

                    $notification = NotificationTemplate::getRowValue(JOB_REQUEST_COMPLETED, $request_details->user, $request_details->provider, $request_details);

                    $content = $notification ? $notification : JOB_REQUEST_COMPLETED;

                    if (Setting::get('user_notification_control')) {

                    UserNotification::save_notification($request_details->user, $content,$request_details->provider_id,$request_details->id, JOB_REQUEST_COMPLETED);

                    }

                    if (Setting::get('sms_notification')) {

                        SmsTemplate::getRowValue(JOB_REQUEST_COMPLETED, $request_details->user);
                            
                    }

                }

                $response_array = Helper::null_safe(['success' => true,'status' => $request_details->status,'message' => Helper::success_message(213) , 'code' => 213]);
            } else {

                $response_array = ['success' => false , 'error' => Helper::error_message(132) , 'error_code' => 132];
            }
        }
        return response()->json($response_array , 200);
    
    }

    /**
     *
     * @method request_cancel()
     *
     * @uses to upadte request status
     * 
     * @created vidhya R
     *
     * @edited vidhya R
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function request_cancel(Request $request) {

        Log::info('Provider Request cancel started');

        $provider = Provider::find($request->id);

        $request_cancel = true;

        if(!ProviderRepo::request_validation($request->all() , $errors , $provider , $request_cancel)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $request_details = Requests::find($request->request_id);

            $check_status = [PROVIDER_NONE,PROVIDER_ASSIGNED,PROVIDER_ACCEPTED,PROVIDER_STARTED];

            // Check whether request cancelled previously

            if($request_details->status != REQUEST_CANCELLED) {

                Log::info('Provider Cancel Status Check SUCCESS');

                // Check whether request eligible for cancellation

                if(in_array($request_details->provider_status, $check_status)) {

                    Log::info('Provider Cancel - Provider Status Check SUCCESS');

                    /** update status of the request to cancellation */

                    $request_details->status = REQUEST_CANCELLED;

                    $request_details->cancelled_date = date('Y-m-d H:i:s');

                    $request_details->cancelled_by = PROVIDER;

                    $request_details->cancelled_reason = $request->reason ?: $request->cancelled_reason;

                    $request_details->save();

                    if($request_details->provider_id != false) {

                        $provider = Provider::find($request_details->provider_id);

                        if($request_details->job_type == REQUEST_NOW) {

                            $provider->is_available = PROVIDER_AVAILABLE;
                        }

                        // Check for later request , provider available change

                        if($request_details->job_type == REQUEST_LATER) {

                            if(in_array($request_details->status , [PROVIDER_STARTED , PROVIDER_ARRIVED])) {

                                $provider->is_available = PROVIDER_AVAILABLE;
                                
                            }
                        }

                        $provider->save();
                   
                    }

                    // Send Push Notification to User

                    $title = Helper::push_message(8028); $message = Helper::push_message(8029);

                    $page_type = PUSH_USER_POSTED_TAB;


                    if ($request_details->user) {

                        $email = EmailTemplate::getRowValue(PROVIDER_CANCEL_REQUEST, $request_details->user, $request_details->provider, $request_details);

                        $notification = NotificationTemplate::getRowValue(PROVIDER_CANCEL_REQUEST, $request_details->user, $request_details->provider, $request_details);

                        $content = $notification ? $notification : PROVIDER_CANCEL_REQUEST;

                        if (Setting::get('user_notification_control')) {

                            UserNotification::save_notification($request_details->user, PROVIDER_CANCEL_REQUEST,$request_details->provider_id,$request_details->id, PROVIDER_CANCEL_REQUEST);

                        }

                        if (Setting::get('sms_notification')) {

                            SmsTemplate::getRowValue(PROVIDER_CANCEL_REQUEST, $request_details->user);
                                
                        }

                    }

                    $this->dispatch(new sendPushNotification($request_details->user_id,USER,PUSH_USER_JOB_DASHBOARD,$title,$message , "" ,"", $page_type));

                    // No longer need request specific rows from RequestMeta
                    RequestsMeta::where('request_id', '=', $request->request_id)->forceDelete();

                    // Send mail notifications

                    $response_array = ['success' => true,'request_id' => $request_details->id , 'message' => Helper::success_message(207) , 'code' => 207];
               
                } else {
                    
                    Log::info('Provider Cancel - Provider Status Check FAILED');

                    $response_array = ['success' => false, 'error' => Helper::error_message(133), 'error_code' => 133 ];
                }
            
            } else {

                Log::info('Provider Cancel Status Check FAILED');

                $response_array = ['success' => false, 'error' => Helper::error_message(120), 'error_code' => 120 ];
            }
        }

        Log::info('Provider cancel completed');

        return response()->json($response_array, 200);
    
    }

    /**
     * @uses to cancel bidding 
     * 
     * @param integer - request_id
     *
     * @return json response
    */

    public function request_bidding_reject(Request $request) {

        Log::info('Provider Request Bidding cancel started');

        $provider = Provider::find($request->id);

        $request_details = Requests::find($request->request_id);

        $check_status = [PROVIDER_ASSIGNED];

        // Check whether request cancelled previously

        if($request_details->status != REQUEST_CANCELLED) {

            Log::info('Provider Bidding Cancel Status Check SUCCESS');

            // Check whether request eligible for cancellation

            if(in_array($request_details->provider_status, $check_status)) {

                Log::info('Provider Bidding Cancel - Provider Status Check SUCCESS');

                /*Update status of the request to cancellation*/
                $request_details->status = REQUEST_CANCELLED;
                $request_details->cancelled_date = date('Y-m-d H:i:s');
                $request_details->cancelled_by = PROVIDER;
                $request_details->provider_reason = $request->reason;
                $request_details->save();

                // No longer need request specific rows from RequestMeta

                RequestsMeta::where('request_id', '=', $request->request_id)->forceDelete();

                // Send mail notifications

                $response_array = ['success' => true,'request_id' => $request_details->id , 'message' => Helper::success_message(230) , 'code' => 230];
           
            } else {
                
                Log::info('Provider Cancel - Provider Status Check FAILED');

                $response_array = ['success' => false, 'error' => Helper::error_message(133), 'error_code' => 133 ];
            }
        
        } else {

            Log::info('Provider Cancel Status Check FAILED');

            $response_array = ['success' => false, 'error' => Helper::error_message(120), 'error_code' => 120 ];
        }

        Log::info('Provider Bidding cancel completed');

        return response()->json($response_array, 200);
    
    }

    /**
     * @uses to get the available jobs
     * 
     * @param integer - request_id
     *
     * @return json response
     */

    public function available_requests(Request $request) {

        // Get the provider selected categories and sub categories

        $provider_services = ProviderService::where('provider_id' , $request->id)->where('status' , APPROVED)->get();

        $category_ids = $sub_category_ids = []; 

        if($provider_services) {

            foreach ($provider_services as $key => $provider_service) {

                #check provider is subscribed to category subscriptions AND expiry date for subscription
                $has_subscriptions = is_category_subscription($provider_service->category_id);
                                
                $is_subscribed = NO;
                
                if($has_subscriptions) {

                    $is_subscribed = check_provider_subscribed_category($request->id, $provider_service->category_id);

                    if($is_subscribed) {
                        
                        $category_ids[] = $provider_service->category_id;

                        $sub_category_ids[] = $provider_service->sub_category_id;
                    }

                } else {
                   
                    #if cateogory has no paid subscriptions
                    $category_ids[] = $provider_service->category_id;

                    $sub_category_ids[] = $provider_service->sub_category_id;
                
                }

            }
        }

        // Check the provider have categories and sub categories

        if($category_ids && $sub_category_ids) {

            $ongoing_confirmed_dates = [];

            // Get Ongoing Requests dates

            $ongoing_confirmed_requests = ProviderRepo::check_ongoing_confirmed_requests($request->id);

            if(count($ongoing_confirmed_requests) > 0) {

                foreach ($ongoing_confirmed_requests as $oc => $ongoing_confirmed_request) {
                    $ongoing_confirmed_dates[] = $ongoing_confirmed_request->request_date;
                }
            }

            $bidded_requests = ProviderRepo::get_bidded_request_ids($request->id);

            $timezone = "";

            if($provider_details = Provider::find($request->id)) {
                $timezone = $provider_details->timezone;
            }

            $request_date = common_date(date("Y-m-d H:i:s"), $timezone, "d M y");

            $check_status = [REQUEST_CREATED , REQUEST_WAITING];

            $requests_query = Requests::where('job_type' , REQUEST_LATER)
                        ->where('request_type' , '!=' , REQUEST_TYPE_DIRECT)
                        ->where('request_date' , '>=' , $request_date)
                        ->whereIn('category_id' , $category_ids)
                        ->whereIn('sub_category_id' , $sub_category_ids)
                        ->whereIn('status' , $check_status)
                        ->orderBy('updated_at' , 'desc');

            if(is_array($bidded_requests)) {
                $requests_query = $requests_query->whereNotIn('id' , $bidded_requests);
            }

            if($ongoing_confirmed_dates) {
                $requests_query = $requests_query->whereNotIn('request_date' , $ongoing_confirmed_dates);
            }

            $is_web = '';

            if ($request->has('is_web')) {

                if($request->is_web) {

                    $is_web = 1;
                }
            }

            if ($is_web) {

                if ($request->s_k) {

                    $requests_query->where('name', 'like', '%' . $request->s_k . '%');
                } 

                if($request->sorting_key) {

                    if ($request->sorting_key == 'asc' || $request->sorting_key == 'desc') {

                        $requests_query->orderBy('name', $request->sorting_key);

                    } else if ($request->sorting_key == REQUEST_NOW || $request->sorting_key == REQUEST_LATER) {

                        $requests_query->where('job_type', $request->sorting_key);

                    } else {

                        $requests_query->orderBy('updated_at' , 'desc');
                    }
                }

                if(!$request->searching_key && !$request->sorting_key) {

                    $requests_query->orderBy('updated_at' , 'desc');
                }

                $requests = $requests_query->skip($request->skip)->take(10)->get();

            } else {

                $requests = $requests_query->get();
            }

            $result = [];

            if($requests) {

                $timezone = get_provider_timezone($request->id);

                // base_response($requests , $group = RESPONSE_SINGLE / RESPONSE_MULTIPLE , $bid_response = "" , $timezone = "")

                $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE ,"", $timezone , $request->id));
            }

            $response_array = ['success' => true , 'data' => $result];
        
        } else {

            $response_array = ['success' => false , 'error' => Helper::error_message(136) , 'error_code' => 136 ];
        }

        return response()->json($response_array, 200);    
    }

    /**
     * @uses to get the confirmed jobs
     * 
     * @param integer - request_id
     *
     * @return json response
     */

    public function confirmed_requests(Request $request) {

        $request_date = date('Y-m-d H:i:s');

        $check_status = [REQUEST_ONGOING];

        $provider_status = [PROVIDER_ACCEPTED];

        $requests_query = Requests::where('provider_id' , $request->id)
                    // where('job_type' , REQUEST_LATER)
                    // ->where('request_date' , '>=' , $request_date)
                    ->whereIn('status' , $check_status)
                    ->whereIn('provider_status' , $provider_status)
                    ->orderBy('updated_at' , 'desc');
        $is_web = '';

        if ($request->has('is_web')) {

            if($request->is_web) {

                $is_web = 1;

            }
        }

        if ($is_web) {

             if ($request->s_k) {

                $requests_query->where('name', 'like', '%' . $request->s_k . '%');

            } 

            if($request->sorting_key) {

                if ($request->sorting_key == 'asc' || $request->sorting_key == 'desc') {

                    $requests_query->orderBy('name', $request->sorting_key);

                } else {

                    $requests_query->orderBy('updated_at' , 'desc');

                }

            }

            if(!$request->searching_key && !$request->sorting_key) {

                $requests_query->orderBy('updated_at' , 'desc');

            }

            $requests = $requests_query->skip($request->skip)->take(10)->get();

        } else {

            $requests = $requests_query->get();

        }

        $result = [];

        if($requests){

            $timezone = get_provider_timezone($request->id);

            // base_response($requests , $group = RESPONSE_SINGLE / RESPONSE_MULTIPLE , $bid_response = "" , $timezone = "")

            $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE ,"", $timezone));
        }

        $response_array = ['success' => true , 'data' => $result];

        return response()->json($response_array, 200);

    }

    /**
     * @uses to get the ongoing jobs
     * 
     * @param integer - request_id
     *
     * @return json response
     */

    public function ongoing_requests(Request $request) {

        $request_date = date('Y-m-d H:i:s');

        $provider_id = $request->id;

        $check_status = [REQUEST_ONGOING , REQUEST_COMPLETE_PENDING , WAITING_PAYMENT_CONFIRM,REQUEST_RATING];

        $provider_status = [PROVIDER_NONE, PROVIDER_ASSIGNED , PROVIDER_ACCEPTED , PROVIDER_RATED];

        $requests_query = Requests::where('provider_id' , $request->id)
                        // where('job_type' , REQUEST_LATER)
                        // ->whereNotIn('requests.provider_status', array(PROVIDER_RATED))
                        ->whereIn('status' , $check_status)
                        ->whereNotIn('provider_status' , $provider_status)
                    
                        ->orderBy('updated_at' , 'desc');

        $is_web = '';

        if ($request->has('is_web')) {

            if($request->is_web) {

                $is_web = 1;

            }
        }



        if ($is_web) {

             if ($request->s_k) {

                $requests_query->where('name', 'like', '%' . $request->s_k . '%');

            } 


            if($request->sorting_key) {

                if ($request->sorting_key == 'asc' || $request->sorting_key == 'desc') {

                    $requests_query->orderBy('name', $request->sorting_key);

                } else {

                    $requests_query->orderBy('updated_at' , 'desc');

                }

            }

            if(!$request->searching_key && !$request->sorting_key) {

                $requests_query->orderBy('updated_at' , 'desc');

            }

            $requests = $requests_query
            ->orWhere(function($q) use ($provider_id) {
                             $q->where('requests.provider_id', $provider_id)
                                ->where('provider_status', PROVIDER_SERVICE_COMPLETED)                                  
                               ->where('requests.status', REQUEST_COMPLETED);
                         })
            ->skip($request->skip)->take(10)->get();

        } else {

            $requests = $requests_query
                        ->orWhere(function($q) use ($provider_id) {
                             $q->where('requests.provider_id', $provider_id)
                                ->where('provider_status', PROVIDER_SERVICE_COMPLETED)                                  
                               ->where('requests.status', REQUEST_COMPLETED);
                         })->get();

        }

        $result = [];

        if($requests){

            $timezone = get_provider_timezone($request->id);

            // base_response($requests , $group = RESPONSE_SINGLE / RESPONSE_MULTIPLE , $bid_response = "" , $timezone = "")

            $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE ,"", $timezone));
        }

        $response_array = ['success' => true , 'data' => $result];

        return response()->json($response_array, 200);

    }

    /**
     * @uses to get the bidded jobs
     * 
     * @param integer - request_id
     *
     * @return json response
     */

    public function bidded_requests(Request $request) {

        $request_date = date('Y-m-d H:i:s');

        $check_status = [REQUEST_META_BID,REQUEST_META_ASSIGNED];

        $requests_meta = RequestsMeta::where('provider_id' , $request->id)->whereIn('status' , $check_status)->select('request_id')->get()->toArray();

        $result = [];


        if($requests_meta) {

            $check_status = array(REQUEST_COMPLETED,REQUEST_CANCELLED,REQUEST_NO_PROVIDER_AVAILABLE,REQUEST_TIME_EXCEED_CANCELLED);

            $requests_query = Requests::whereIn('id', $requests_meta)
                    ->whereNotIn('status' , $check_status)
                    ->orderBy('updated_at' , 'desc');

            $is_web = '';

            if ($request->has('is_web')) {

                if($request->is_web) {

                    $is_web = 1;

                }
            }

            if ($is_web) {


                if ($request->s_k) {

                    $requests_query->where('name', 'like', '%' . $request->s_k . '%');

                } 

                if($request->sorting_key) {

                    if ($request->sorting_key == 'asc' || $request->sorting_key == 'desc') {

                        $requests_query->orderBy('name', $request->sorting_key);

                    } else {

                        $requests_query->orderBy('updated_at' , 'desc');

                    }

                }

                if(!$request->searching_key && !$request->sorting_key) {

                    $requests_query->orderBy('updated_at' , 'desc');

                }

                $requests = $requests_query->skip($request->skip)->take(10)->get();

            } else {

                $requests = $requests_query->get();

            }


            if($requests){
                
                $timezone = get_provider_timezone($request->id);

                // base_response($requests , $group = RESPONSE_SINGLE / RESPONSE_MULTIPLE , $bid_response = "" , $timezone = "")

                $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE ,$request->id ,$timezone));
            }
            
        }

        $response_array = ['success' => true , 'data' => $result];

        return response()->json($response_array , 200);

    }

    /**
     * @uses to get the Direct jobs
     * 
     * @param integer - request_id
     *
     * @return json response
     */

    public function direct_requests(Request $request) {

        $requests_query = Requests::where('provider_id' , $request->id)
                        ->where('status' , REQUEST_CREATED)
                        ->where('job_type' , REQUEST_LATER)
                        ->where('request_type' , REQUEST_TYPE_DIRECT)
                        ->orderBy('updated_at' , 'desc');
        $is_web = '';

        if ($request->has('is_web')) {

            if($request->is_web) {

                $is_web = 1;

            }
        }

        if ($is_web) {

             if ($request->s_k) {

                $requests_query->where('name', 'like', '%' . $request->s_k . '%');

            } 

            if($request->sorting_key) {

                if ($request->sorting_key == 'asc' || $request->sorting_key == 'desc') {

                    $requests_query->orderBy('name', $request->sorting_key);

                } else {

                    $requests_query->orderBy('updated_at' , 'desc');

                }

            }

            if(!$request->searching_key && !$request->sorting_key) {

                $requests_query->orderBy('updated_at' , 'desc');

            }

            $requests = $requests_query->skip($request->skip)->take(10)->get();

        } else {

            $requests = $requests_query->get();

        }

        $result = [];

        if($requests) {

            $timezone = get_provider_timezone($request->id);

            // base_response($requests , $group = RESPONSE_SINGLE / RESPONSE_MULTIPLE , $bid_response = "" , $timezone = "")

            $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE , "" , $timezone));

        }

        $response_array = ['success' => true , 'data' => $result];

        return response()->json($response_array , 200);

    }

    /**
     * @uses to get the completed jobs
     * 
     * @param integer - request_id
     *
     * @return json response
     */

    public function history(Request $request) {

        try {

            $base_query = Requests::where('provider_id' , $request->id);

            $provider_details = Provider::where('id' , $request->id)->first();
            
            $check_status = [REQUEST_RATING, REQUEST_COMPLETED];

            if(!$provider_details) {

                throw new Exception(Helper::error_message(134), 134);          
            }
            
            if($request->cancelled_jobs == YES) {

                $base_query = $base_query->whereIn('cancelled_by' ,[USER,PROVIDER]);
           
            } else {
                                
                if($provider_details->device_type != DEVICE_WEB) {

                    $cancelled_status = [REQUEST_CANCELLED, REQUEST_NO_PROVIDER_AVAILABLE, REQUEST_TIME_EXCEED_CANCELLED];
                    
                    $check_status = array_merge($cancelled_status,$check_status);
                }

                $base_query = $base_query->whereIn('status', $check_status);
            }

            $requests = $base_query->orderBy('created_at' , 'desc')->get();

            $result = [];

            if($requests) {

                $timezone = get_provider_timezone($request->id);

                $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE ,"", $timezone));
            }

            $response_array = ['success' => true , 'data' => $result];

            return response()->json($response_array, 200);

        } catch (Exception $e) {
            
            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success'=> false, 'error' => $error_message , 'error_code' => $error_code];

            return response()->json($response_array);
        }

    }

    /**
     * @uses to get the particular request details
     * 
     * @param 
     *
     * @return json response
     *
     */

    public function single_request(Request $request) {

        if(!RequestRepo::requests_exists($request->all(), $errors)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $timezone = get_provider_timezone($request->id);

            $response_array = RequestRepo::view($request ,PROVIDER,$timezone);
        }

        return response()->json($response_array, 200);
    
    }

    /**
     * @uses to get the particular request details
     * 
     * @param 
     *
     * @return json response
     */

    public function requests_chat_history(Request $request) {

        $messages = ChatMessage::where('provider_id', $request->id)
                ->where('request_id', $request->request_id);

        if($request->user_id) {

            ChatMessage::where('user_id', $request->user_id)
                ->where('request_id', $request->request_id)
                ->where('provider_id' , $request->id)
                ->where('type' , 'pu')
                ->update(['delivered' => 1]);
        }

        $response_array = Helper::null_safe(array(
            'success' => true,
            'data' => $messages->get()->toArray(),
        ));
    
        return response()->json($response_array, 200);
    }

    /**
     * @uses to update payment mail ID
     * 
     * @param 
     *
     * @return json response
     */

    public function payment_update(Request $request) {

        $validator = Validator::make($request->all() , [
            'paypal_email' => 'email',
            ]);

        if($validator->fails()) {

            $error = implode(',', $validator->messages()->all());

            $response_array = ['success' => false , 'error' => $error , 'error_code' => 101];

        } else {

            $provider = Provider::find($request->id);
            $provider->paypal_email = $request->paypal_email;
            $provider->save();
            
            $response_array = ['success' => true,'data' => $provider , 'message' => Helper::success_message(223) , 'code' => 223];

        }

        return response()->json($response_array, 200);
    
    }

    /**
     * User profile
     *
     *
     */

    public function user_profile(Request $request) {

        $validator = Validator::make($request->all() , [
            'user_id' => 'required|exists:users,id',
            ]);

        if($validator->fails()) {

            $error = implode(',', $validator->messages()->all());

            $response_array = ['success' => false , 'error' => $error , 'error_code' => 101];

        } else {

            $user = User::userResponse($request->user_id)->find($request->user_id);

            if($user) {

                $user = Helper::null_safe($user->toArray());
            }

            $response_array = ['success' => true , 'data' => $user];

        }

        return response()->json($response_array, 200);
    
    }

    /**
     * @uses to get the ongoing later requests details
     * 
     * @param 
     * 
     * @return json response
     */

    public function request_details(Request $request) {

        $validator = Validator::make($request->all() , [
                'request_id' => 'required|exists:requests,id'
            ]);

        if ($validator->fails()) {

            $errors = implode(',',$validator->messages()->all());

            $response_array = ['success' => false,'error' => $errors,'error_code' => 101];

        } else {
        
            $request_details = Requests::find($request->request_id);

            $requests_data = $invoice = [];

            if($request_details) {

                $timezone = get_provider_timezone($request->id);

            $response_array = RequestRepo::view($request ,PROVIDER,$timezone);

                $requests_data = RequestRepo::single_request_response($request_details , "" , $timezone);                   
            }

            $requests_data = Helper::null_safe($requests_data);
            
            $response_array = ['success' => true,'data' => $requests_data];
        }
    
        return response()->json($response_array, 200);
    } 


    public function change_status_notifications(Request $request) {

        $model =ProviderNotification::select(
                    'provider_notifications.id as id', 
                    'provider_notifications.notification as notification', 
                    'users.picture as picture',
                    'users.name as name',
                    'provider_notifications.user_id as user_id',
                    'provider_notifications.request_id as request_id',
                    'requests.unique_id as unique_id',
                    'provider_notifications.status as notify_status',
                    'provider_notifications.created_at as created_at')
                ->where('provider_notifications.provider_id', $request->id)
                ->leftJoin('users', 'users.id', '=', 'provider_notifications.user_id')
                ->leftJoin('requests', 'requests.id', '=', 'provider_notifications.request_id')
                ->where('provider_notifications.status', DEFAULT_FALSE)
                ->orderBy('provider_notifications.created_at', 'desc')
                ->get();

        $notifications = [];

        foreach ($model as $key => $value) {

            $value->status = DEFAULT_TRUE;
            $value->save();

            $notifications[] = ["notify_id"=>$value->id,
                                "notification"=>$value->notification,
                                "picture"=>$value->picture ? $value->picture : asset('placeholder.png'),
                                "name"=>$value->name,
                                "user_id"=>$value->user_id,
                                "request_id"=>$value->request_id,
                                "unique_id"=>$value->unique_id,
                                "created_at"=>$value->created_at->diffForHumans(),
                                'url'=>route('provider.requests.view' , ['request_unique_id' => $value->unique_id]),
                                "status"=>$value->notify_status]; 

        }

        return response()->json(['cnt'=>count($notifications), 'notifications'=>$notifications]);
    }

    public function change_job_status_notifications(Request $request) {

        $model = ProviderNotification::where('provider_id', $request->id)->where('job_status',DEFAULT_TRUE)->where('status', DEFAULT_FALSE)->get();

        foreach ($model as $key => $value) {

            $value->status = DEFAULT_TRUE;
            $value->save();

        }
    }

    public function change_direct_notifications(Request $request) {

        $model = ProviderNotification::where('provider_id', $request->id)->where('job_status',2)->where('status', DEFAULT_FALSE)->get();

        foreach ($model as $key => $value) {

            $value->status = DEFAULT_TRUE;
           
            $value->save();
        }
    }

    /** 
     * @method send_redeem_request()
     *
     * @uses Provider Send Redeem request to Admin
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param 
     * 
     * @return JSON response
     *
     */
    public function send_redeem_request(Request $request) {

        if(Setting::get('redeem_control') == REDEEM_OPTION_DISABLED) {

            $response_array = ['success' => false , 'error' => Helper::error_message(201) , 'error_code' => 201];

            return response($response_array, 200);
        }

        //  Get admin configured - Minimum Provider Credit

        $minimum_redeem = Setting::get('minimum_redeem' , 0);

        // Get Provider Remaining Credits 

        $provider_payment_details = ProviderPayment::where('provider_id' , $request->id)->first();

        if($provider_payment_details) {

            $current_wallet_amount = $provider_payment_details->wallet;

            // check the provider have more than minimum credits

            if($current_wallet_amount > $minimum_redeem) {

                $redeem_amount = abs(intval($current_wallet_amount - $minimum_redeem));

                // Check the redeems is not empty

                if($redeem_amount) {

                    // Save Redeem Request

                    $redeem_request = new RedeemRequest;

                    $redeem_request->provider_id = $request->id;

                    $redeem_request->request_amount = $redeem_amount;

                    $redeem_request->status = false;

                    $redeem_request->save();

                    // Update Redeems details 

                    $provider_payment_details->wallet = abs($provider_payment_details->wallet-$redeem_amount);

                    $provider_payment_details->save();

                    /** @todo Send mail notification to admin */ 

                    // if($admin_details = get_admin_mail()) {

                    //     $subject = tr('provider_redeeem_send_title');

                    //     $page = "emails.redeems.redeem-request-send";

                    //     $email = $admin_details->email;
                        
                    //     Helper::send_email($page,$subject,$email,$admin_details);
                    // }

                    $response_array = ['success' => true];

                } else {

                    $response_array = ['success' => false , 'error' => Helper::error_message(203) , 'error_code' => 203];
                }

            } else {
                
                $response_array = ['success' => false , 'error' => Helper::error_message(202) ,'error_code' => 202];
            }

        } else {

            $response_array = ['success' => false , 'error' => Helper::error_message(205) , 'error_code' => 205];
        }
        
        return response()->json($response_array , 200);
    }

    /** 
     * @method redeem_request_cancel()
     *
     * @uses cancel redeem request which created by provider
     *
     * @created Vidhya R
     *
     * @edited Vidhya R
     *
     * @param integer redeem_request_id
     * 
     * @return JSON response
     *
     */

    public function redeem_request_cancel(Request $request) {

        $validator = Validator::make($request->all() , [
            'redeem_request_id' => 'required|exists:redeem_requests,id,provider_id,'.$request->id,
            ] ,
            [
                'exists' => "The selected redeem request is not belongs to the login provider."
            ]
            );

         if ($validator->fails()) {

            $error_messages = implode(',', $validator->messages()->all());

            $response_array = array('success' => false, 'error' => $error_messages, 'error_code' => 101);

        } else {

            if($provider_payment_details = ProviderPayment::where('provider_id' , $request->id)->first()) {

                if($redeem_request_details = RedeemRequest::find($request->redeem_request_id)) {

                    // Check status to cancel the redeem request

                    if(in_array($redeem_request_details->status, [REDEEM_REQUEST_SENT , REDEEM_REQUEST_PROCESSING])) {
                        // Update the redeeem 

                        $provider_payment_details->wallet = $provider_payment_details->wallet + abs($redeem_request_details->request_amount);

                        $provider_payment_details->save();

                        // Update the redeeem request Status

                        $redeem_request_details->status = REDEEM_REQUEST_CANCEL;

                        $redeem_request_details->save();

                        $response_array = ['success' => true];


                    } else {
                        $response_array = ['success' => false ,  'error' => Helper::error_message(204) , 'error_code' => 204];
                    }

                } else {
                    $response_array = ['success' => false ,  'error' => Helper::error_message(205) , 'error_code' => 205];
                }

            } else {

                $response_array = ['success' => false ,  'error' => Helper::error_message(205) , 'error_code' =>205 ];
            }

        }

        return response()->json($response_array , 200);

    }

    /** 
     * @method provider_notifications()
     *
     * @uses list of provider notifications
     *
     * @created Shobana C
     *
     * @edited 
     *
     * @param 
     * 
     * @return JSON response
     *
     */

    public function provider_notifications(Request $request) {
                    
        $new_value = ProviderNotification::where('provider_id', $request->id)->where('status', DEFAULT_FALSE)->get();

        $new_cnt = count($new_value);

        $new_response = $this->available_requests($request)->getData();

        $new_job_notify = 0;

        if($new_response->success) {

            $new_job_notify = count($new_response->data) - (\Session::get('total_request_cnt') ? \Session::get('total_request_cnt') : 0);
        }

        if ($new_job_notify < 0) {

            $new_job_notify = 0;
            
        }

        $direct_job_notify = ProviderNotification::where('provider_id', $request->id)
                    ->where('status', DEFAULT_FALSE)
                    ->where('job_status', 2)->count();

        $bidded_job_notify = ProviderNotification::where('provider_id', $request->id)
                    ->where('status', DEFAULT_FALSE)
                    ->where('job_status', 3)->count();

        return response()->json(['cnt'=>$new_cnt, 'new_job_notify'=>$new_job_notify,
                'direct_job_notify'=>$direct_job_notify, 'bidded_job_notify'=>$bidded_job_notify]);
    
    }

    /** 
     * @method provider_notifications()
     *
     * @uses redeems details
     *
     * @created Vidhya R
     *
     * @edited 
     *
     * @param 
     * 
     * @return JSON response
     *
     */

    public function redeems(Request $request) {

        if(Setting::get('redeem_control') == REDEEM_OPTION_ENABLED) {

            $redeems = ProviderPayment::where('provider_id' , $request->id)->select('total' , 'paid_amount as paid' , 'wallet as remaining' , 'status')->first();

            if(count($redeems) == 0) {

                $data = ['total' => 0.00 , 'paid' => 0.00 , 'remaining' => 0.00 , 'currency' => Setting::get('currency') , 'redeem_requests' => []];

            } else {

                $data = [];

                $data = $redeems->toArray();

                $currency = Setting::get('currency');

                $data['currency'] = $currency;

                $data['redeem_requests'] = [];

                $model = RedeemRequest::where('provider_id' , $request->id)
                        ->select('request_amount' , 
                             \DB::raw("'$currency' as currency"),
                             \DB::raw('DATE_FORMAT(created_at , "%e %b %y") as requested_date'),
                             'paid_amount',
                             \DB::raw('DATE_FORMAT(updated_at , "%e %b %y") as paid_date'),
                             'status',
                             'id as redeem_request_id'
                         )
                        ->orderBy('created_at', 'desc')
                        ->get();



                foreach ($model as $key => $value) {

                    $redeem_status = redeem_request_status($value->status);

                    $redeem_request_cancel_allow = in_array($value->status, [REDEEM_REQUEST_SENT , REDEEM_REQUEST_PROCESSING]) ? REDEEM_REQUEST_CANCEL_ALLOWED : REDEEM_REQUEST_CANCEL_NOT_ALLOWED;

                    $redeem_request_data = [];
                    
                    $redeem_request_data = [
                        'redeem_request_id' => $value->redeem_request_id,
                        'request_amount' => $value->request_amount,
                        'redeem_status' => $redeem_status,
                        'currency' => $value->currency,
                        'requested_date' => $value->requested_date,
                        'paid_amount' => $value->paid_amount,
                        'paid_date' => $value->paid_date,
                        'redeem_request_cancel_allow' => $redeem_request_cancel_allow,
                        'status' => $value->status
                        ];

                    array_push($data['redeem_requests'], $redeem_request_data);

                }

            }

            $response_array = ['success' => true , 'data' => $data];

        } else {
            $response_array = ['success' => false , 'error' => Helper::error_message(147) , 'error_code' => 147];
        }

        return response()->json($response_array , 200);
    
    }

    /**
     * @method logout()
     *
     * @uses to logout function of the provider
     *
     * @created Vidhya R 
     *
     * @edited Vidhya R 
     *
     * @param integer id
     *
     * @param integer token
     *
     * @return JSON Reponse 
     */

    public function logout(Request $request) {

        $provider_details = Provider::find($request->id);

        if(count($provider_details) == 0) {

            $response_array = ['success' => false , 'error' => Helper::error_message(137) , 'error_code' => 137];

            return response()->json($response_array , 200);

        }

        $provider_details->device_token = "";

        $provider_details->save();

        $response_array = ['success' => true , 'message' => tr('logout_success') ];

        return response()->json($response_array , 200);

    }

    /**
     * @method subscriptions() 
     *
     * @uses 
     *
     * @created Vithya R 
     *
     * @updated Vithya R
     *
     * @param
     *
     * @return json repsonse
     */     

    public function subscriptions(Request $request) {

        try {

            $base_query = ProviderSubscription::where('provider_subscriptions.status', APPROVED)->CommonResponse();

            #get category based subscriptions
            if($request->category_id) {
                
                $category_details = Category::find($request->category_id);

                if( !$category_details) {

                    throw new Exception(tr('category_not_found'), 101);
                }

                $base_query = $base_query->where('category_id',$request->category_id);
            }
            
            $skip = $request->skip ?: 0;

            $take = $request->take ?: ( Setting::get('admin_take_count') ?: 12);

            $provider_subscriptions = $base_query->skip($skip)->take($take)->orderBy('updated_at', 'desc')->get();

            $response_array = ['success' => true, 'data' => $provider_subscriptions];

            return response()->json($response_array,200);

        } catch(Exception $e) {

            $error = $e->getMessage(); $code = $e->getCode();

            $response_array = ['success' => false, 'error' => $error, 'error_code' => $code];

            return response()->json($response_array);

        }

    }

    /**
     * @method subscriptions_payment_by_stripe() 
     *
     * @uses 
     *
     * @created Vithya R
     *
     * @updated Vithya R
     *
     * @param
     *
     * @return json repsonse
     */     

    public function subscriptions_payment_by_stripe(Request $request) {

        try {

            $validator = Validator::make($request->all(), [
                'provider_subscription_id' => 'required|exists:provider_subscriptions,id',
            ],
            [
                'provider_subscription_id' => Helper::error_message(30001)
            ]
            );

            if ($validator->fails()) {

                // Error messages added in response for debugging

                $error = implode(',',$validator->messages()->all());

                throw new Exception($error, 101);

            }

            DB::beginTransaction();

            // Check Subscriptions

            $subscription_details = ProviderSubscription::where('id', $request->provider_subscription_id)->where('status', APPROVED)->first();

            if (!$subscription_details) {

                throw new Exception(Helper::error_message(30001), 30001);
            }

            // Check provider card details

            $provider_card_details = ProviderCard::where('provider_id', $request->id)->where('is_default',YES)->first();

            if (!$provider_card_details) {

                throw new Exception(Helper::error_message(143), 143);
            }

            $customer_id = $provider_card_details->customer_id;

            // Check stripe configuration
        
            $stripe_secret_key = Setting::get('stripe_secret_key');

            if(!$stripe_secret_key) {

                throw new Exception(Helper::error_message(902), 902);
            } 

            \Stripe\Stripe::setApiKey($stripe_secret_key);

            $total = $subscription_details->amount;

            $currency_code = Setting::get('currency_code') ?: "USD";

            $stripe_payment_response =  \Stripe\Charge::create(array(
                "amount" => $total * 100,
                "currency" => $currency_code,
                "customer" => $customer_id,              
            ));

            $payment_id = $stripe_payment_response->id;

            $amount = $stripe_payment_response->amount/100;

            $paid_status = $stripe_payment_response->paid;
            
            /** category subscriptions **/
            $category_subscriptions = ProviderSubscription::where('category_id',$subscription_details->category_id)->pluck('id');

            /** get category based previous provider subscription payment **/
            $previous_payment_base_query = ProviderSubscriptionPayment::where('provider_id' , $request->id)->where('status', PAID)->where('is_current_subscription', YES);            

            if($category_subscriptions) {
                
                $previous_payment_base_query = $previous_payment_base_query->whereIn('provider_subscription_id',$category_subscriptions);
            }
            
            $previous_payment = $previous_payment_base_query->first();

            $provider_subscription_payment = new ProviderSubscriptionPayment;

            $provider_subscription_payment->expiry_date = date('Y-m-d H:i:s',strtotime("+{$subscription_details->plan} months"));

            if ($previous_payment) { 

                if (strtotime($previous_payment->expiry_date) >= strtotime(date('Y-m-d H:i:s'))) {

                    $provider_subscription_payment->expiry_date = date('Y-m-d H:i:s', strtotime("+{$subscription_details->plan} months", strtotime($previous_payment->expiry_date)));

                    $previous_payment->is_current_subscription = NO;
                }
            }

            $provider_subscription_payment->provider_id = $request->id;

            $provider_subscription_payment->provider_subscription_id = $request->provider_subscription_id;

            $provider_subscription_payment->payment_id = $payment_id;

            $provider_subscription_payment->subscription_amount = $total ?: 0.00;

            $provider_subscription_payment->subscribed_by = PROVIDER;

            $provider_subscription_payment->amount = $amount ?: 0.00;

            $provider_subscription_payment->status = PAID;

            $provider_subscription_payment->payment_mode = CARD;

            $provider_subscription_payment->is_current_subscription = YES;

            if ($provider_subscription_payment->save()) {
                
                if($previous_payment) {

                    $previous_payment->save(); 
                }

                $provider_details = Provider::find($request->id);

                // $provider_details->subscription_expired = NO;

                // $provider_details->expiry_date = $provider_subscription_payment->expiry_date;

                $provider_details->provider_type = PAID_PROVIDER;

                if ($provider_details->save()) {

                } else {

                    Log::info("Provider Saving...");

                    throw new Exception(Helper::error_message(30002));
                }

            } else {

                throw new Exception(Helper::error_message(30002));
                
            }

            DB::commit();

            $response_array = ['success' => true , 'message' => Helper::success_message(30301), 'code' => 30301];

            return response()->json($response_array, 200);

        // } catch(Stripe_CardError | Stripe_InvalidRequestError | Stripe_AuthenticationError | Stripe_ApiConnectionError | Stripe_Error $e) {
        } catch(Stripe_Error $e) {

            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            if($provider_subscription_payment) {

                $provider_subscription_payment->status = UNPAID;

                $provider_subscription_payment->cancelled_reason = $error_message;

                $provider_subscription_payment->is_cancelled = YES;

                $provider_subscription_payment->save();

            }

            DB::commit();

            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => 30002];

            return response()->json($response_array);

        } catch(Exception $e) {

            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            // Something else happened, completely unrelated to Stripe

            DB::rollback();

            if(isset($provider_subscription_payment)) {

                $provider_subscription_payment->status = UNPAID;

                $provider_subscription_payment->cancelled_reason = $error_message;
                
                $provider_subscription_payment->is_cancelled = YES;

                $provider_subscription_payment->save();

            }

            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

            return response()->json($response_array);

        }

    }

    /**
     * @method subscriptions_history() 
     *
     * @uses 
     *
     * @created Vithya R 
     *
     * @updated Vithya R
     *
     * @param
     *
     * @return json repsonse
     */     

    public function subscriptions_history(Request $request) {

        Log::info(url()->current());

        Log::info("Request Data".print_r($request->all(), true));

        try {

            $base_query = ProviderSubscriptionPayment::where('provider_id', $request->id)->select('provider_subscription_payments.id as provider_subscription_payment_id', 'provider_subscription_payments.*');

            $skip = $request->skip ?: 0;

            $take = $request->take ?: ( Setting::get('admin_take_count') ?: 12);

            $provider_subscription_payments = $base_query->skip($skip)->take($take)->orderBy('updated_at', 'desc')->get();

            foreach ($provider_subscription_payments as $key => $payment_details) {

                $payment_details->currency = Setting::get('currency', 1);

                $payment_details->title = $payment_details->description = "";

                $subscription_details = ProviderSubscription::find($payment_details->provider_subscription_id);

                if($subscription_details) {

                    $payment_details->title = $subscription_details->title ?: "";

                    $payment_details->description = $subscription_details->description ?: "";

                    $subscription_category_details = Category::find($subscription_details->category_id);

                    $payment_details->category_name = $subscription_category_details ? $subscription_category_details->name: "";
                }

                unset($payment_details->id);
            }

            $response_array = ['success' => true, 'data' => $provider_subscription_payments];

            return response()->json($response_array,200);

        } catch(Exception $e) {

            $error = $e->getMessage(); $code = $e->getCode();

            $response_array = ['success' => false, 'error' => $error, 'error_code' => $code];

            return response()->json($response_array);

        }

    }

    /**
     * @method notifications_status_update()
     *
     * @uses To enable/disable notifications of email / push notification
     *
     * @created Vithya R
     *
     * @updated Vithya R
     *
     * @param 
     *
     * @return response of details
     */
    public function notifications_status_update(Request $request) {

        try {

            $validator = Validator::make(
                $request->all(),
                array(
                    'status' => 'required|numeric',
                    'notification_type'=>'required'
                )
            );

            if($validator->fails()) {

                $error = implode(',', $validator->messages()->all());

                throw new Exception($error, 101);
            }

            DB::beginTransaction();

            $provider_details = Provider::find($request->id);

            if ($request->notification_type == NOTIFICATION_TYPE_EMAIL) {

                $provider_details->is_email_notification = $request->status;

            }

            if ($request->notification_type == NOTIFICATION_TYPE_SMS) {

                $provider_details->is_sms_notification = $request->status;

            }

            if ($request->notification_type == NOTIFICATION_TYPE_PUSH) {

                $provider_details->is_push_notification = $request->status;

            }

            $provider_details->save();

            DB::commit();

            if($request->status ) {

                $message = tr('notification_enable');

            } else {

                $message = tr('notification_disable');

            }

            $data = ['id' => $provider_details->id , 'token' => $provider_details->token];

            $response_array = [
                'success' => true ,'message' => $message, 
                'is_email_notification' => (int) $provider_details->is_email_notification,  // Don't remove int (used ios)
                'is_sms_notification' => (int) $provider_details->is_sms_notification,  // Don't remove int (used ios)
                'is_push_notification' => (int) $provider_details->is_push_notification,    // Don't remove int (used ios)
                'data' => $data
            ];
                
            return response()->json($response_array , 200);

        } catch (Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            $code = $e->getCode();

            $response_array = ['success'=>false, 'error'=>$error, 'error_code'=>$code];

            return response()->json($response_array);
        }

    }

    /**
     * @method cards_list()
     *
     * @uses get the provider payment mode and cards list
     *
     * @created Vidhya R
     *
     * @updated vithya R
     *
     * @param integer id
     * 
     * @return
     */

    public function cards_list(Request $request) {

        try {

            $provider_cards = ProviderCard::where('provider_id' , $request->id)->select('id as provider_card_id' , 'customer_id' , 'last_four' ,'card_name', 'card_token' , 'is_default' )->get();

            $card_payment_mode = $payment_modes = [];

            $card_payment_mode['name'] = "Card";

            $card_payment_mode['is_default'] = 1;

            array_push($payment_modes , $card_payment_mode);

            $data['payment_modes'] = $payment_modes;   

            $data['cards'] = $provider_cards ? $provider_cards : []; 

            return $this->sendResponse($message = "", $code = 200, $data);

        } catch(Exception $e) {

            return $this->sendResponse($error = $e->getMessage(), $error_code = 101);

        }
    
    }
    
    /**
     * @method cards_add()
     *
     * @uses Update the selected payment mode 
     *
     * @created Vidhya R
     *
     * @updated vithya R
     *
     * @param Form data
     * 
     * @return JSON Response
     */

    public function cards_add(Request $request) {

        try {

            DB::beginTransaction();

            if(Setting::get('stripe_secret_key')) {

                \Stripe\Stripe::setApiKey(Setting::get('stripe_secret_key'));

            } else {

                throw new Exception(Helper::error_message(164), 164);
            }
        
            $validator = Validator::make(
                    $request->all(),
                    [
                        'card_token' => 'required',
                    ]
                );

            if ($validator->fails()) {

                $error = implode(',',$validator->messages()->all());
             
                throw new Exception($error , 101);

            } else {

                Log::info("INSIDE CARDS ADD");

                $provider_details = Provider::find($request->id);

                if(!$provider_details) {

                    throw new Exception(Helper::error_message(504), 504);
                    
                }

                // Get the key from settings table
                
                $customer = \Stripe\Customer::create([
                        "card" => $request->card_token,
                        "email" => $provider_details->email,
                        "description" => "Customer for ".Setting::get('site_name'),
                    ]);

                if($customer) {

                    $customer_id = $customer->id;

                    $card_details = new ProviderCard;

                    $card_details->provider_id = $request->id;

                    $card_details->customer_id = $customer_id;

                    $card_details->card_token = $customer->sources->data ? $customer->sources->data[0]->id : "";

                    $card_details->card_name = $customer->sources->data ? $customer->sources->data[0]->brand : "";

                    $card_details->last_four = $customer->sources->data[0]->last4 ? $customer->sources->data[0]->last4 : "";

                    $card_details->month = $request->month ? $request->month : "00";

                    $card_details->year = $request->year ? $request->year : "2018";

                    // Check is any default is available

                    $check_card_details = ProviderCard::where('provider_id',$request->id)->count();

                    $card_details->is_default = $check_card_details ? 0 : 1;

                    if($card_details->save()) {

                        if($provider_details) {

                            $provider_details->provider_card_id = $check_card_details ? $provider_details->provider_card_id : $card_details->id;

                            $provider_details->save();
                        }

                        $data = ProviderCard::where('id' , $card_details->id)->select('id as provider_card_id' , 'customer_id' , 'last_four' ,'card_name', 'card_token' , 'is_default' )->first();

                        $response_array = ['success' => true , 'message' => Helper::success_message(233), 'data' => $data];

                    } else {

                        throw new Exception(Helper::error_message(165), 165);
                        
                    }
               
                } else {

                    throw new Exception(Helper::error_message(165) , 165);
                    
                }
            
            }

            DB::commit();

            return response()->json($response_array , 200);

        } catch(Stripe_CardError $e) {

            Log::info("error1");

            $error1 = $e->getMessage();

            $response_array = ['success' => false , 'error' => $error1 ,'error_code' => 165];

            return response()->json($response_array , 200);

        } catch (Stripe_InvalidRequestError $e) {

            // Invalid parameters were supplied to Stripe's API

            Log::info("error2");

            $error2 = $e->getMessage();

            $response_array = ['success' => false , 'error' => $error2 ,'error_code' => 165];

            return response()->json($response_array , 200);

        } catch (Stripe_AuthenticationError $e) {

            Log::info("error3");

            // Authentication with Stripe's API failed
            $error3 = $e->getMessage();

            $response_array = ['success' => false , 'error' => $error3 ,'error_code' => 165];

            return response()->json($response_array , 200);

        } catch (Stripe_ApiConnectionError $e) {
            Log::info("error4");

            // Network communication with Stripe failed
            $error4 = $e->getMessage();

            $response_array = ['success' => false , 'error' => $error4 ,'error_code' => 165];

            return response()->json($response_array , 200);

        } catch (Stripe_Error $e) {
            Log::info("error5");

            // Display a very generic error to the provider, and maybe send
            // yourself an email
            $error5 = $e->getMessage();

            $response_array = ['success' => false , 'error' => $error5 ,'error_code' => 165];

            return response()->json($response_array , 200);

        } catch (\Stripe\StripeInvalidRequestError $e) {

            Log::info("error7");

            // Log::info(print_r($e,true));

            $response_array = ['success' => false , 'error' => Helper::error_message(165) ,'error_code' => 165];

            return response()->json($response_array , 200);

        } catch(Exception $e) {

            DB::rollback();

            $error_message = $e->getMessage();

            Log::info("catch FUNCTION INSIDE".$error_message);

            return $this->sendError($e->getMessage(), $e->getCode());
        }
   
    }

    /**
     * @method cards_delete()
     *
     * @uses delete the selected card
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer provider_card_id
     * 
     * @return JSON Response
     */

    public function cards_delete(Request $request) {

        Log::info("cards_delete");

        // @todo vidhya fix this exceptiion for website
        
        $provider_details = $this->loginProvider ?: Provider::providerResponse($request->id)->first();

        DB::beginTransaction();

        try {
    
            $provider_card_id = $request->provider_card_id;

            $validator = Validator::make($request->all(),
                [
                    'provider_card_id' => 'required|integer|exists:provider_cards,id,provider_id,'.$request->id,
                ],
                [
                    'exists' => 'The :attribute doesn\'t belong to provider:'.$provider_details->name,
                ]
            );

            if ($validator->fails()) {

               $error = implode(',', $validator->messages()->all());

                throw new Exception($error, 101);

            } else {

                $provider_details = Provider::find($request->id);

                if(!$provider_details) {

                    throw new Exception(Helper::error_message(504), 504);
                    
                }

                ProviderCard::where('id',$provider_card_id)->delete();

                if($provider_details->payment_mode = CARD) {

                    // Check he added any other card

                    if($check_card = ProviderCard::where('provider_id' , $request->id)->first()) {

                        $check_card->is_default =  DEFAULT_TRUE;

                        $provider_details->provider_card_id = $check_card->id;

                        $check_card->save();

                    } else { 

                        $provider_details->payment_mode = COD;

                        $provider_details->provider_card_id = DEFAULT_FALSE;
                    
                    }
               
                }

                // Check the deleting card and default card are same

                if($provider_details->provider_card_id == $provider_card_id) {

                    $provider_details->provider_card_id = DEFAULT_FALSE;

                    $provider_details->save();
                }
                
                $provider_details->save();
            
                $response_array = ['success' => true , 'message' => Helper::success_message(224) , 'code' => 224];

            }

            DB::commit();
    
            return response()->json($response_array , 200);

        } catch(Exception $e) {

            DB::rollback();

            return $this->sendError($e->getMessage(), $e->getCode());
        
        }
    }

    /**
     * @method cards_default()
     *
     * @uses update the selected card as default
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id
     * 
     * @return JSON Response
     */
    public function cards_default(Request $request) {

        // Log::info("cards_default"); // @todo vidhya fix this exceptiion for website

        $provider_details = $this->loginProvider ?: Provider::providerResponse($request->id)->first();

        try {

            DB::beginTransaction();

            $validator = Validator::make($request->all(),
                [
                    'provider_card_id' => 'required|integer|exists:provider_cards,id,provider_id,'.$request->id,
                ],
                [
                    'exists' => 'The :attribute doesn\'t belong to provider:'.$provider_details->name,
                ]
            );

            if($validator->fails()) {

                $error = implode(',', $validator->messages()->all());

                throw new Exception($error, 101);
                   
            }

            $provider_details = Provider::find($request->id);

            if(!$provider_details) {

                throw new Exception(Helper::error_message(504), 504);
                
            }

            $old_default_cards = ProviderCard::where('provider_id' , $request->id)->where('is_default', DEFAULT_TRUE)->update(['is_default' => DEFAULT_FALSE]);

            $card = ProviderCard::where('id' , $request->provider_card_id)->update(['is_default' => DEFAULT_TRUE]);

            $provider_details->provider_card_id = $request->provider_card_id;

            $provider_details->save();

            $response_array = ['success' => true, 'message'=>Helper::success_message(225), 'code'=>225];
               
            DB::commit();

            return response()->json($response_array , 200);

        } catch(Exception $e) {

            DB::rollback();

            return $this->sendError($e->getMessage(), $e->getCode());
        
        }
    
    }

    /** 
     * @method revenues
     *
     * @uses provider revenue dashboard
     *
     * @created Vidhya R
     * 
     * @updated Vidhya R
     *
     * @param -
     * 
     * @return JSON Response
     */
    
    public function revenues(Request $request) {
        
        try {

            $data = new \stdClass;

            $data->currency = $currency = Setting::get('currency', '$');

            // total_earnings

            $provider_payment_details = ProviderPayment::where('provider_id', $request->id)->first();

            if(!$provider_payment_details) {

                $provider_payment_details = new ProviderPayment;

                $provider_payment_details->provider_id = $request->id;

                $provider_payment_details->status = 1;

                $provider_payment_details->save();
            }

            $total_earnings = $provider_payment_details->provider_amount ?: 0.00;
            
            $data->total_earnings = $total_earnings ?: 0.00; 

            $data->total_earnings_formatted = $currency." ".$total_earnings ?: 0.00; 

            // Today Earnings
            
            $today_earnings = RequestPayment::where('provider_id', $request->id)->whereDate('paid_date', '=',date('Y-m-d'))->where('is_paid' , YES)->sum('provider_amount');

            $data->today_earnings = $today_earnings ?: 0.00; 

            $data->today_earnings_formatted = $currency." ".$data->today_earnings ?: 0.00; 

            // TRIPS DETAILS

            // total_upcoming_jobs

            $total_upcoming_jobs = Requests::where('provider_id' , $request->id)->
                    where('job_type' , REQUEST_LATER)->whereIn('status' , [REQUEST_ONGOING])->whereIn('provider_status' , [PROVIDER_ACCEPTED])->count();

            $data->total_upcoming_jobs = $total_upcoming_jobs ?: 0;

            // total_completed_jobs

            $total_completed_jobs = Requests::where('provider_id' , $request->id)->whereIn('status' , [REQUEST_RATING , REQUEST_COMPLETED])
                        ->where('provider_status' , PROVIDER_RATED)
                        ->count();

            $data->total_completed_jobs = $total_completed_jobs ?: 0;

            // total_cancelled_jobs

            $total_cancelled_jobs = Requests::where('provider_id' , $request->id)->whereIn('status' , [REQUEST_CANCELLED , REQUEST_TIME_EXCEED_CANCELLED])
                        ->count();

            $data->total_cancelled_jobs = $total_cancelled_jobs ?: 0;

            // Last 10 days revenues

            $last_x_days_revenues = [];

            $start  = new \DateTime('-10 days', new \DateTimeZone('UTC'));

            $period = new \DatePeriod($start, new \DateInterval('P1D'), 10);

            $dates = $last_x_days_revenues = [];

            foreach ($period as $date) {

                $current_date = $date->format('Y-m-d');

                $last_x_days_data = new \stdClass;

                $last_x_days_data->date = $current_date;

                $last_x_days_total_earnings = RequestPayment::where('provider_id', $request->id)->whereDate('paid_date', '=', $current_date)->where('is_paid' , YES)->sum('provider_amount');

                $last_x_days_data->total_earnings = $last_x_days_total_earnings ?: 0.00;

                $last_x_days_data->total_earnings_formatted = $currency." ".$last_x_days_data->total_earnings ?: 0.00;

                array_push($last_x_days_revenues, $last_x_days_data);
            }

            $data->last_x_days_revenues = $last_x_days_revenues;

            // Service based revenues

            $provider_services = ProviderService::where('provider_id', $request->id)
                                ->where('status',APPROVED)->get();

            $service_based_revenues = [];

            foreach ($provider_services as $key => $provider_service_details) {

                if($sub_category = SubCategory::find($provider_service_details->sub_category_id)) {

                    $service_data = new \stdClass;

                    $service_data->sub_category_id = $provider_service_details->sub_category_id;

                    $service_data->sub_category_name = $sub_category->name;

                    $service_data->sub_category_picture = $sub_category->picture;

                    $service_total_earnings = Requests::where('requests.provider_id', $request->id)
                                                ->where('sub_category_id', $provider_service_details->sub_category_id)
                                                ->leftJoin('request_payments', 'request_payments.request_id', '=', 'requests.id')
                                                ->sum('request_payments.provider_amount');

                    $service_total_earnings = $service_total_earnings ?: 0.00;

                    $service_data->total_earnings = $service_total_earnings;

                    $service_data->total_earnings_formatted = $currency." ".$service_total_earnings;

                    array_push($service_based_revenues, $service_data);
                }
                
            }

            $data->service_based_revenues = $service_based_revenues;

            return $this->sendResponse($message = "", $code = 200, $data);

        } catch(Exception $e) {

            return $this->sendResponse($error = $e->getMessage(), $error_code = 101);

        }
    
    }

    
}
