<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Helpers\Helper;

use App\Repositories\CommonRepository as CommonRepo;

use App\Repositories\UserRepository as UserRepo;

use App\Repositories\RequestRepository as RequestRepo;

use App\Jobs\sendPushNotification;

use App\User;

use App\Card;

use App\Settings;

use App\NotificationTemplate;

use App\Provider;

use App\TempRequests;

use App\Requests;

use App\RequestsMeta;

use App\RequestPayment;

use App\UserRating;

use App\ProviderRating;

use App\ChatMessage;

use App\AbuseProvider;

use App\ProviderService;

use App\EmailTemplate;

use App\ProviderNotification;

use App\UserNotification;

use App\SmsTemplate;

use App\Category;

use App\SubCategory;

use Validator;

use Setting;

use DB;

use Log;

use Hash;

use Exception;

use App\UserFavProvider;

use App\UserCoupon;

use App\Coupon;

use App\ProviderGallery;

use App\UserReferrer;

use App\Referral;

class UserApiController extends Controller {

    protected $loginUser;

    public function __construct(Request $request) {

        // Log::info(url()->current());

        // Log::info("Request Data".print_r($request->all(), true));

        $this->middleware('UserApiVal' , array('except' => ['test' ,'register' , 'login' , 'forgot_password', 'referrals_check']));

        if($request->id) {
            
            $this->loginUser = User::find($request->id);

            $this->timezone = $this->loginUser->timezone ?? "America/New_York";

        }

    }

    /**
     * @method register()
     *
     * @uses store the register user details or social login
     *
     * @created Vidhya R 
     *
     * @updated Vidhya R
     *
     * @param - 
     *
     * @return JSON Response
     */

    public function register(Request $request) {

        $allow_register_operation = true;

        // Check the device type , device token and login by

        if(!CommonRepo::basic_validation($request->all(), $errors)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $login_by = $request->login_by;

            $allowed_social_logins = ['facebook','google'];

            if(in_array($login_by,$allowed_social_logins)) {

                // Check the social unique id and form fields

                if(!CommonRepo::social_validation($request->all(), $errors)) {
                    
                    $allow_register_operation = false;

                    $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];
                }

            } else {

                $check_mail_action = true;

                if($request->has('email')) {

                     $check_email_exist = User::where('email' , $request->email)->withTrashed()->count();

                    if($check_email_exist) {

                        $allow_register_operation = $check_mail_action = false;

                        $response_array = ['success' => false , 'error' => Helper::error_message(102) , 'error_code' => 102];
                    } 
                }

                // Validate manual registration fields

                if($check_mail_action) {

                    if(!CommonRepo::manual_validation($request->all(), $errors)) {

                        $allow_register_operation = false;

                        $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];
                    }
                }
           
            }

            if($allow_register_operation) {

                $check_user = User::where('email' , $request->email)->first();

                if($check_user) {

                    // Generate new tokens

                    $check_user->token = Helper::generate_token();

                    $check_user->token_expiry = Helper::generate_token_expiry();

                    // Save device details
                    
                    $check_user->device_token = $request->device_token;

                    $check_user->device_type = $request->device_type;

                    $check_user->login_by = $request->login_by;

                    $check_user->timezone = $request->has('timezone') ? $request->timezone : "";

                    $check_user->save();

                    $user_details = UserRepo::update($request , $check_user->id);

                } else {

                    $user_details = UserRepo::store($request);
                }

                $user_details = Helper::null_safe($user_details);

                if($user_details['is_verified'] == USER_EMAIL_NOT_VERIFIED && !in_array($login_by,$allowed_social_logins)) {

                    $response_array = array('success' => false , 'error' => Helper::error_message(503) , 'error_code' => 503);

                    return response()->json($response_array, 200);
                    
                } else {

                    if( $user_details['status'] == USER_DECLINED ) {
                    
                        $response = array('success' => false , 'error' => Helper::error_message(502) , 'error_code' => 502);
                        
                        return response()->json($response, 200);
                    }

                    $user_details['is_appstore_upload'] = Setting::get('is_appstore_upload') ?: 0;

                    $response_array = ['success' => true , 'data' => $user_details , 'message' => Helper::success_message(204) , 'code' => 204];
               
                }

                \Log::info('Registration completed');

            }
        }

        return response()->json($response_array, 200);
    
    }

    /**
     * @method login()
     *
     * @uses check manual login and allow the user to access
     *
     * @created Vidhya R 
     *
     * @updated Vidhya R
     *
     * @param string email, string password
     *
     * @return JSON Response
     */

    public function login(Request $request) {

        try {

            DB::beginTransaction();

            $response_array = array();
        
            // Check the device type , device token and login by

            if(!CommonRepo::basic_validation($request->all(), $errors)) {

                throw new Exception($errors, 101);
                
            } else {

                if(!CommonRepo::login_validation($request->all(), $errors)) {

                    throw new Exception($errors, 101);

                } else {

                    $user_details = User::where('email', '=', $request->email)->first();

                    if(!$user_details) {
                                 
                        throw new Exception(Helper::error_message(504), 504);

                    }

                    if($user_details->is_verified == USER_EMAIL_NOT_VERIFIED) {
                            
                        throw new Exception(Helper::error_message(503), 503);

                    }

                    if($user_details->status == USER_DECLINED) {
                        
                        throw new Exception(Helper::error_message(502), 502);

                    }

                    // Validate the user credentials

                    if(\Hash::check($request->password, $user_details->password)) {

                        // Generate new tokens
                        $user_details->token = Helper::generate_token();

                        $user_details->token_expiry = Helper::generate_token_expiry();

                        // Save device details
                        $user_details->device_token = $request->device_token;

                        $user_details->device_type = $request->device_type;

                        $user_details->login_by = $request->login_by;

                        $user_details->timezone = $request->has('timezone') ? $request->timezone : "";

                        if($user_details->save()) {

                            DB::commit();

                            $data = User::userResponse($user_details->id)->first();

                            if($data) {

                                $data = $data->toArray();

                                $data['is_appstore_upload'] = Setting::get('is_appstore_upload') ?: 0;
                            }

                            $response_array = ['success' => true , 'data' => $data];

                            return response()->json($response_array , 200);

                        } else {

                            throw new Exception(Helper::error_message(405), 405);
                            

                        }

                    } else {

                        throw new Exception(Helper::error_message(105), 105);

                    }

                }
            }

        } catch(Exception $e) {

            DB::rollback();

            $error = $e->getMessage();

            $code = $e->getCode();

            $response_array = ['success'=>false, 'error'=> $error, 'error_code'=>$code];

            return response()->json($response_array);

        }    
    }

    /**
     * @method forgot_password()
     *
     * @uses based on the entered email address, validate and send email
     *
     * @created Vidhya R 
     *
     * @updated Vidhya R
     *
     * @param string email
     *
     * @return JSON Response
     */

    public function forgot_password(Request $request) {

        if(!CommonRepo::email_validation($request->all(), $errors)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {
            $response_array = UserRepo::forgot_password($request);
        }

        return response()->json($response_array, 200);
    
    }

    /**
     * @method change_password()
     *
     * @uses based on the entered email address, validate and send email
     *
     * @created Vidhya R 
     *
     * @updated Vidhya R
     *
     * @param string email
     *
     * @return JSON Response
     */

    public function change_password(Request $request) {

        if(!CommonRepo::change_password_validation($request->all(), $errors)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $user = User::find($request->id);

            if(\Hash::check($request->old_password,$user->password)) {

                $user->password = \Hash::make($request->password);

                $user->save();

                $response_array = Helper::null_safe(array('success' => true , 'message' => tr('password_change_success')));
            } else {
                $response_array = array('success' => false , 'error' => tr('password_not_match') , 'error_code' => 131);
            }
        }

        return response()->json($response_array,200);

    }

    public function profile(Request $request) {

        $user = User::userResponse($request->id)->find($request->id);

        if($user) {
            $user = Helper::null_safe($user->toArray());
        }

        $response_array = ['success' => true , 'data' => $user];

        return response()->json($response_array, 200);

    }

    /**
     * @method update_profile()
     *
     * @uses to update user details
     * 
     * @param Request Object 
     * 
     * @return JSON Response
     */

    public function update_profile(Request $request) {

        $user_id = $request->id;
        
        $validator = Validator::make(
            $request->all(),
            array(
                'name' => 'max:255',
                'email' => 'email|unique:users,email,'.$user_id.'|max:255',
                'picture' => 'mimes:jpeg,bmp,png',
                'cover' => 'mimes:jpeg,bmp,png',
                'gender' => 'in:male,female,others',
                'device_token' => '',
                'timezone' => '',
                'mobile' => 'numeric|digits_between:4,16' 
            ));

        if ($validator->fails()) {
            // Error messages added in response for debugging
            $errors = implode(',',$validator->messages()->all());

            $response_array = ['success' => false,'error' => $errors,'error_code' => 101];

        } else {
            
            $response_array = UserRepo::update($request,$user_id);

            if($response_array) {
                $user = Helper::null_safe($response_array);
            }

            $response_array = ['success' => true , 'data' => $user];
        }

        return response()->json($response_array, 200);
   
    }

    /**
     * @method delete_account()
     *
     * @uses to delete user details
     * 
     * If the provider from social login , no need password
     *
     * @param Request Object 
     * 
     * @return JSON Response
     */

    public function delete_account(Request $request) {

        $validator = Validator::make(
            $request->all(),
            array(
                'password' => '',
            ));

        if ($validator->fails()) {

            $errors = implode(',',$validator->messages()->all());

            $response_array = ['success' => false,'error' => $errors,'error_code' => 101];
        } else {

            $user = User::find($request->id);

            $allow = 0;

            if($user->login_by != 'manual') {
                $allow = 1;
            } else {

                if(Hash::check($request->password, $user->password)) {
                    $allow = 1;
                } else {
                    $allow = 0 ;
                    $response_array = array('success' => false , 'error' => Helper::error_message(107) ,'error_code' => 107);
                }

            }

            if($allow) {

                $user = User::where('id',$request->id)->first();

                if($user) {

                    // Check any ongoing requests and any unpaid requests 

                    $is_delete_account_allow = Helper::check_user_delete_account($request->id);

                    if($is_delete_account_allow) {

                        // Delete the become provider details as well

                        $provider = $user->provider_id ? User::find($user->provider_id) : "";

                        if($provider) {
                            $provider->forceDelete();
                        }

                        $user->forceDelete();

                        $response_array = array('success' => true , 'message' => tr('user_account_delete_success'));

                    } else {

                        $response_array = ['success' => false , 'error' => Helper::error_message(508) , 'error_code' => 508];

                    }

                } else {
                    $response_array = array('success' =>false , 'error' => Helper::error_message(146), 'error_code' => 146);
                }

            } else {
                $response_array = array('success' => false , 'error' => Helper::error_message(107) ,'error_code' => 107);
            }
        }

		return response()->json($response_array,200);
	
    }

    /**
     * @method settings()
     * 
     * @uses To enable/disable the push notification in mobile
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param object $request - push notification status
     *
     * @return boolean 
     */

    public function settings(Request $request) {

        DB::beginTransaction();

        try {

            $user_details = User::find($request->id);

            $user_details->push_notification_status = $user_details->push_notification_status ? 0 : 1;

            $user_details->save();

            DB::commit();

            $message = tr('push_notification_disable');

            if($user_details->push_notification_status) {

                $message = tr('push_notification_enable');

            }

            $response_array = array('success' => true, 'message' => $message , 'push_notification_status' => $user_details->push_notification_status);

            return response()->json($response_array, 200);

        } catch(Exception $e) {

            DB::rollback();

            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

            return response()->json($response_array);
        
        }
    }

    public function get_categories(Request $request) {

        $categories = getCategories();

        if($categories) {

            $response_array = array('success' => true , 'categories' => $categories->toArray());

        } else {
            $response_array = array('success' => false,'error' => Helper::error_message(135),'error_code' => 135);
        }

        $response = response()->json($response_array, 200);
        return $response;
    }

    /**
     * @method request_status_check_now()
     * 
     * @uses to get the ongoing On Demand request details
     * 
     * @param 
     * 
     * @return json response
     */

    public function request_status_check_now(Request $request) {

        $timezone = get_user_timezone($request->id);

        $check_status = [REQUEST_NONE,REQUEST_CANCELLED,REQUEST_NO_PROVIDER_AVAILABLE,REQUEST_TIME_EXCEED_CANCELLED , REQUEST_COMPLETED];
        
        $requests = Requests::where('user_id', '=', $request->id)
                            ->whereNotIn('status' , $check_status)
                            ->where('job_type' , REQUEST_NOW)
                            ->orderBy('requests.job_type' , 'ASC')
                            ->get();

        $requests_data = $invoice = [];

        if($requests) {

            foreach ($requests as  $request_details) {

                $requests_data[] = RequestRepo::single_request_response($request_details ,  "" , $timezone);

                $allowed_status = [WAITING_PAYMENT_CONFIRM,REQUEST_COMPLETE_PENDING,REQUEST_COMPLETED,REQUEST_RATING];

                if(in_array($request_details->status, $allowed_status)) {

                    $invoice_query = RequestPayment::where('request_id' , $request_details->id)
                                    ->leftJoin('requests' , 'request_payments.request_id' , '=' , 'requests.id')
                                    ->leftJoin('users' , 'requests.user_id' , '=' , 'users.id')
                                    ->leftJoin('cards' , 'users.card_id' , '=' , 'cards.id');

                    if($request_details->user->payment_mode == CARD) {
                        $invoice_query = $invoice_query->where('cards.is_default' , DEFAULT_TRUE) ;  
                    }

                    $invoice = $invoice_query->select('requests.provider_id' , 'request_payments.total_time',
                    'requests.price_per_hour',
                                        'request_payments.payment_mode as payment_mode' , 
                                        'request_payments.base_price',
                                        'request_payments.time_price' , 'request_payments.tax_price' , 
                                        'request_payments.total',
                                        'cards.card_token','cards.customer_id','cards.last_four')
                                    ->get()
                                    ->toArray();
                                    
                    $invoice = Helper::null_safe($invoice);
                }
            }
        }

        $requests_data = Helper::null_safe($requests_data);
        
        $response_array = ['success' => true,'data' => $requests_data,'invoice' => $invoice];
    
        return response()->json($response_array, 200);
    
    } 

    /**
     * @method request_status_check_now()
     * 
     * @uses to get the ongoing later requests details
     * 
     * @param 
     * 
     * @return json response
     */

    public function request_status_check_later(Request $request) {

        $timezone = get_user_timezone($request->id);

        $check_status = [REQUEST_NONE,REQUEST_CANCELLED,REQUEST_NO_PROVIDER_AVAILABLE,REQUEST_TIME_EXCEED_CANCELLED , REQUEST_COMPLETED];

        $allowed_provider_status = [PROVIDER_NONE,PROVIDER_ASSIGNED,PROVIDER_ACCEPTED];

        $requests = Requests::where('user_id', '=', $request->id)
                            ->whereNotIn('status' , $check_status)
                            ->whereNotIn('provider_status' , $allowed_provider_status)
                            ->where('job_type' , REQUEST_LATER)
                            ->orderBy('requests.job_type' , 'ASC')
                            ->get();

        $requests_data = $invoice = [];

        if($requests) {

            foreach ($requests as  $req_details) {

                $requests_data[] = RequestRepo::single_request_response($req_details, "" , $timezone);

                $allowed_status = [WAITING_PAYMENT_CONFIRM,REQUEST_COMPLETE_PENDING,REQUEST_COMPLETED,REQUEST_RATING];

                if(in_array($req_details->status, $allowed_status)) {

                    $invoice_query = RequestPayment::where('request_id' , $req_details->id)
                                    ->leftJoin('requests' , 'request_payments.request_id' , '=' , 'requests.id')
                                    ->leftJoin('users' , 'requests.user_id' , '=' , 'users.id')
                                    ->leftJoin('cards' , 'users.card_id' , '=' , 'cards.id');

                    if($req_details->user->payment_mode == CARD) {
                        $invoice_query = $invoice_query->where('cards.is_default' , DEFAULT_TRUE) ;  
                    }

                    $invoice = $invoice_query->select('requests.provider_id' , 'request_payments.total_time','requests.price_per_hour',
                                        'request_payments.payment_mode as payment_mode' , 
                                        'request_payments.base_price',
                                        'request_payments.time_price' , 'request_payments.tax_price' , 
                                        'request_payments.total',
                                        'cards.card_token','cards.customer_id','cards.last_four')
                                    ->get()
                                    ->toArray();
                }
            }
        }
        
        $response_array = Helper::null_safe(array(
            'success' => true,
            'data' => $requests_data,
            'invoice' => $invoice
        ));
    
        $response = response()->json($response_array, 200);
        return $response;
    
    } 

    /**
     * @method request_later_details()
     * 
     * @uses to get the ongoing later requests details
     * 
     * @param 
     * 
     * @return json response
     */

    public function request_later_details(Request $request) {

        $timezone = get_user_timezone($request->id);

        $validator = Validator::make($request->all() , [
                'request_id' => 'required|exists:requests,id,user_id,'.$request->id
            ]);

        if ($validator->fails()) {

            $errors = implode(',',$validator->messages()->all());

            $response_array = ['success' => false,'error' => $errors,'error_code' => 101];

        } else {

            $check_status = [REQUEST_NONE,REQUEST_CANCELLED,REQUEST_NO_PROVIDER_AVAILABLE,REQUEST_TIME_EXCEED_CANCELLED , REQUEST_COMPLETED];
            
            $request_details = Requests::where('id' ,$request->request_id)->whereNotIn('status' , $check_status)->first();

            $requests_data = $invoice = [];

            if($request_details) {

                $requests_data[] = RequestRepo::single_request_response($request_details , "" , $timezone);

                $allowed_status = [WAITING_PAYMENT_CONFIRM,REQUEST_COMPLETE_PENDING,REQUEST_COMPLETED,REQUEST_RATING];

                if(in_array($request_details->status, $allowed_status)) {

                    $invoice_query = RequestPayment::where('request_id' , $request_details->id)
                                    ->leftJoin('requests' , 'request_payments.request_id' , '=' , 'requests.id')
                                    ->leftJoin('users' , 'requests.user_id' , '=' , 'users.id')
                                    ->leftJoin('cards' , 'users.card_id' , '=' , 'cards.id');

                    if($request_details->user->payment_mode == CARD) {
                        $invoice_query = $invoice_query->where('cards.is_default' , DEFAULT_TRUE) ;  
                    }

                    $invoice = $invoice_query->select('requests.provider_id' , 'request_payments.total_time','requests.price_per_hour',
                                        'request_payments.payment_mode as payment_mode' , 
                                        'request_payments.base_price',
                                        'request_payments.time_price' , 'request_payments.tax_price' , 
                                        'request_payments.total',
                                        'cards.card_token','cards.customer_id','cards.last_four')
                                    ->get()
                                    ->toArray();
                                    
                    $invoice = Helper::null_safe($invoice);
                }

            }

            $requests_data = Helper::null_safe($requests_data);
            
            $response_array = ['success' => true,'data' => $requests_data,'invoice' => $invoice];

        }
    
        return response()->json($response_array, 200);
    
    } 


    /**
     * @method request_details()
     * 
     * @uses to get the ongoing later requests details
     * 
     * @param  
     * 
     * @return json response
     */

    public function request_details(Request $request) {

        $timezone = get_user_timezone($request->id);

        $validator = Validator::make($request->all() , [
                'request_id' => 'required|exists:requests,id,user_id,'.$request->id
            ]);

        if ($validator->fails()) {

            $errors = implode(',',$validator->messages()->all());

            $response_array = ['success' => false,'error' => $errors,'error_code' => 101];

        } else {
        
            $request_details = Requests::find($request->request_id);

            $requests_data = $invoice = [];

            if($request_details) {

                $requests_data = RequestRepo::single_request_response($request_details , "" , $timezone);
                   
            }

            $requests_data = Helper::null_safe($requests_data);
            
            $response_array = ['success' => true,'data' => $requests_data];

        }
    
        return response()->json($response_array, 200);
    
    } 

    /**
     * @method waiting_request_cancel()
     *
     * @uses In request now - Before any provider accept the user can cancel the request
     * 
     * @param 
     *
     * @return json response
     */

    public function waiting_request_cancel(Request $request) {

        Log::info('WAITING Request cancel ==== Started');

        $requests = Requests::where('user_id' , $request->id)->where('job_type' , REQUEST_NOW)->where('status' , REQUEST_CREATED)->get();

        if(count($requests) > 0) {

            foreach ($requests as $key => $request_details) {
                
                $request_details->status = REQUEST_CANCELLED;
                $request_details->cancelled_date = date('Y-m-d H:i:s');
                $request_details->cancelled_by = USER;
                $request_details->save();

                $request_details_meta = RequestsMeta::where('request_id' , $request_details->id);

                if($current = $request_details_meta->where('status' , REQUEST_META_OFFERED)->first()) {

                    $current_provider = $current->provider_id;
                    
                    if($provider = Provider::find($current_provider)) {

                        $provider->waiting_to_respond = WAITING_TO_RESPOND_NORMAL;
                        $provider->save();

                        // Send Notification to the currently offered provider

                        $title = Helper::push_message(8002);
                        $message = Helper::push_message(8003);

                        $user_details = $request_details->user  ? $request_details->user : '';

                        $content = EmailTemplate::getRowValue(USER_CANCEL_REQUEST, $provider, $user_details, $request_details);

                        $notification = NotificationTemplate::getRowValue(USER_CANCEL_REQUEST, $provider, $user_details, $request_details);

                        $content = $notification ? $notification : USER_CANCEL_REQUEST;

                        if (Setting::get('provider_notification_control')) {
                            ProviderNotification::save_notification($provider, $content, $request->id, $request_details->id, USER_CANCEL_REQUEST);

                        }

                        if (Setting::get('sms_notification')) {

                            SmsTemplate::getRowValue(USER_CANCEL_REQUEST, $provider);
                                
                        }

                        
                       $this->dispatch(new sendPushNotification($provider->id,PROVIDER,PUSH_PROVIDER_HOME,$title,$message ));
                    }
               
                }

                $delete_request_meta = RequestsMeta::where('request_id' , $request_details->id)->delete();
            
            }
    
            $response_array = array('success' => true , 'message' => Helper::success_message(207) , 'code' => 207);

        } else {
            $response_array = ['success' => false, 'error' => Helper::error_message(158)  , 'error_code' => 158];
        }
       
        Log::info('WAITING Request cancel ==== completed');

        return response()->json($response_array , 200);

    }

    /**
     * @method request_update()
     *
     * @uses In request now - Before any provider accept the user can cancel the request
     * 
     * @param 
     *
     * @return json response
     */

    public function request_update(Request $request) {

         $validator = Validator::make($request->all() , [
                'request_id' => 'required|exists:requests,id',
            ]);

        if($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false , 'error' => Helper::error_message(101) ,'error_code' => 101];

        } else {
            $response_array = RequestRepo::update($request);
        }

        return response()->json($response_array,200);

    }

    /**
     * @method assign_request()
     *
     * @uses In request later - Assign provider to that request
     * 
     * @created vidhya R
     *
     * @updated vidhya R
     *
     * @param integer request_id
     *
     * @param integer request_meta_id
     *
     * @return json response
     */

    public function assign_request(Request $request) {

        $validator = Validator::make($request->all() , [
                'request_id' => 'required|exists:requests,id',
                'request_meta_id' => 'required|exists:requests_meta,id',
            ]);

        if($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false , 'error' => Helper::error_message(101) ,'error_code' => 101];

        } else {

            $request_details = Requests::find($request->request_id);

            // Check the job type is REQUEST_LATER

            if($request_details->job_type != REQUEST_LATER) {

                $response_array = ['success' => false , 'error' => Helper::error_message(148) , 'error_code' => 148];

                return response()->json($response_array , 200);
            }

            // Instead of find method, Because we need to check whether provider bidded or not

            $request_meta = RequestsMeta::where('id' , $request->request_meta_id)->where('status' , REQUEST_META_BID)->first();

            // Check whether request cancelled previously

            if($request_meta && $request_details->status != REQUEST_PROVIDER_ASSIGNED) {
            
                $allowed_status = [REQUEST_CREATED,REQUEST_WAITING];

                if(!in_array($request->status, $allowed_status)) {

                    // Change status of the requests_meta

                    if($request_details = Requests::find($request->request_id)) {

                        $request_details->status = REQUEST_PROVIDER_ASSIGNED;
                        $request_details->provider_status = PROVIDER_ASSIGNED;
                        
                        $request_details->price_per_hour = $request->price_per_hour ? $request->price_per_hour : $request_details->user_price;

                        $request_details->assign_time = date('Y-m-d H:i:s');
                        $request_details->save();

                        $request_meta = RequestsMeta::find($request->request_meta_id);
                        $request_meta->status = REQUEST_META_ASSIGNED;
                        $request_meta->save();

                        // Send notification to the provider

                        $title = Helper::push_message(8000); $message = Helper::push_message(8001);

                        $page_type = PUSH_PROVIDER_BIDDED_TAB;

                        $this->dispatch(new sendPushNotification($request_meta->provider_id,PROVIDER,PUSH_PROVIDER_SINGLE,$title,$message , "" , $request_details->id , $page_type));

                        // Disable remaining lists in request meta table

                        // Cancel Remaining providers bids

                        RequestsMeta::where('request_id' , $request->request_id)->where('provider_id' , '!=' , $request_meta->provider_id)->update(['status' => REQUEST_META_TIMEDOUT]);

                        $response_array = ['success'=> true , 'request_id' => $request_details->id , 'message' => Helper::success_message(214)];

                    } else {
                        $response_array = ['success' => false , 'error' => Helper::error_message(124) , 'error_code' => 124];
                    }

                } else {
                    $response_array = ['success' => false , 'error' => Helper::error_message(123) , 'error_code' => 123];
                }

            } else {
                $response_array = ['success' => false , 'error' => Helper::error_message(122) , 'error_code' => 122];
            }
        
        }

        return response()->json($response_array,200);

    }

    /**
     * @method reject_request()
     *
     * @uses In request later - Assign provider to that request
     * 
     * @param 
     *
     * @return json response
     */

    public function reject_request(Request $request) {

        $validator = Validator::make($request->all() , [
                'request_id' => 'required|exists:requests,id',
                'request_meta_id' => 'required|exists:requests_meta,id',
            ]);

        if($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false , 'error' => Helper::error_message(101) ,'error_code' => 101];

        } else {

            $request_details = Requests::find($request->request_id);

            // Check the job type is REQUEST_LATER

            if($request_details->job_type != REQUEST_LATER) {

                $response_array = ['success' => false , 'error' => Helper::error_message(148) , 'error_code' => 148];

                return response()->json($response_array , 200);
            }

            $request_meta = RequestsMeta::where('id' , $request->request_meta_id)->where('status' , REQUEST_META_BID)->first();

            // Check whether request cancelled previously

            if($request_meta && $request_details->status != REQUEST_META_TIMEDOUT) {
            
                $allowed_status = [REQUEST_CREATED,REQUEST_WAITING];

                if(!in_array($request->status, $allowed_status)) {

                    // Change status of the requests_meta

                    if($request_details = Requests::find($request->request_id)) {

                        $request_meta = RequestsMeta::find($request->request_meta_id);
                        $request_meta->status = REQUEST_META_TIMEDOUT;
                        $request_meta->save();

                        // Send notification to the provider

                        $title = Helper::push_message(8004);
                        $message = Helper::push_message(8005);

                        $this->dispatch(new sendPushNotification($request_meta->provider_id,PROVIDER,PUSH_PROVIDER_JOB_DASHBOARD,$title,$message));


                        $response_array = ['success'=> true , 'request_id' => $request_details->id , 'message' => Helper::success_message(215)];

                        $provider_detail = $request_meta->metaProvider ? $request_meta->metaProvider : '';

                        $user_details = $request_details->metaUser ? $request_details->metaUser : '';

                        if ($provider_detail) {

                            $email = EmailTemplate::getRowValue(USER_REJECT_BID, $provider_detail, $user_details, $request_details);

                            $notification = NotificationTemplate::getRowValue(USER_REJECT_BID, $provider_detail, $user_details, $request_details);

                            $content = $notification ? $notification : USER_REJECT_BID;

                            if (Setting::get('user_notification_control')) {

                                ProviderNotification::save_notification($provider_detail, $content, $request_details->user_id, $request_details->id, USER_REJECT_BID);

                            }

                            if (Setting::get('sms_notification')) {

                                SmsTemplate::getRowValue(USER_REJECT_BID, $provider_detail);
                                
                            }

                        }

                    } else {
                        $response_array = ['success' => false , 'error' => Helper::error_message(124) , 'error_code' => 124];
                    }

                } else {
                    $response_array = ['success' => false , 'error' => Helper::error_message(123) , 'error_code' => 123];
                }

            } else {
                $response_array = ['success' => false , 'error' => Helper::error_message(122) , 'error_code' => 122];
            }
        }

        return response()->json($response_array,200);

    }

    /**
     * @method cancel_request()
     *
     * @uses Cancel the requests
     * 
     * @param 
     *
     * @return json response
     */

    public function cancel_request(Request $request) {

        $user = User::find($request->id);
    
        if(!UserRepo::request_validation($request->all() , $errors , $user)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $request_details = Requests::find($request->request_id);

            $allowed_cancel_status = [PROVIDER_NONE,PROVIDER_ASSIGNED,PROVIDER_ACCEPTED,PROVIDER_STARTED];

            // Check whether request cancelled previously

            if($request_details->status != REQUEST_CANCELLED) {

                // Check whether request eligible for cancellation

                if(in_array($request_details->provider_status, $allowed_cancel_status)) {

                    // Update status of the request to cancellation

                    $request_details->status = REQUEST_CANCELLED;

                    $request_details->cancelled_date = date('Y-m-d H:i:s');

                    $request_details->cancelled_by = USER;

                    $request_details->cancelled_reason = $request->cancelled_reason;
                    
                    $request_details->save();

                    // If request has confirmed provider then release him to available status

                    if($request_details->provider_id != false){

                        $provider = Provider::find($request_details->provider_id);

                        if($request_details->job_type == REQUEST_NOW) {

                            $provider->is_available = PROVIDER_AVAILABLE;
                            
                            $provider->waiting_to_respond = WAITING_TO_RESPOND_NORMAL;
                        }

                        $provider->save();

                        // Send Push Notification to Provider

                        $title = Helper::push_message(8006); $message = Helper::push_message(8007);

                        $user_details = $request_details->user ? $request_details->user  : '';

                        $email = EmailTemplate::getRowValue(USER_CANCEL_REQUEST, $provider, $user_details, $request_details);

                        $notification = NotificationTemplate::getRowValue(USER_CANCEL_REQUEST, $provider, $user_details, $request_details);

                        $content = $notification ? $notification : USER_CANCEL_REQUEST;

                        if (Setting::get('provider_notification_control')) {

                            ProviderNotification::save_notification($provider, $email, $request_details->user_id,$request_details->id, USER_CANCEL_REQUEST);

                        }

                        if (Setting::get('sms_notification')) {

                            SmsTemplate::getRowValue(USER_CANCEL_REQUEST, $provider);
                                
                        }


                        $this->dispatch(new sendPushNotification($request_details->provider_id,PROVIDER,PUSH_PROVIDER_HOME,$title,$message));

                        // Send mail notification to provider
                    }

                    // No longer need request specific rows from RequestMeta
                    RequestsMeta::where('request_id', '=', $request_details->request_id)->delete();

                    $response_array = ['success' => true,'job_type' => $request_details->job_type,'request_id' => $request_details->id , 'message' => Helper::success_message(207) , 'code' => 207];

                } else {
                    $response_array = ['success' => false, 'error' => Helper::error_message(121), 'error_code' => 121];
                }

            } else {
                $response_array = ['success' => false, 'error' => Helper::error_message(120), 'error_code' => 113];
            }
        }

        return response()->json($response_array, 200);
    
    }

    /**
     * @method requests_payment
     *
     * @uses Payment section for the request
     * 
     * @created vidhya R
     * 
     * @updated vidhya R
     *
     * @param 
     *
     * @return JSON response
     */

    public function requests_payment(Request $request) {

        Log::info(print_r($request->all() , true));

        try {

            DB::beginTransaction();

            $validator = Validator::make($request->all(),
                array(
                        'request_id' => 'required|exists:requests,id,user_id,'.$request->id,
                        'payment_mode' => 'required|in:'.COD.','.PAYPAL.','.CARD.'|exists:settings,key,value,1',
                        'is_paid' => 'required',
                        'coupon_code'=>'exists:coupons,coupon_code',
                    ),
                array(
                        'exists' => Helper::error_message(141),
                    )
                );

            if($validator->fails()) {

                $errors = implode(',', $validator->messages()->all());

                throw new Exception($errors, 101);
            } 

            Log::info('User payment Started');

            $request_details = Requests::where('id',$request->request_id)->where('status' , REQUEST_COMPLETE_PENDING)->first();

            $user_details = User::find($request->id);

            //Check current status of the request

            if($request_details && intval($request_details->status) != REQUEST_RATING ) {

                $total = $actual_total = 0;

                $request_payment = RequestPayment::where('request_id' , $request->request_id)->first();

                if($request_payment) {

                    $request_payment->payment_mode = $request->payment_mode;

                    $request_payment->save();

                    $total = $actual_total = $request_payment->total;
                
                }

                // coupon code start

                $coupon_amount = 0; $coupon_reason = '';

                $is_coupon_applied = COUPON_NOT_APPLIED;
                  
                if($request->coupon_code) {

                    Log::info("coupon code inside");

                    $coupon_details = Coupon::where('coupon_code', $request->coupon_code)->first();

                    Log::info("coupon_details".print_r($coupon_details , true));

                    Log::info("COUPON_APPROVED".COUPON_APPROVED);

                    if (count($coupon_details) == 0 ) {

                        Log::info("coupon code - not found");

                        $coupon_reason = tr('coupon_delete_reason');

                    } else {
                        
                        if ($coupon_details->status != COUPON_APPROVED) {

                            Log::info("coupon code - coupon_inactive_reason");

                            $coupon_reason = tr('coupon_inactive_reason');

                        } else {

                            Log::info("coupon code - check coupon code user check");

                            $check_coupon = $this->coupon_code_user_use_check($user_details, $coupon_details)->getData();


                            if ($check_coupon->success) {

                                Log::info("coupon code - check success");

                                $is_coupon_applied = COUPON_APPLIED;

                                $amount_convertion = $coupon_details->amount;

                                if ($coupon_details->amount_type == PERCENTAGE) {

                                    $amount_convertion = amount_convertion($coupon_details->amount, $actual_total);

                                }

                                if ($amount_convertion < $actual_total) {

                                    $total = $actual_total - $amount_convertion;

                                    $coupon_amount = $amount_convertion;

                                } else {

                                    $total = 0;

                                    $coupon_amount = $amount_convertion;
                                    
                                }

                                // Create user applied coupon

                                if($check_coupon->code == 2002) {

                                    $user_coupon = UserCoupon::where('user_id', $user_details->id)
                                            ->where('coupon_code', $request->coupon_code)
                                            ->first();

                                    // If user coupon not exists, create a new row

                                    if ($user_coupon) {

                                        if ($user_coupon->no_of_times_used < $coupon_details->per_users_limit) {

                                            $user_coupon->no_of_times_used += 1;

                                            $user_coupon->save();

                                        }

                                    }

                                } else {

                                    $user_coupon = new UserCoupon;

                                    $user_coupon->user_id = $user_details->id;

                                    $user_coupon->coupon_code = $request->coupon_code;

                                    $user_coupon->no_of_times_used = 1;

                                    $user_coupon->save();

                                }

                            } else {

                                $coupon_reason = $check_coupon->error;
                                
                            }

                        }

                    }

                }

                Log::info("is_coupon_applied".$is_coupon_applied);

                Log::info("coupon_code".$request->coupon_code);

                // coupon code end

                if($total == 0) {

                    $request_payment->status = DEFAULT_TRUE;

                    $request_payment->payment_id = "COUPON-ZERO"."-".uniqid();

                    $request_details->is_paid = $request_payment->is_paid = DEFAULT_TRUE;

                    $request_payment->paid_date = date('Y-m-d H:i:s');

                    $request_details->status = REQUEST_RATING;

                    $request_details->total = $total;

                    Log::info("Total".$total);
        
                    // Update provider earnings 

                    Helper::provider_payment($request_details->provider_id, $actual_total , CARD);

                } else if($request->payment_mode == COD) {

                    $request_details->status = WAITING_PAYMENT_CONFIRM;
                    
                    $request_payment->payment_id = uniqid();
                    
                    $request_payment->status = DEFAULT_TRUE;

                    Log::info("Total".$total);

                    // No need to spilit the payment now. On provider confirmation will spilit the payment

                } else if($request->payment_mode == CARD || $request->payment_mode == PAYPAL) {

                    $stripe_secret_key = Setting::get('stripe_secret_key');

                    if($stripe_secret_key) {

                        \Stripe\Stripe::setApiKey($stripe_secret_key);

                    } else {

                        throw new Exception(Helper::error_message(164), 164);
                    }

                    $check_card_exists = User::where('users.id' , $request->id)
                                    ->leftJoin('cards' , 'users.id','=','cards.user_id')
                                    ->where('cards.id' , $user_details->card_id)
                                    ->where('cards.is_default' , DEFAULT_TRUE);

                    if($check_card_exists->count() == 0) {

                        throw new Exception(Helper::error_message(143), 143);
                    }

                    $user_card = $check_card_exists->first();

                    if($total != 0) {

                        $customer_id = $user_card->customer_id;

                        try {

                            $user_charge =  \Stripe\Charge::create(array(
                                "amount" => $total * 100,
                                "currency" => "usd",
                                "customer" => $customer_id,
                            ));

                           $payment_id = $user_charge->id;
                           
                           $amount = $user_charge->amount/100;
                           
                           $paid_status = $user_charge->paid;

                            if($paid_status) {

                                $request_payment->status = DEFAULT_TRUE;

                                $request_payment->payment_id = $payment_id;

                                $request_details->is_paid = $request_payment->is_paid = DEFAULT_TRUE;

                                $request_payment->paid_date = date('Y-m-d H:i:s');

                                $request_details->status = REQUEST_RATING;

                                $request_details->total = $total;

                                Log::info("Total".$total);

                                // Update provider earnings 

                                Helper::provider_payment($request_details->provider_id, $total , CARD , $request_payment->admin_amount , $request_payment->provider_amount);

                            } else {

                                throw new Exception(Helper::error_message(176), 176);

                            }

                        } catch(\Stripe\Error\RateLimit $e) {

                            $error_message = $e->getMessage();

                            $error_code = $e->getCode();

                            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

                            return response()->json($response_array);


                        } catch(\Stripe\Error\Card $e) {

                            $error_message = $e->getMessage();

                            $error_code = $e->getCode();

                            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

                            return response()->json($response_array);

                        } catch (\Stripe\Error\InvalidRequest $e) {
                            // Invalid parameters were supplied to Stripe's API
                           
                            $error_message = $e->getMessage();

                            $error_code = $e->getCode();

                            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

                            return response()->json($response_array);

                        } catch (\Stripe\Error\Authentication $e) {

                            // Authentication with Stripe's API failed

                            $error_message = $e->getMessage();

                            $error_code = $e->getCode();

                            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

                            return response()->json($response_array);

                        } catch (\Stripe\Error\ApiConnection $e) {

                            // Network communication with Stripe failed

                            $error_message = $e->getMessage();

                            $error_code = $e->getCode();

                            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

                            return response()->json($response_array);

                        } catch (\Stripe\Error\Base $e) {
                          // Display a very generic error to the user, and maybe send
                            
                            $error_message = $e->getMessage();

                            $error_code = $e->getCode();

                            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

                            return response()->json($response_array);

                        } catch (Exception $e) {
                            // Something else happened, completely unrelated to Stripe

                            $error_message = $e->getMessage();

                            $error_code = $e->getCode();

                            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

                            return response()->json($response_array);
                        
                        }

                    } else {

                        $response_array = ['success' => false, 'error' => Helper::error_message(142) , 'error_code' => 142];

                        return response()->json($response_array , 200);
                    }

                }

                $request_details->payment_mode = $request->payment_mode ? $request->payment_mode : COD;

                $request_payment->currency = $request_details->currency ? $request_details->currency : Setting::get('currency');

                $request_payment->sub_total = $request_payment->base_price + $request_payment->time_price;

                $request_payment->actual_total = $actual_total ? $actual_total : 0.00;

                $request_details->save();

                $request_payment->save();

                if($is_coupon_applied == COUPON_APPLIED) {

                    $request_payment->coupon_code = $request->coupon_code ? $request->coupon_code : "";

                    $request_payment->coupon_type = $coupon_details->amount_type ? $coupon_details->amount_type : "";

                    $request_payment->coupon_amount = $amount_convertion ? $amount_convertion : 0.00;

                    $request_payment->is_coupon_applied = YES;

                    $request_payment->total = $request_details->total = $request_payment->total - $amount_convertion;
                
                }

                $request_details->save();

                $request_payment->save();

                DB::commit();

                // Send notification to the provider Start

                if($user_details) {

                    $title =  "The"." ".$user_details->first_name.' '.$user_details->last_name." done the payment";

                } else {

                    $title = Helper::tr('request_completed_user_title');
                }

                // Send Push Notification to Provider

                $title = $user_details ? "The"." ".$user_details->first_name.' '.$user_details->last_name." done the payment" : Helper::push_message(8008);

                $message = $user_details ? "The"." ".$user_details->first_name.' '.$user_details->last_name." done the payment" : Helper::push_message(8009);

                if ($request_details->provider) {

                    $email = EmailTemplate::getRowValue(USER_PAYMENT_COMPLETED, $request_details->provider, $request_details->user, $request_details);

                    $notification = NotificationTemplate::getRowValue(USER_PAYMENT_COMPLETED, $request_details->provider, $request_details->user, $request_details);

                    $content = $notification ? $notification : USER_PAYMENT_COMPLETED;

                    if (Setting::get('provider_notification_control')) {

                        ProviderNotification::save_notification($request_details->provider, $content, $request_details->user_id,$request_details->id, USER_PAYMENT_COMPLETED);

                    }

                    if (Setting::get('sms_notification')) {

                        SmsTemplate::getRowValue(USER_PAYMENT_COMPLETED, $request_details->provider);
                                
                    }
                }

                $this->dispatch(new sendPushNotification($request_details->provider_id,PROVIDER,PUSH_PROVIDER_ONGOING,$title,$message));

                // Send mail notification to provider , user and admin

                $subject = tr('mail_request_complete_bill');

                $page = "emails.provider.invoice";

                $email = $request_details->provider->email;
                
                Helper::send_email($page,$subject,$email,$request_details);

                $subject = tr('mail_request_complete_bill');

                $page = "emails.admin.invoice";

                $email = get_admin_mail()->email;
                
                Helper::send_email($page,$subject,$email,$request_details);

                $subject = tr('mail_request_complete_bill');

                $page = "emails.user.invoice";

                $email = $user_details->email;
                
                Helper::send_email($page,$subject,$email,$request_details);

                $response_array = array('success' => true , 'message' => Helper::success_message(227) , 'code' => 120);

            } else {

                throw new Exception( Helper::error_message(144), 144);
                
            }

            \Log::info('User payment END');

            return response()->json($response_array , 200);

        } catch(Exception $e) {

            DB::rollback();

            $response_array = ['success' => false , 'error' => $e->getMessage() , 'error_code' => $e->getCode()];

            return response()->json($response_array, 200);
        }

    }

    /**
     * @method requests_rating()
     *
     * @uses After request complete, the user will rate the provider
     * 
     * @created vidhya R
     *
     * @updated vidhya R
     * 
     * @param form data
     *
     * @return json response
     */

    public function requests_rating(Request $request) {

        $user = User::find($request->id);

        $validator = Validator::make(
            $request->all(),
            array(
                'request_id' => 'required|integer|exists:requests,id,user_id,'.$user->id.'|unique:user_ratings,request_id',
                'rating' => 'required|integer|in:'.RATINGS,
                'comment' => 'max:255',            
                ),
            array(
                'exists' => 'The :attribute doesn\'t belong to user:'.$user->id,
                'unique' => 'The :attribute already rated.'
            )
        );
    
        if ($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false, 'error' => $errors, 'error_code' => 101];
        
        } else {

            $request_details = Requests::where('id' ,$request->request_id)->where('status' ,REQUEST_RATING)->first();


            if ($request_details && intval($request_details->status) != REQUEST_COMPLETED) {
                //Save Rating
                $rating = new UserRating();

                $rating->provider_id = $request_details->provider_id;
                $rating->user_id = $request_details->user_id;
                $rating->request_id = $request_details->id;
                $rating->rating = $request->rating;
                $rating->comment = $request->comment ? $request->comment: '';
                $rating->save();

                // abused provider update 

                if($request->is_abuse) {

                    $request_details->is_abused_provider = 1;

                    // check the record exists

                    $check_abuse_provider = AbuseProvider::where('user_id' , $request->id)->where('provider_id' , $request_details->provider_id)->first();

                    $abuse = new UserFavProvider;

                    if($check_abuse_provider) {

                        $abuse = $check_abuse_provider;
                    }

                    $abuse = new AbuseProvider;

                    $abuse->user_rating_id = $rating->id;

                    $abuse->user_id = $request->id;

                    $abuse->request_id = $request_details->id;

                    $abuse->provider_id = $request_details->provider_id;

                    $abuse->status = 1;

                    $abuse->save();

                    // check the provider in fav list, if yes- remove fav provider

                    UserFavProvider::where('provider_id' , $request_details->provider_id)->where('user_id' , $request_details->user_id)->delete();
                }

                // Fav provider list 

                if($request->is_fav_provider) {
                    
                    $request_details->is_fav_provider = $request->is_fav_provider ? $request->is_fav_provider : 0;

                    // check the record exists

                    $check_fav_provider = UserFavProvider::where('user_id' , $request->id)->where('provider_id' , $request_details->provider_id)->first();

                    $fav_provider_details = new UserFavProvider;

                    if($check_fav_provider) {

                        $fav_provider_details = $check_fav_provider;
                    }

                    $fav_provider_details->user_id = $request_details->user_id;

                    $fav_provider_details->provider_id = $request_details->provider_id;

                    $fav_provider_details->request_id = $request_details->id;

                    $fav_provider_details->status = 1;
                    
                    $fav_provider_details->save();

                }

                $request_details->status = REQUEST_COMPLETED;

                $request_details->save();

                if ($request_details->provider) {

                    $email = EmailTemplate::getRowValue(JOB_REQUEST_COMPLETED, $request_details->provider,$request_details->user, $request_details);

                    $notification = NotificationTemplate::getRowValue(JOB_REQUEST_COMPLETED, $request_details->provider,$request_details->user, $request_details);

                    $content = $notification ? $notification : JOB_REQUEST_COMPLETED;

                    if (Setting::get('provider_notification_control')) {

                        ProviderNotification::save_notification($request_details->provider, $content,$request_details->user_id, $request_details->id, JOB_REQUEST_COMPLETED);

                    }

                     if (Setting::get('sms_notification')) {

                            SmsTemplate::getRowValue(JOB_REQUEST_COMPLETED, $request_details->provider);
                                
                        }
                }

                $response_array = ['success' => true , 'message' => Helper::success_message(213) , 'code' => 213];

            } else {

                $response_array = ['success' => false,'error' => Helper::error_message(132),'error_code' => 132];
            }
        }

        return response()->json($response_array, 200);
   
    } 

    /**
     * @method history()
     *
     * @uses to get the completed jobs
     * 
     * @created vidhya R
     *
     * @updated vidhya R
     *
     * @param
     *
     * @return JSON Response
     */

    public function history(Request $request) {

        try {
            
            $base_query= Requests::where('user_id' , $request->id);
           
            $user_details= User::where('id' , $request->id)->first();
            
            $check_status = [REQUEST_RATING, REQUEST_COMPLETED];

            if(!$user_details) {

                throw new Exception(Helper::error_message(178), 178);          
            }
            
            /** only for website **/
            if($request->cancelled_jobs == YES) {

                $base_query = $base_query->whereIn('cancelled_by' , [USER,PROVIDER]);

            } else {
                                
                if($user_details->device_type != DEVICE_WEB) {

                    $cancelled_status = [REQUEST_CANCELLED, REQUEST_NO_PROVIDER_AVAILABLE, REQUEST_TIME_EXCEED_CANCELLED];
                    
                    $check_status = array_merge($cancelled_status,$check_status);

                }

                $base_query = $base_query->whereIn('status', $check_status);
            }

            $requests = $base_query->orderBy('created_at' , 'desc')->get();
      
            $result = [];
            
            if($requests) {
                
                $timezone = get_user_timezone($request->id);

                $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE , "" , $timezone));
            }

            $response_array = ['success' => true , 'data' => $result];

            return response()->json($response_array, 200);
        
            
        } catch (Exception $e) {
           
            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success'=> false, 'error' => $error_message , 'error_code' => $error_code];

            return response()->json($response_array);
        }
    }

    /**
     * @method single_request()
     *
     * @uses to get the particular request details
     * 
     * @created vidhya R
     *
     * @updated vidhya R
     *
     * @param integer request_id
     *
     * @return JSON Response
     */

    public function single_request(Request $request) {

        if(!RequestRepo::requests_exists($request->all(), $errors)) {

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $timezone = get_user_timezone($request->id);

            $response_array = RequestRepo::view($request , USER , $timezone);
        }

        return response()->json($response_array, 200);
    
    }

     /**
     * Used to get the drafts requests
     * 
     * @param : 
     *
     * @return : json response
     */

    public function drafts(Request $request) {

        $request_date = date('Y-m-d H:i:s');

        $query = TempRequests::where('user_id' , $request->id);

        $is_web = '';

        if ($request->has('is_web')) {

            if($request->is_web) {

                $is_web = 1;

            }
        }

        if ($is_web) {

            if ($request->s_k) {

                $query->where('name', 'like', '%' . $request->s_k . '%');

            } 

            if($request->sorting_key) {

                if ($request->sorting_key == 'asc' || $request->sorting_key == 'desc') {

                    $query->orderBy('name', $request->sorting_key);

                } else if ($request->sorting_key == REQUEST_NOW || $request->sorting_key == REQUEST_LATER) {

                    $query->where('job_type', $request->sorting_key);

                } else {

                    $query->orderBy('updated_at' , 'desc');

                }

            }

            if(!$request->searching_key && !$request->sorting_key) {

                $query->orderBy('updated_at' , 'desc');

            }

            $requests = $query->skip($request->skip)->take(10)->get();

        } else {

            $requests = $query->orderBy('updated_at' , 'desc')->get();

        }

        $result = [];

        if($requests){
            $result = Helper::null_safe(RequestRepo::temp_base_response($requests , RESPONSE_MULTIPLE));
        }

        $response_array = ['success' => true , 'data' => $result];

        

        return response()->json($response_array, 200);

    }

     /**
     * Used to get the posted requests
     * 
     * @param 
     *
     * @return json response
     */

    public function posted_requests(Request $request) {

        $timezone = get_user_timezone($request->id);

        $request_date = common_date(date("Y-m-d H:i:s"), $timezone, "d M y");

        $check_status = [REQUEST_CREATED , REQUEST_WAITING , REQUEST_PROVIDER_ASSIGNED];

        $query = Requests::where('job_type' , REQUEST_LATER)
                    ->where('user_id' , $request->id)
                    // ->where('request_date' , '>=' , $request_date)
                    ->whereIn('status' , $check_status)
                    ->where('job_type' , REQUEST_LATER);

        $is_web = '';

        if ($request->has('is_web')) {

            if($request->is_web) {

                $is_web = 1;

            }
        }

        if ($is_web) {


            if ($request->s_k) {

                $query->where('name', 'like', '%' . $request->s_k . '%');

            } 

            if($request->sorting_key) {

                if ($request->sorting_key == 'asc' || $request->sorting_key == 'desc') {

                    $query->orderBy('name', $request->sorting_key);

                } else {

                    $query->orderBy('updated_at' , 'desc');

                }

            }

            if(!$request->searching_key && !$request->sorting_key) {

                $query->orderBy('updated_at' , 'desc');

            }

            $requests = $query->skip($request->skip)->take(10)->get();

        } else {

            $requests = $query->orderBy('created_at' , 'desc')->get();

        }

        $result = [];

        if($requests) {
            
            // base_response($requests , $group = RESPONSE_SINGLE / RESPONSE_MULTIPLE , $bid_response = "" , $timezone = "") $this->api_user_details ? $this->api_user_details->timezone : ""

            $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE ,"" , $timezone));
        }

        $response_array = ['success' => true , 'data' => $result];

        return response()->json($response_array, 200);
    
    }

     /**
     * Used to get the confirmed requests
     * 
     * @param 
     *
     * @return json response
     */

    public function confirmed_requests(Request $request) {

        $timezone = get_user_timezone($request->id);
        
        $request_date = date('Y-m-d H:i:s');

        $check_status = [REQUEST_ONGOING]; $provider_status = [PROVIDER_ACCEPTED];

        $query = Requests::where('user_id' , $request->id)
                    ->where('job_type' , REQUEST_LATER)
                    // ->where('request_date' , '>=' , $request_date)
                    ->whereIn('status' , $check_status)
                    ->whereIn('provider_status' , $provider_status)
                    ->orderBy('updated_at' , 'desc');

        $is_web = '';

        if ($request->has('is_web')) {

            if($request->is_web) {

                $is_web = 1;

            }
        }

        if ($is_web) {


            if ($request->s_k) {

                $query->where('name', 'like', '%' . $request->s_k . '%');

            } 

            if($request->sorting_key) {

                if ($request->sorting_key == 'asc' || $request->sorting_key == 'desc') {

                    $query->orderBy('name', $request->sorting_key);

                } else {

                    $query->orderBy('updated_at' , 'desc');

                }

            }

            if(!$request->searching_key && !$request->sorting_key) {

                $query->orderBy('updated_at' , 'desc');

            }

            $requests = $query->skip($request->skip)->take(10)->get();

        } else {

            $requests = $query->get();

        }

        $result = [];

        if($requests) {

            $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE ,  "" , $timezone));
        }

        $response_array = ['success' => true , 'data' => $result];

        return response()->json($response_array, 200);

    }

     /**
     * Used to get the ongoing requests
     * 
     * @param : 
     *
     * @return : json response
     */

    public function ongoing_requests(Request $request) {

        $timezone = get_user_timezone($request->id);

        $request_date = date('Y-m-d H:i:s');

        $check_status = [REQUEST_ONGOING , REQUEST_COMPLETE_PENDING , WAITING_PAYMENT_CONFIRM,REQUEST_RATING];

        $provider_status = [PROVIDER_NONE, PROVIDER_ASSIGNED , PROVIDER_ACCEPTED];

        $query = Requests::where('user_id' , $request->id)
                    // ->where('job_type' , REQUEST_LATER)
                    // ->where('request_date' , '>=' , $request_date)
                    ->whereIn('status' , $check_status)
                    ->whereNotIn('provider_status' , $provider_status)
                    ->orderBy('updated_at' , 'desc');

        $is_web = '';

        if ($request->has('is_web')) {

            if($request->is_web) {

                $is_web = 1;

            }
        }

        if ($is_web) {

            
            if ($request->s_k) {

                $query->where('name', 'like', '%' . $request->s_k . '%');

            } 

            if($request->sorting_key) {

                if ($request->sorting_key == 'asc' || $request->sorting_key == 'desc') {

                    $query->orderBy('name', $request->sorting_key);

                } else {

                    $query->orderBy('updated_at' , 'desc');

                }

            }

            if(!$request->searching_key && !$request->sorting_key) {

                $query->orderBy('updated_at' , 'desc');

            }


            $requests = $query->skip($request->skip)->take(10)->get();

        } else {

            $requests = $query->get();

        }

        $result = [];

        if($requests) {

            $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE , "" , $timezone));
        }

        $response_array = ['success' => true , 'data' => $result];

        return response()->json($response_array, 200);

    }

    public function get_payment_modes(Request $request) {

        $payment_modes = array();

        $modes = Settings::whereIn('key' , array('cod','paypal','card'))->where('value' , 1)->get();

        if($modes) {
            foreach ($modes as $key => $mode) {
                $payment_modes[$key] = $mode->key;
            }
        }

        $response_array = Helper::null_safe(array('success' => true , 'payment_modes' => $payment_modes));

        return response()->json($response_array,200);
    
    }

    /**
     * @method get_user_payment_modes()
     *
     * @uses to user payment modes + cards list
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id
     *
     * @param string token
     *
     * @return json response 
     */

    public function get_user_payment_modes(Request $request) {

        $user_details = User::find($request->id);

        $payment_data = $data = $card_data = $payment_modes = $other_data = [];

        $payment_modes['cod'] = $payment_modes['card'] = 1;

        $configurations['stripe_secret_key'] = Setting::get('stripe_secret_key');

        $configurations['stripe_publishable_key'] = Setting::get('stripe_publishable_key');

        if($user_cards = Card::where('user_id' , $request->id)->get()) {
            
            foreach ($user_cards as $c => $card) {
                $data['id'] = $card->id;
                $data['customer_id'] = $card->customer_id;
                $data['card_token'] = $card->card_token;
                $data['last_four'] = $card->last_four;
                $data['is_default']= $card->is_default;
                $data['email'] = $card->paypal_email;
                $data['card_type'] = $card->card_type;

                array_push($card_data, $data);
            }
        
        }

        $get_payment_modes = ['card' , 'cod'];

        $payment_modes = $payment_mode_data = [];

        foreach ($get_payment_modes as $key => $payment_mode_details) {

            if($key == 0) {
                $payment_mode_data['name'] = 'Card';
                $payment_mode_data['value'] = 'card';
                $payment_mode_data['status'] = 1;
            } else {
                $payment_mode_data['name'] = 'COD';
                $payment_mode_data['value'] = 'cod';
                $payment_mode_data['status'] = 1;
            }

            array_push($payment_modes, $payment_mode_data);
           
        }

        $other_data['payment_modes'] = $payment_modes;

        $other_data['configurations'] = $configurations;

        $response_array = Helper::null_safe(['success' => true, 'payment_mode' => $user_details->payment_mode , 'card' => $card_data , 'data' => $other_data]);
       
        return response()->json($response_array , 200);

    }

    /**
     * @method payment_mode_update()
     *
     * @uses to update the default payment mode
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id
     *
     * @param string token
     *
     * @return json response 
     */

    public function payment_mode_update(Request $request) {

        try {

            $validator = Validator::make($request->all() ,
                array(
                    'payment_mode' => 'required|in:'.COD.','.PAYPAL.','.CARD,
                    )
                );

             if($validator->fails()) {

                $error = implode(',', $validator->messages()->all());

                throw new Exception($error, 101);
                
            } else {

                $user = User::where('id', '=', $request->id)->update(array('payment_mode' => $request->payment_mode));

                $response_array = Helper::null_safe(['success' => true , 'message' => Helper::success_message(234) , 'code' => 234]);

                return response()->json($response_array , 200);

            }


        } catch(Exception $e) {

            DB::rollback();

            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success'=> false, 'error' => $error_message , 'error_code' => $error_code];

            return response()->json($response_array);
        }

    }

    /**
     * @method cards_delete()
     *
     * @uses to delete the card 
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id
     *
     * @param string token
     *
     * @return json response 
     */

    public function cards_delete(Request $request) {

        $card_id = $request->card_id;

        $validator = Validator::make(
            $request->all(),
            array(
                'card_id' => 'required|integer|exists:cards,id,user_id,'.$request->id,
            ),
            array(
                'exists' => 'The :attribute doesn\'t belong to user:'.$request->id
            )
        );

        if ($validator->fails()) {

           $errors = implode(',', $validator->messages()->all());

           $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $card_details = Card::where('id',$request->card_id)->first();

            if(count($card_details) == 0 ) {

                $response_array = ['success' => false , 'error' => Helper::error_message(507) , 'error_code' => 507];
                
                return response()->json($response_array , 200);
            }

            if($card_details->is_default) {

                $response_array = array('success' => false , 'error' => Helper::error_message(506) , 'code' => 506);

                return response()->json($response_array , 200);

            }

            $user = User::find($request->id);

            $set_default = 0;

            Card::where('id',$request->card_id)->forceDelete();

            if($user) {

                if($user->payment_mode = CARD) {

                    // Check he added any other card

                    if($check_card = Card::where('user_id' , $request->id)->first()) {

                        $check_card->is_default =  DEFAULT_TRUE;

                        $user->card_id = $check_card->id;

                        $check_card->save();

                    } else {
                        $user->payment_mode = COD;
                        $user->card_id = DEFAULT_FALSE;
                    }
               
                }

                $user->save();
            }

            $response_array = array('success' => true , 'message' => Helper::success_message(224) , 'code' => 224);
        }

        return response()->json($response_array , 200);
    
    }

    /**
     * @method cards_default()
     *
     * @uses to get the providers list based services(For later - Selecte Provider)
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, token
     *
     * @return json response 
     */

    public function cards_default(Request $request) {

        Log::info("cards_default".print_r($request->all() , true));

        $validator = Validator::make(
            $request->all(),
            array(
                'card_id' => 'required|integer|exists:cards,id,user_id,'.$request->id,
            ),
            array(
                'exists' => 'The :attribute doesn\'t belong to user:'.$request->id
            )
        );

        if($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());
            $response_array = array('success' => false, 'error' => $errors, 'error_code' => 101);

        } else {

            $user = User::find($request->id);

            $old_default = Card::where('user_id' , $request->id)->where('is_default', DEFAULT_TRUE)->update(array('is_default' => DEFAULT_FALSE));

            $card = Card::where('id' , $request->card_id)->update(array('is_default' => DEFAULT_TRUE));

            if($card) {
                if($user) {
                    $user->card_id = $request->card_id;
                    $user->save();
                }
                
                $response_array = array('success' => true , 'message' => Helper::success_message(225) , 'code' =>225 );

            } else {
                $response_array = array('success' => false , 'error' => tr('something_went_wrong'));
            }
        }
        return response()->json($response_array , 200);

    }

    /**
     * @method provider_details()
     *
     * @uses to get the provider details
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, token
     *
     * @return json response 
     */

    public function provider_details(Request $request) {

        $validator = Validator::make($request->all() , [
                'provider_id' => 'required|exists:providers,id',
                'request_id' => 'exists:requests,id',
                'temp_request_id' => 'exists:temp_requests,id',
            ]);

        if($validator->fails()) {
            
            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];
        
        } else {

            $provider_details = Provider::find($request->provider_id);

            $data = [];

            if($provider_details) {

                $data['provider_id'] = $provider_details->id;
                $data['provider_name'] = $provider_details->name;
                $data['provider_picture'] = $provider_details->picture;
                $data['provider_email'] = $provider_details->email;
                $data['provider_mobile'] = $provider_details->mobile;
                $data['promo_video'] = $provider_details->promo_video;

                $data['service_name'] = "" ; $data['price_per_hour'] = 0;

                $request_details = array();

                if($request->has('request_id')) {

                    $request_details = Requests::find($request->request_id);
                }

                if($request->has('temp_request_id')) {

                    $request_details = TempRequests::find($request->temp_request_id);
                
                }

                if($request_details) {

                    $sub_category_details = SubCategory::find($request_details->sub_category_id);

                    $data['service_name'] = $sub_category_details->name ?: "-";

                    if($sub_category_details) {

                        $provider_service = ProviderService::where('provider_id' , $request->provider_id)->where('sub_category_id' , $request_details->sub_category_id)->first();

                        if($provider_service) {

                            $data['price_per_hour'] = $provider_service->price_per_hour ?: $sub_category_details->price_per_hour;
                        }
                    
                    }
                
                }

                $data['rating'] = UserRating::where('provider_id', $provider_details->id)->avg('rating') ?: 0;

                $rating_list = [];

                if($ratings = $provider_details->userRatings) {

                    foreach ($ratings as $key => $rating_details) {

                        $rating_data['user_name'] = $rating_details->user->name;
                        $rating_data['user_picture'] = $rating_details->user->picture;
                        $rating_data['comment'] = $rating_details->comment;
                        $rating_data['rating'] = $rating_details->rating;

                        array_push($rating_list, $rating_data);
                        
                    }

                }

                $data['ratings'] = $rating_list;

                $gallery_list = [];

                $galleries = ProviderGallery::where('provider_id' , $provider_details->id)->get();

                if($galleries) {

                    foreach ($galleries as $key => $gallery_details) {

                        $gallery_data['provider_gallery_id'] = $gallery_details->id;

                        $gallery_data['picture'] = $gallery_details->picture;

                        array_push($gallery_list, $gallery_data);
                    }
                
                }

                $data['gallery'] = $gallery_list;

            }

            $response_array = ['success' => true , 'data' => $data];
        }

        return response()->json($response_array , 200);

    }

    /**
     * @method providers_list()
     *
     * @uses to get the providers list based services(For later - Selecte Provider)
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, token
     *
     * @return json response 
     */

    public function providers_list(Request $request) {

        // Need category_id , sub_category_id and location details

        $validator = Validator::make($request->all() , [
                'temp_request_id' => 'required|exists:temp_requests,id',
            ]);

        if($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false , ' error' => $errors , 'error_code' => 101];

        } else {

            Log::info("TEMP REQUEST ID".$request->temp_request_id);

            if(!$temp_request = TempRequests::find($request->temp_request_id)) {

                Log::info('TEMP REQUEST NOT FOUND');

                $response_array = ['success' => false , 'error' => Helper::error_message(112) , 'error_code' => 112];

                Log::info('TEMP REQUEST response_array :::::'.print_r($response_array , true));

                return response()->json($response_array , 200);
            }

            $response_array = RequestRepo::provider_list($request , $temp_request , REQUEST_TYPE_LOCATION);
            
            if($response_array['success']) {

                $ids =  explode(',' , $response_array['provider_ids']);

                $sub_category_id = $temp_request->sub_category_id;

                $providers =  Provider::whereIn('id' , $ids)->get();

                $data = [];

                if($providers) {

                    foreach ($providers as $key => $provider_details) {

                        $provider_service = $provider_details->providerServices()->where('sub_category_id' , $sub_category_id)->first();

                        $list['provider_id'] = $provider_details->id;

                        $list['provider_name'] = $provider_details->name;

                        $list['provider_picture'] = $provider_details->picture;

                        $list['currency'] = Setting::get('currency') ? Setting::get('currency') : "$";

                        $list['price_per_hour'] = $provider_service ?  $provider_service->price_per_hour : Setting::get('price_per_hour');

                        $list['service_name'] = $provider_service->subCategory ?  $provider_service->subCategory->name : "";

                        $list['rating'] = UserRating::where('provider_id', $provider_details->id)->avg('rating') ?: 0;

                       array_push($data, $list);
                    
                    }
                
                }

                $temp_request = TempRequests::tempResponse($request->temp_request_id)->find($request->temp_request_id);

                Log::info("TEMP REQUESTS".print_r($temp_request , true));

                $response_array = ['success' => true , 'total' => count($providers), 'temp_request_id' => $request->temp_request_id,'temp_request' => $temp_request,'data' => $data];

            }
        
        }

        return response()->json($response_array, 200);

    }

    /**
     * @method requests_bids()
     *
     * @uses to list of bidded providers for the selected requests
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, token
     *
     * @return json response 
     */

    public function requests_bids(Request $request) {

        $validator = Validator::make($request->all() , [
            'request_id' => 'required|exists:requests,id,user_id,'.$request->id,
        ]);

        if($validator->fails()) {

            $errors = implode(',', $validator->messages()->all());

            $response_array = ['success' => false , 'error' => $errors , 'error_code' => 101];

        } else {

            $timezone = get_user_timezone($request->id);

            $request_details = Requests::find($request->request_id);

            if($request_details) {

                $bids = $request_details->bids;

                $data = $list = [];

                if(count($bids)) {

                    foreach ($bids as $key => $bid_details) {

                        $data['request_id'] = $bid_details->request_id;
                        $data['request_meta_id'] = $bid_details->id;
                        $data['provider_id'] = $bid_details->provider_id;
                        $data['provider_name'] = $bid_details->metaProvider ? $bid_details->metaProvider->name : "";
                        $data['provider_picture'] = $bid_details->metaProvider ? $bid_details->metaProvider->picture : "";
                        $data['currency'] = Setting::get('currency') ? Setting::get('currency') : "$";
                        $data['bid_amount'] = $bid_details->bid_amount;
                        $data['bid_status'] = $bid_details->status;
                        $data['service_name'] = $request_details->subCategory ?  $request_details->subCategory->name : "";                      
                        $data['rating'] = UserRating::where('provider_id', $bid_details->provider_id)->avg('rating') ?: 0;

                        array_push($list , $data);
                    
                    }

                }

                $response_array = ['success' => true , 'data' => $list];

            } else {
                $response_array = ['success' => false , 'error' => Helper::error_message(124) , 'error_code' => 124 ];
            }

        }

        return response()->json($response_array, 200);
    
    }

    /**
     * @method requests_chat_history()
     *
     * @uses to get the messages list between user and provider
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, token
     *
     * @return json response 
     */

    public function requests_chat_history(Request $request) {

        $messages = ChatMessage::where('user_id', $request->id)
                ->where('request_id', $request->request_id);

        if($request->provider_id) {

            ChatMessage::where('user_id', $request->id)
                ->where('request_id', $request->request_id)
                ->where('provider_id' , $request->provider_id)
                ->where('type' , 'pu')
                ->update(['delivered' => 1]);
        }

        $response_array = Helper::null_safe(array(
            'success' => true,
            'data' => $messages->get()->toArray(),
        ));
    
        return response()->json($response_array, 200);
    
    }


    /**
     * @method change_status_notifications()
     *
     * @uses to change the status of the notifications for the selected user
     *
     * @created Shobana C
     *
     * @updated Vidhya R
     *
     * @param integer id, token
     *
     * @return json response 
     */

    public function change_status_notifications(Request $request) {

        $model = UserNotification::where('user_id', $request->id)->where('status', DEFAULT_FALSE)->get();

        $new_value = UserNotification::select('user_notifications.id as id', 
                            'user_notifications.notification as notification', 
                            'providers.picture as picture',
                            'providers.name as name',
                            'user_notifications.provider_id as provider_id',
                            'user_notifications.request_id as request_id',
                            'requests.unique_id as unique_id',
                            'user_notifications.created_at as created_at',
                            'user_notifications.status')
                        ->where('user_notifications.user_id', $request->id)
                        ->leftJoin('providers', 'providers.id', '=', 'user_notifications.provider_id')
                        ->leftJoin('requests', 'requests.id', '=', 'user_notifications.request_id')
                        ->orderBy('created_at', 'desc')
                        ->where('user_notifications.status', DEFAULT_FALSE)->get();

        $notifications = [];

        foreach ($new_value as $key => $value) {

            $value->status = DEFAULT_TRUE;
            $value->save();


            $notifications[] = ["notify_id"=>$value->id,
                                "notification"=>$value->notification,
                                "picture"=>$value->picture ? $value->picture : asset('placeholder.png'),
                                "name"=>$value->name,
                                "provider_id"=>$value->provider_id,
                                "request_id"=>$value->request_id,
                                "unique_id"=>$value->unique_id,
                                "created_at"=>$value->created_at->diffForHumans(),
                                'url'=>route('user.jobs.view' , ['request_unique_id' => $value->unique_id]),
                                "status"=>$value->status]; 

        }

        return response()->json(['cnt'=>count($notifications), 'notifications'=>$notifications]);

    }

    /**
     * @method change_confirmed_notifications()
     *
     * @uses to change the status of the confirmed notifications for the selected user
     *
     * @created Shobana C
     *
     * @updated Vidhya R
     *
     * @param integer id, token
     *
     * @return json response 
     */

    public function change_confirmed_notifications(Request $request) {

        $model = UserNotification::where('user_id', $request->id)->where('status', DEFAULT_FALSE)
            ->where('job_status', 1)->get();

        foreach ($model as $key => $value) {

            $value->status = DEFAULT_TRUE;

            $value->save();

        }
    }

    /**
     * @method change_ongoing_notifications()
     *
     * @uses to change the status of the ongoing notifications for the selected user
     *
     * @created Shobana C
     *
     * @updated Vidhya R
     *
     * @param integer id, token
     *
     * @return json response 
     */

    public function change_ongoing_notifications(Request $request) {

        $model = UserNotification::where('user_id', $request->id)->where('status', DEFAULT_FALSE)
            ->where('job_status', 2)->get();

        foreach ($model as $key => $value) {

            $value->status = DEFAULT_TRUE;

            $value->save();

        }

    }

    /**
     * @method user_notifications()
     *
     * @uses to get the notifications for the selected user
     *
     * @created Shobana C
     *
     * @updated Vidhya R
     *
     * @param integer id, token
     *
     * @return json response 
     */
    
    public function user_notifications(Request $request) {

        $new_value = UserNotification::select('user_notifications.id as notify_id', 
                            'user_notifications.notification as notification', 
                            'providers.picture as picture',
                            'providers.name as name',
                            'user_notifications.provider_id as provider_id',
                            'user_notifications.request_id as request_id',
                            'requests.unique_id as unique_id',
                            'user_notifications.created_at as created_at',
                            'user_notifications.status')
                        ->where('user_notifications.user_id', $request->id)
                        ->leftJoin('providers', 'providers.id', '=', 'user_notifications.provider_id')
                        ->leftJoin('requests', 'requests.id', '=', 'user_notifications.request_id')
                        ->where('user_notifications.status', DEFAULT_FALSE)->get();

        $notifications = [];

        foreach ($new_value as $key => $value) {
        
            $notifications[] = ["notify_id"=>$value->notify_id,
                                "notification"=>$value->notification,
                                "picture"=>$value->picture ? $value->picture : asset('placeholder.png'),
                                "name"=>$value->name,
                                "provider_id"=>$value->provider_id,
                                "request_id"=>$value->request_id,
                                "unique_id"=>$value->unique_id,
                                "created_at"=>$value->created_at->diffForHumans(),
                                'url'=>route('user.jobs.view' , ['request_unique_id' => $value->unique_id]),
                                "status"=>$value->status]; 

        }

        $new_cnt = count($new_value);

        $confirmed_requests = UserNotification::where('user_id', $request->id)->where('status', DEFAULT_FALSE)
            ->where('job_status', 1)->count();

        $ongoing_requests = UserNotification::where('user_id', $request->id)->where('status', DEFAULT_FALSE)
            ->where('job_status', 2)->count();

        return response()->json(['cnt'=>$new_cnt, 'new_notify'=>$notifications, 'confirmed_requests'=>$confirmed_requests, 'ongoing_requests'=>$ongoing_requests]);

    }

    /**
     * @method logout()
     *
     * @uses to logout function of the user
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param int id
     *
     * @param int token
     *
     * @return json response 
     */

    public function logout(Request $request) {

        $user_details = User::find($request->id);

        if(count($user_details) == 0) {

            $response_array = ['success' => false , 'error' => Helper::error_message(137) , 'error_code' => 137];

            return response()->json($response_array , 200);

        }

        $user_details->device_token = "";

        $user_details->save();

        $response_array = ['success' => true , 'message' => tr('logout_success') ];

        return response()->json($response_array , 200);

    }

    /**
     * @method cancelled_requests()
     *
     * @uses cancelled jobs list
     *
     * @param integer id
     *
     * @param integer token
     *
     * @return JSON reponse 
     */

    public function cancelled_requests(Request $request) {

        $timezone = get_user_timezone($request->id);

        $request_date = common_date(date("Y-m-d H:i:s"), $this->timezone, "d M y");

        $check_status = [REQUEST_CANCELLED, REQUEST_NO_PROVIDER_AVAILABLE, REQUEST_TIME_EXCEED_CANCELLED];

        $query = Requests::where('user_id' , $request->id)
                    ->whereIn('status' , $check_status);

        $is_web = '';

        if ($request->has('is_web')) {

            if($request->is_web) {

                $is_web = 1;

            }
        }

        if ($is_web) {

            if ($request->s_k) {

                $query->where('name', 'like', '%' . $request->s_k . '%');

            } 

            if($request->sorting_key) {

                if ($request->sorting_key == 'asc' || $request->sorting_key == 'desc') {

                    $query->orderBy('name', $request->sorting_key);

                } else {

                    $query->orderBy('updated_at' , 'desc');

                }

            }

            if(!$request->searching_key && !$request->sorting_key) {

                $query->orderBy('updated_at' , 'desc');

            }

            $requests = $query->skip($request->skip)->take(10)->get();

        } else {

            $requests = $query->get();

        }

        $result = [];

        if($requests) {
            
            // base_response($requests , $group = RESPONSE_SINGLE / RESPONSE_MULTIPLE , $bid_response = "" , $timezone = "") $this->api_user_details ? $this->api_user_details->timezone : ""

            $result = Helper::null_safe(RequestRepo::base_response($requests , RESPONSE_MULTIPLE ,"" , $timezone));
        }

        $response_array = ['success' => true , 'data' => $result];

        return response()->json($response_array, 200);

    }


    /**
     * @method search()
     * 
     * @uses to get the search details based on the search key
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param string 
     *
     * @return boolean 
     */

    public function search(Request $request) {

        DB::beginTransaction();

        try {

            // CATEGORY_APPROVED  status handled in model.
            
            $categories = Category::where('categories.name', 'like', '%' . $request->key . '%')->baseResponse()->get();

            // SUB_CATEGORY_APPROVED  status handled in model.

            $sub_categories = SubCategory::where('sub_categories.name', '%like%', $request->key)->baseResponse()->get();

            $data['categories'] = tr('search_categories');

            $data['category_data'] = $categories ? $categories->toArray() : [];

            $data['sub_categories'] = tr('search_sub_categories');

            $data['sub_category_data'] = $sub_categories ? $sub_categories->toArray() : [];

            $data['jobs'] = tr('jobs');

            $data['job_data'] = [];

            $response_array = ['success' => true , 'data' => $data];

            return response()->json($response_array, 200);

        } catch(Exception $e) {

            DB::rollback();

            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

            return response()->json($response_array);
        
        }
   
    }

    /**
     * @method cards_add()
     * 
     * @uses to add card by stripe payment gateway
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param card token
     *
     * @return JSON 
     */

    public function cards_add(Request $request) {

        try {

            DB::beginTransaction();

            if(Setting::get('stripe_secret_key')) {

                \Stripe\Stripe::setApiKey(Setting::get('stripe_secret_key'));

            } else {

                throw new Exception(Helper::error_message(164), 164);
            }
        
            $validator = Validator::make(
                    $request->all(),
                    [
                        'card_token' => 'required',
                        'last_four' => '',
                    ]
                );

            if ($validator->fails()) {

                $error = implode(',',$validator->messages()->all());
             
                throw new Exception($error , 101);

            } else {

                Log::info("INSIDE CARDS ADD");

                $this->loginUser = count($this->loginUser) > 0 ? $this->loginUser : User::find($request->id);

                $description = Setting::get('site_name')."-".$this->loginUser ? $this->loginUser->email: "";

                $user_details = User::find($request->id);

                if(!$user_details) {

                    throw new Exception(Helper::error_message(504), 504);
                    
                }

                $stripe_gateway_details = [
                        
                        "card" => $request->card_token,
                        
                        "email" => $user_details->email,
                        
                        "description" => "Customer for ".Setting::get('site_name'),
                        
                    ];


                // Get the key from settings table
                
                $customer = \Stripe\Customer::create($stripe_gateway_details);
                
                if($customer) {

                    $card_details = new Card;

                    $card_details->user_id = $request->id;

                    $card_details->customer_id = $customer->id;

                    $card_details->card_token = $customer->sources->data ? $customer->sources->data[0]->id : "";

                    $card_details->card_type = $customer->sources->data ? $customer->sources->data[0]->brand : "";

                    $card_details->last_four = $customer->sources->data[0]->last4 ? $customer->sources->data[0]->last4 : "";

                    // check the user having any cards 

                    $check_user_cards = Card::where('user_id',$request->id)->first();

                    $card_details->is_default = $check_user_cards ? 0 : 1;

                    if($card_details->save()) {

                        $this->loginUser->card_id = $check_user_cards ? $this->loginUser->card_id : $card_details->id;

                        $this->loginUser->save();

                        $response_array = array('success' => true , 'message' => Helper::success_message(233)); 

                    } else {

                        throw new Exception(Helper::error_message(165), 165);

                    }
               
                } else {

                    throw new Exception(Helper::error_message(165) , 165);
                    
                }
            
            }

            DB::commit();

            return response()->json($response_array , 200);

        } catch(Stripe_CardError $e) {

            Log::info("error1");

            $error1 = $e->getMessage();

            $response_array = array('success' => false , 'error' => $error1 ,'error_code' => 165);

            return response()->json($response_array , 200);

        } catch (Stripe_InvalidRequestError $e) {

            // Invalid parameters were supplied to Stripe's API

            Log::info("error2");

            $error2 = $e->getMessage();

            $response_array = array('success' => false , 'error' => $error2 ,'error_code' => 165);

            return response()->json($response_array , 200);

        } catch (Stripe_AuthenticationError $e) {

            Log::info("error3");

            // Authentication with Stripe's API failed
            $error3 = $e->getMessage();

            $response_array = array('success' => false , 'error' => $error3 ,'error_code' => 165);

            return response()->json($response_array , 200);

        } catch (Stripe_ApiConnectionError $e) {
            Log::info("error4");

            // Network communication with Stripe failed
            $error4 = $e->getMessage();

            $response_array = array('success' => false , 'error' => $error4 ,'error_code' => 165);

            return response()->json($response_array , 200);

        } catch (StripeInvalidRequestError $e) {

            Log::info("error7");

            $response_array = array('success' => false , 'error' => Helper::error_message(165) ,'error_code' => 165);

            return response()->json($response_array , 200);

        } catch (Stripe_Error $e) {
            Log::info("error5");

            // Display a very generic error to the user, and maybe send
            // yourself an email
            $error5 = $e->getMessage();

            $response_array = array('success' => false , 'error' => $error5 ,'error_code' => 165);

            return response()->json($response_array , 200);

        } catch(Exception $e) {

            DB::rollback();

            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            Log::info("catch FUNCTION INSIDE".$error_message);

            $response_array = ['success'=>false, 'error'=> $error_message , 'error_code' => $error_code];

            return response()->json($response_array , 200);
        
        }
   
    }

    /**
     * @method coupon_code_apply()
     *
     * @uses Apply coupon to request payment if the user having coupon codes
     *
     * @created vidhya R
     *
     * @updated vidhya R
     *
     * @param object $request - User details, request details
     *
     * @return response of coupon details with amount
     *
     */
    public function coupon_code_apply(Request $request) {

        /**********************
         |
         | The coupon code will follow the below logics
         |
         | step 1: check the basic validations
         |
         | step 2: check the request and user record existence and check the request is not cancelled
         |
         | step 3: check the coupon code record, status and expiry date
         |
         | step 4: check the user is eligible to use the coupon code
         |
         | step 5: If user eligiable, calculate the amout
        */

        try {

            DB::beginTransaction();

            // step 1: check the basic validations

            $validator = Validator::make($request->all(), 
                [
                    'coupon_code' => 'required|exists:coupons,coupon_code',  
                    'request_id'=>'required|exists:requests,id,user_id,'.$request->id          
                ], array(
                    'coupon_code.exists' => Helper::error_message(168),
                    'request_id.exists' => tr('request_not_found'),
                ));
            
            if ($validator->fails()) {

                $error = implode(',', $validator->messages()->all());

                throw new Exception($error, 101);
            
            }

            // step 2: check the request and user record existence and check the request is not cancelled

            $user_details = User::find($request->id);

            $request_details = Requests::where('user_id' , $request->id)->where('requests.id' , $request->request_id)->first();

            if(count($request_details) == 0) {

                throw new Exception(Helper::error_message(124), 124);
                
            }

            // Check the requests is cancelled

            if(in_array($request_details->status, [REQUEST_CANCELLED, REQUEST_NO_PROVIDER_AVAILABLE, REQUEST_TIME_EXCEED_CANCELLED])) {

                throw new Exception(Helper::error_message(173), 173);
                
            }

            // step 3: check the coupon code record, status and expiry date

            $coupon_details = Coupon::where('coupon_code', $request->coupon_code)->first();

            if (count($coupon_details) == 0) {

                throw new Exception(Helper::error_message(168), 168);
                
            }

            if($coupon_details->status != 1) {

                throw new Exception(Helper::error_message(169), 169);

            }

            // check the expiry date

            if(strtotime($coupon_details->expiry_date) < strtotime(date('Y-m-d'))) {

                throw new Exception(Helper::error_message(172), 172);
                
            }

            // step 4: check the user is eligible to use the coupon code

            $check_coupon = $this->coupon_code_user_use_check($user_details, $coupon_details)->getData();

            if ($check_coupon->success) {

                // step 5: If the user is eligiable, calculate the amount

                $amount_convertion = $coupon_details->amount;

                if ($coupon_details->amount_type == PERCENTAGE) {

                    $amount_convertion = amount_convertion($coupon_details->amount, $request_details->total);

                }

                if ($request_details->total > $amount_convertion && $amount_convertion > 0) {

                    $amount = $request_details->total - $amount_convertion;

                    $data['currency'] = $request_details->currency;

                    $data['remaining_amount'] = $amount;

                    $data['coupon_amount'] = $amount_convertion;

                    $data['coupon_code'] = $coupon_details->coupon_code;

                    $data['original_coupon_amount'] = $coupon_details->amount_type == PERCENTAGE ? $coupon_details->amount.'%' : Setting::get('currency').$coupon_details->amount;

                    $response_array = ['success'=> true, 'data'=> $data];

                } else {

                    $amount = 0;

                    $data['remaining_amount'] = $amount;

                    $data['coupon_amount'] = $amount_convertion;

                    $data['coupon_code'] = $coupon_details->coupon_code;

                    $data['original_coupon_amount'] = $coupon_details->amount_type == PERCENTAGE ? $coupon_details->amount.'%' : Setting::get('currency').$coupon_details->amount;

                    $response_array = ['success'=> true, 'data'=> $data];

                }

            } else {

                throw new Exception($check_coupon->error, $check_coupon->error_code);
                
            }

            return response()->json($response_array);

        } catch(Exception $e) {

            DB::rollback();

            $error_message = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success' => false , 'error' => $error_message , 'error_code' => $error_code];

            return response()->json($response_array, 200);
        }

    }

    /**
     * @method coupon_code_user_use_check()
     *
     * @uses To check the coupon code applicable to the user or not
     *
     * @created vidhya R
     *
     * @updated vidhya R
     *
     * @param objects $coupon - Coupon details
     *
     * @param objects $user - User details
     *
     * @return response of success/failure message
     */
    public function coupon_code_user_use_check($user_details, $coupon_details) {

        try {

            $sum_of_users = UserCoupon::where('coupon_code', $coupon_details->coupon_code)->sum('no_of_times_used');

            if ($sum_of_users < $coupon_details->no_of_users_limit) {

            } else {

                throw new Exception(Helper::error_message(170), 170);
                
            }

            $user_coupon_details = UserCoupon::where('user_id', $user_details->id)
                                    ->where('coupon_code', $coupon_details->coupon_code)
                                    ->first();

            // If user coupon not exists, create a new row

            if ($user_coupon_details) {

                if ($user_coupon_details->no_of_times_used < $coupon_details->per_users_limit) {

                    $response_array = ['success' => true, 'message'=> Helper::success_message(238), 'code' =>  2002];

                } else {

                    throw new Exception(Helper::error_message(171) , 171);
                }

            } else {

                $response_array = ['success' => true, 'message'=> Helper::success_message(239), 'code' => 2001];

            }

            return response()->json($response_array);

        } catch (Exception $e) {

            $response_array = ['success'=>false, 'error'=>$e->getMessage()];

            return response()->json($response_array);
        }

    }

    /**
     * @method fav_providers()
     *
     * @uses to get the fav provider list
     * 
     * @created vidhya R
     *
     * @updated vidhya R
     *
     * @param 
     *
     * @return JSON Response
     */

    public function fav_providers(Request $request) {

        try {

            $fav_providers = UserFavProvider::where('user_id' , $request->id)->get();

            $data = $fav_data = [];

            foreach ($fav_providers as $key => $fav_provider_details) {

                $provider_details = Provider::find($fav_provider_details->provider_id);

                if($provider_details) {

                    $fav_data['user_fav_provider_id'] = $fav_provider_details->id;

                    $fav_data['provider_id'] = $fav_provider_details->provider_id;

                    $fav_data['provider_name'] = $provider_details->name;

                    $fav_data['provider_picture'] = $provider_details->picture;

                    $fav_data['user_id'] = $fav_provider_details->user_id;

                    $fav_data['provider_rating'] = 4;

                    array_push($data, $fav_data);
                }
            }

            $response_array = ['success' => true , 'data' => $data];

            return response()->json($response_array , 200);

        } catch(Exception $e) {

            $error = $e->getMessage();

            $response_array = ['success' => false , 'error' => $error , 'error_code' => 101];

            return response()->json($response_array , 200);
        }
    
    }

    /**
     * @method fav_providers_remove()
     *
     * @uses to remove the selected fav provider
     * 
     * @created vidhya R
     *
     * @updated vidhya R
     *
     * @param 
     *
     * @return JSON Response
     */

    public function fav_providers_remove(Request $request) {

        try {

            $fav_provider_details = UserFavProvider::where('provider_id' , $request->user_fav_provider_id)->delete();

            if(!$fav_provider_details){

                throw new Exception(Helper::error_message(175), 175);
            }
            
            $response_array = ['success' => true , 'message' => Helper::success_message(240) , 'code' => 240];
            
            return response()->json($response_array , 200);

        } catch(Exception $e) {

            $error = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success' => false , 'error' => $error , 'error_code' => $error_code];

            return response()->json($response_array , 200);
        }
    
    }

    /**
     * @method referrals()
     *
     * @uses signup user through referrals
     *
     * @created Vithya R
     *
     * @updated Vithya R
     *
     * @param string referral_code 
     *
     * @return redirect signup page
     */
    public function referrals(Request $request) {

        try {

            $user_details =  User::find($request->id);

            $user_referrer_details = UserReferrer::where('user_id', $user_details->id)->first();

            if(!$user_referrer_details) {

                $user_referrer_details = new UserReferrer;

                $user_referrer_details->user_id = $user_details->id;

                $user_referrer_details->referral_code = uniqid();

                $user_referrer_details->total_referrals = $user_referrer_details->total_referrals_earnings = 0 ;

                $user_referrer_details->save();

            }

            unset($user_referrer_details->id);

            $referrals = Referral::where('parent_user_id', $user_details->id)->CommonResponse()->orderBy('created_at', 'desc')->get();

            foreach ($referrals as $key => $referral_details) {

                $user_details = User::find($referral_details->user_id);

                $referral_details->username = $referral_details->picture = "";

                if($user_details) {

                    $referral_details->username = $user_details->name ?: "";

                    $referral_details->picture = $user_details->picture ?: "";

                }

            }

            $user_referrer_details->currency = Setting::get('currency', '$');

            // share message start

            $share_message = tr('referral_code_share_message', Setting::get('site_name', 'STREAMTUBE'));


            $share_message = str_replace('<%referral_code%>', $user_referrer_details->referral_code, $share_message);

            $share_message = str_replace("<%referral_commission%>", formatted_amount(Setting::get('referral_commission', 10)),$share_message);

            $referrals_signup_url = route('referrals_signup', ['referral_code' => $user_referrer_details->referral_code]);

            $user_referrer_details->share_message = $share_message." ".$referrals_signup_url;


            $user_referrer_details->share_message = $share_message." ".$referrals_signup_url;

            $user_referrer_details->referrals = $referrals;

            $response_array = ['success' => true, 'data' => $user_referrer_details];

            return response()->json($response_array, 200);

        } catch(Exception $e) {

            $error_messages = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success' => false, 'error' => $error_messages, 'error_code' => $error_code];

            return response()->json($response_array);

        }
    }

    /**
     * @method referrals_check()
     *
     * @uses check valid referral
     *
     * @created Vithya R
     *
     * @updated Vithya R
     *
     * @param string referral_code 
     *
     * @return redirect signup page
     */
    public function referrals_check(Request $request){

        try {

            $validator = Validator::make($request->all(),[
                    'referral_code' =>'required|exists:user_referrers,referral_code',
                ],
                [
                    'exists' => Helper::error_message(50101),
                ]
            );

            if ($validator->fails()) {

                $error_messages = implode(',', $validator->messages()->all());

                throw new Exception($error_messages, 101);
                
            }

            $check_referral_code =  UserReferrer::where('referral_code', $request->referral_code)->where('status', APPROVED)->first();

            if(!$check_referral_code) {

                throw new Exception(Helper::error_message(50101), 50101);
                
            }

            $response_array = ['success' => true, 'message' => Helper::success_message(50001), 'code' => 50001];

            return response()->json($response_array, 200);

        } catch(Exception $e) {

            $error_messages = $e->getMessage();

            $error_code = $e->getCode();

            $response_array = ['success' => false, 'error' => $error_messages, 'error_code' => $error_code];

            return response()->json($response_array, 200);

        }
    }
}

