<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Helpers\Helper;

use App\User;

use App\Provider;

use App\Requests;

use App\TempRequests;

use App\Page;

use DateTimeZone;

use Log;

use App\Card;

use App\UserNotification;

use Auth;

use App\UserRating;

use App\UserFavProvider;

use App\RequestPayment;

use App\Referral;

use App\UserReferrer;

class UserController extends Controller {

    protected $UserAPI;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(UserApiController $API) {

        $this->UserAPI = $API;
        
        $this->middleware('WebUserVal', ['except' => ['choose_register' , 'choose_login']]);

        $this->middleware('auth', ['except' => ['choose_register' , 'choose_login', 'referrals_signup' ]]);
    }


    /**
     * @method index()
     *
     * @uses index page 
     *
     * @created Vidhya R 
     *
     * @updated Vidhya R
     *
     * @param - 
     *
     * @return view page
     */

    public function index(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
            'is_web'=>DEVICE_WEB,
            'skip'=>0
        ]);
        
        counter('home');

        $job_active = '';

        // Check any ongoing ondemand and later requests 

        $ondemand_requests = $this->UserAPI->request_status_check_now($request)->getData();

        $later_requests = $this->UserAPI->request_status_check_later($request)->getData();

        $display = false;


        // If any ongoing requests means redirect to the ongoing page

        if($ondemand_requests->success || $later_requests->success) {

            if($ondemand_requests->data || $later_requests->data) {

                $ondemand_requests = $ondemand_requests->data;

                $later_requests = $later_requests->data;

                $display = true;

                return view('user.jobs.ongoing')->with('ondemand_requests' , $ondemand_requests)
                    ->with('later_requests' , $later_requests);
            }
        } 

        $drafts_response = $this->UserAPI->drafts($request)->getData();

        if($drafts_response->success) 
            $drafts = $drafts_response->data;

        $posted_response = $this->UserAPI->posted_requests($request)->getData();

        if($posted_response->success) 
            $posted_requests = $posted_response->data;

        $confirmed_response = $this->UserAPI->confirmed_requests($request)->getData();

        if($confirmed_response->success) 
            $confirmed_requests = $confirmed_response->data;

        $ongoing_response = $this->UserAPI->ongoing_requests($request)->getData();

        if($ongoing_response->success) 
            $ongoing_requests = $ongoing_response->data;

        $job_active = $request->active ? $request->active : 'posted-jobs';

        if ($job_active == 'confirm-jobs' && confirmed_job_notifications() > 0) {

            $model = UserNotification::where('user_id', Auth::user()->id)->where('status', DEFAULT_FALSE)
            ->where('job_status', 1)->get();

            foreach ($model as $key => $value) {

                $value->status = DEFAULT_TRUE;
                $value->save();

            }

        }
        
        if ($job_active == 'ongoing-jobs' && ongoing_job_notifications() > 0) {

            $model = UserNotification::where('user_id', Auth::user()->id)->where('status', DEFAULT_FALSE)
            ->where('job_status', 2)->get();

             \Log::info("Job Active ".print_r($model, true));

            foreach ($model as $key => $value) {

                $value->status = DEFAULT_TRUE;
                $value->save();

            }

        }

        return view('user.jobs.index')->with('posted_requests', $posted_requests)
                            ->with('drafts' , $drafts)
                            ->with('confirmed_requests' , $confirmed_requests)
                            ->with('ongoing_requests' , $ongoing_requests)->with('display', $display)
                            ->with('job_active', $job_active)
                            ->with('header_page' , "dashboard");
        
    }

    /**
     * @method profile()
     *
     * @uses display profile details of the logged in user 
     *
     * @created Vidhya R 
     *
     * @updated Vidhya R
     *
     * @param - 
     *
     * @return view page
     */
    public function profile(Request $request) {

        $fav_providers = UserFavProvider::where('user_id', Auth::user()->id)->get();

        $profile_active = $request->profile_active ? $request->profile_active : "profile"; 

        Log::info("profile_active".$profile_active);

        return view('user.account.profile')
                    ->with('page' , 'profile')
                    ->with('subPage' , 'user-profile')
                    ->with('profile_active' , $profile_active)
                    ->with('header_page' , "account")
                    ->with('fav_providers' , $fav_providers);
    }    

    /**
     * @method update_profile()
     *
     * @uses update profile details of the logged in user 
     *
     * @created Vidhya R 
     *
     * @updated Vidhya R
     *
     * @param - Form data
     *
     * @return view page
     */
    public function profile_save(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
        ]);

        $response = $this->UserAPI->update_profile($request)->getData();

        if($response->success) {
            return back()->with('flash_success', tr('profile_updated'));
        } else {
            $message = $response->error;
            return back()->with('flash_error', $message);
        }
        
    }

   /**
     * @method profile_change_password()
     *
     * @uses change the password of the user and logout
     *
     * @created Vidhya R 
     *
     * @updated Vidhya R
     *
     * @param - form data
     *
     * @return \Illuminate\Http\Response
     */

    public function profile_change_password(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
        ]);

        $response = $this->UserAPI->change_password($request)->getData();

        if($response->success) {

            Auth::logout();

            return redirect()->route('user.login.form')->with('flash_success', tr('password_success'));

        } else {

            $message = $response->error;

            return back()->with('flash_error', $message)->with('profile_active' , 'change_password');

        }
    
    }

    /**
     * @method delete_account()
     *
     * @uses delete account based on the selected in user
     *
     * @created Vidhya R 
     *
     * @updated Vidhya R
     *
     * @param - form data
     *
     * @return \Illuminate\Http\Response
     */

    public function delete_account(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token
        ]);

        $response = $this->UserAPI->delete_account($request)->getData();

        if($response->success) {

            return redirect()->route('user.login.form')->with('flash_success', tr('user_account_delete_success'));

        } else {

            return back()->with('flash_error', $response->error);
        }

        return back()->with('flash_error', Helper::error_message(146));

    }

    /**
     * Function name: cards_add()
     *
     * @uses used to add cards
     *
     * @created vidhya R
     *
     * @updated vidhya R
     * 
     * @param integer id
     *
     * @param integer token
     *
     * @return render view page 
     */

    public function cards_add(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'card_token'=>$request->stripeToken,
            'cvv'=>$request->cvc,
            'month'=>$request->expiry_month,
            'year'=>$request->expiry_year,
        ]);

        $response = $this->UserAPI->cards_add($request)->getData();

        if($response->success) {

            return redirect()->route('user.profile' , ['profile_active' => 'billing'])->with('flash_success', $response->message);

        } else {

            return redirect()->route('user.profile' , ['profile_active' => 'billing'])->with('flash_error', $response->error);

        }

    }


    /**
     * Function name: cards_default()
     *
     * @uses used to delete the selected cards
     *
     * @created vidhya R
     *
     * @updated vidhya R
     * 
     * @param integer id
     *
     * @param integer token
     *
     * @return render view page 
     */

    public function cards_delete(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token
        ]);

        $response = $this->UserAPI->cards_delete($request)->getData();

        if($response->success) {

            return redirect()->route('user.profile' , ['profile_active' => 'billing'])->with('flash_success', $response->message);

        } else {

            return redirect()->route('user.profile' , ['profile_active' => 'billing'])->with('flash_error', $response->error);

        }

        return redirect()->route('user.profile' , ['profile_active' => 'billing'])->with('flash_error', Helper::error_message(146));
    
    }

    /**
     * Function name: cards_default()
     *
     * @uses used to set the default card
     *
     * @created vidhya R
     *
     * @updated vidhya R
     * 
     * @param integer id
     *
     * @param integer token
     *
     * @return render view page 
     */

    public function cards_default(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token
        ]);

        $response = $this->UserAPI->cards_default($request)->getData();

        if($response->success) {

            return redirect()->route('user.profile' , ['profile_active' => 'billing'])->with('flash_success', $response->message);
       
        } else {

            return redirect()->route('user.profile' , ['profile_active' => 'billing'])->with('flash_error', $response->error);
        }

        return redirect()->route('user.profile' , ['profile_active' => 'billing'])->with('flash_error', Helper::error_message(146));

    }

    public function notifications(Request $request) {

        return view('user.account.notifications');
    }

    /**
     * function name: provider_profile()
     *
     * @uses display the provider details
     *
     * @created vidhya R
     *
     * @updated vidhya R
     *
     * @param string $provider_unique_id
     *
     * @param string $request_unique_id
     *
     * @return 
     */

    public function provider_profile($provider_unique_id , $request_unique_id = "") {

        $provider = Provider::where('unique_id' , $provider_unique_id)->first();

        if(count($provider) == 0) {

            return back()->with('flash_error' , tr('provider_not_found'));
        }

        $request_details = [];

        if($request_unique_id) {

            $request_details = Requests::where('unique_id' , $request_unique_id)->first();

        }

        $reviews = UserRating::where('provider_id' , $provider->id)->paginate(10);

        return view('user.jobs.provider-profile')->with('provider' , $provider)
                        ->with('request_details' , $request_details)
                        ->with('reviews' , $reviews);
    }

    /**
     * Chat page for user and provider for a particular request
     *
     *
     */

    public function chat($request_unique_id , $provider_unique_id) {

        $request_details = Requests::where('unique_id' , $request_unique_id)->first();

        $provider = Provider::where('unique_id' , $provider_unique_id)->first();

        $messages = [];

        if($request_details && $provider) {

            return view('user.jobs.chat')->with('request_details' , $request_details)->with('provider' , $provider);
        } else {
            return back()->with('flash_error' , tr('something_error'));
        }
    }

    /**
     * Function Name jobs_index
     * Create a job 
     *
     * @return view page with values
     */

    public function jobs_index(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
            'web' => 1,
            'is_web'=>DEVICE_WEB,
            'skip'=>0
        ]);

        // Check any ongoing ondemand and later requests 

        $ondemand_requests = $this->UserAPI->request_status_check_now($request)->getData();

        $later_requests = $this->UserAPI->request_status_check_later($request)->getData();

        $display = false;

        // If any ongoing requests means redirect to the ongoing page

        if($ondemand_requests->success || $later_requests->success) {

            if($ondemand_requests->data || $later_requests->data) {

                $display = true;
            }
        } 

        $drafts_response = $this->UserAPI->drafts($request)->getData();

        if($drafts_response->success) 
            $drafts = $drafts_response->data;

        $posted_response = $this->UserAPI->posted_requests($request)->getData();

        if($posted_response->success) 
            $posted_requests = $posted_response->data;

        $confirmed_response = $this->UserAPI->confirmed_requests($request)->getData();

        if($confirmed_response->success) 
            $confirmed_requests = $confirmed_response->data;

        $ongoing_response = $this->UserAPI->ongoing_requests($request)->getData();

        if($ongoing_response->success) 
            $ongoing_requests = $ongoing_response->data;

        $job_active = $request->active ? $request->active : 'posted-jobs';

        if($job_active == 'confirm-jobs' && confirmed_job_notifications() > 0) {

            $model = UserNotification::where('user_id', \Auth::user()->id)->where('status', DEFAULT_FALSE)
            ->where('job_status', 1)->get();

            foreach ($model as $key => $value) {

                $value->status = DEFAULT_TRUE;
                $value->save();

            }

        }

        if($job_active == 'ongoing-jobs' && ongoing_job_notifications() > 0) {

            $model = UserNotification::where('user_id', \Auth::user()->id)->where('status', DEFAULT_FALSE)
            ->where('job_status', 2)->get();

            foreach ($model as $key => $value) {

                $value->status = DEFAULT_TRUE;
                $value->save();

            }

        }

        return view('user.jobs.index')->with('posted_requests', $posted_requests)
                            ->with('drafts' , $drafts)
                            ->with('confirmed_requests' , $confirmed_requests)
                            ->with('ongoing_requests' , $ongoing_requests)
                            ->with('display',$display)
                            ->with('job_active',$job_active)
                            ->with('header_page' , "dashboard");

    }

    /**
     * Function Name Jobs_Create
     * Create a job 
     *
     * @return view page with values
     */
    public function jobs_create() {
        // Load categories
        $categories = getCategories();
       
        // Render view page 
        return view('user.jobs.create')->with('categories', $categories)->with('header_page' , "create_a_job");
    }

    /**
     * @method requests_update
     *
     * @uses To edit basic job details step for posted jobs 
     *
     * @created Vidhya R
     * 
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     */

    public function requests_update($request_unique_id) {

        // Load jobs model based on the auto increment id (Request id)
        $model = Requests::where("unique_id", $request_unique_id)->first(); 

        if($model) {
             // Render edit page view alog with send details of requests
            return view('user.jobs.edit')->with('request', $model);

        }

        return back()->with('flash_error', tr('something_error'));
    }

    /**
     * @method requests_update_save
     *
     * @uses To edit basic job details step for posted jobs 
     *
     * @created Vidhya R
     * 
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return view page with values
     */
    
    public function requests_update_save(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token
        ]);

        $response = $this->UserAPI->request_update($request)->getData();

        if($response->success) {
            return redirect()->route('user.jobs.view',$response->data->unique_id)->with('flash_success', $response->message);
        }

        return back()->with('flash_error', $response->error);
        
    }

    /**
     * @method requests_draft_edit()
     *
     * @uses used to edit the selected draft request
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return redirect view page or JSON Response
     */ 

    public function requests_draft_edit($request_id) {

        // Load jobs model based on the auto increment id (Request id)
        $model = TempRequests::where("id", $request_id)->first();
        // Load categories
        $categories = getCategories();
        // Load Sub categories
        $subCategories = getSubCategories($model); 

        if($model) {
            // Render edit page view alog with send details of requests
            return view('user.jobs.edit-draft-jobs')->with('request', $model)->with('categories', $categories)->with('subCategories', $subCategories);
        } else {
            return back()->with('flash_error', tr('something_error'));
        }
    }

    /**
     * Function name: requests_draft_delete()
     *
     * @uses used to delete the draft requests
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return redirect view page or JSON Response
     */ 

    public function requests_draft_delete($id) {

        $temp_request = TempRequests::find($id);

        if($temp_request) {
            $temp_request->forceDelete();
        } else {
            return back()->with('flash_error' , tr('something_error'));
        }

        return back()->with('flash_success' , tr('action_success'));
    }

    /**
     * Function name: requests_later_assign()
     *
     * @uses used to get selected request details
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return redirect view page or JSON Response
     */ 

    public function requests_view($request_unique_id) {

        $request_details = Requests::where('unique_id' ,$request_unique_id)->first();

        if($request_details) {

            $request_details->price_type = price_type($request_details->price_type);

            $request_payment_details = $request_details->payments ? $request_details->payments : new RequestPayment;

            return view('user.jobs.view')
                    ->with('request_details' , $request_details)
                    ->with('request_payment_details' , $request_payment_details);
            
        } else {

            return redirect()->route('user.dashboard')->with('flash_error', tr('something_error'));
        }
    }

    /**
     * Function name: requests_later_assign()
     *
     * @uses Assign the bidded provider to that Job or Request
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return redirect view page or JSON Response
     */ 

    public function requests_later_assign(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token
        ]);

        $response = $this->UserAPI->assign_request($request)->getData();

        if($response->success) {
            return back()->with('flash_success', $response->message);
        } else {
            return back()->with('flash_error', $response->error);
        }

        return back();
    }


    /**
     * Function name: requests_reject()
     *
     * @uses used to reject the selected request
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return redirect view page or JSON Response
     */ 

    public function requests_reject(Request $request) {

         $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token
        ]);

        $response = $this->UserAPI->reject_request($request)->getData();

        if($response->success) {
            return back()->with('flash_success', $response->message);
        } else {
            return back()->with('flash_error', $response->error);
        }

        return back();

    }

    /**
     * Function name: requests_cancel()
     *
     * @uses used to cancel the selected request
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return redirect view page or JSON Response
     */ 

    public function requests_cancel(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token
        ]);

        $request_details = Requests::find($request->request_id);

        // Ongoing or after confirmed request cancel

        if($request_details->provider_status) {

            $response = $this->UserAPI->cancel_request($request)->getData();

        // Request Now, before provider accept cancel

        } elseif(!$request_details->provider_status && $request_details->job_type == REQUEST_NOW) {

            $response = $this->UserAPI->waiting_request_cancel($request)->getData();

        } else {

            // Request Later, before provider accept cancel

            $response = $this->UserAPI->cancel_request($request)->getData();
        }

        if($response->success) {
            return redirect()->route('user.dashboard')->with('flash_success', $response->message);
        } else {
            return back()->with('flash_error', $response->error);
        }

        return back();

    }

    /**
     * Function name: refresh_single_request()
     *
     * @uses used to refresh the dashboard page
     *
     * @created Shobana C
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return redirect view page or JSON Response
     */ 

    public function refresh_single_request(Request $request) {

        $request_details = Requests::find($request->request_id);

        // Check the request is cancelled, if yes - redirect to dashboard

        if(in_array($request_details->status, [REQUEST_CANCELLED,REQUEST_NO_PROVIDER_AVAILABLE, REQUEST_TIME_EXCEED_CANCELLED])) {

            return redirect()->route('user.dashboard');
        }

        $response_array = ['success' => true , 'requests' => $request_details];

        return response()->json($response_array,200);
        
    }

    /**
     * Function name: requests_providers_list()
     *
     * @uses used get the list of providers for later request ( direct provider option)
     *
     * @created Vidhya R
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return render view page
     */ 

    public function requests_providers_list(Request $request) {
        
        // Call the common function of providers list from user API controller

        $response = $this->UserAPI->providers_list($request)->getData();

        if($response->success) {

            $viewfile = view('user.jobs.providers_list')->with('providers',$response);
            // return view file
            return $viewfile;

        } else {

            $error = $response->error;

            /** Handle Error Here **/

        }

    }

    /**
     * Function name: requests_payment()
     *
     * @uses After the provider request completion, make the payment for the request
     *
     * @created Shobana C
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return View Page
     */ 
    
    public function requests_payment(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token
        ]);

        $response = $this->UserAPI->requests_payment($request)->getData();

        if($response->success) {

            return back()->with('flash_success', $response->message);
        }

        return back()->with('flash_error', $response->error);
    }

    /**
     * Function name: requests_rating()
     *
     * @uses After the user payment completion, rate the provider for the selected request
     *
     * @created Shobana C
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return View Page
     */ 

    public function requests_rating(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token
        ]);

        $response = $this->UserAPI->requests_rating($request)->getData();

        if($response->success) {
            return back()->with('flash_success', $response->message);
        } else {
            return back()->with('flash_error', $response->error);
        }
        return back();
    }

    /**
     * Function name: requests_history()
     *
     * @uses Get all completed requests based on the logged in user
     *
     * @created Shobana C
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return JSON Response
     */  

    public function requests_history(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token
        ]);

        if($request->cancelled_jobs == YES) {
            $request->request->add([ 
                'cancelled_jobs' => $request->cancelled_jobs
            ]);
        }

        $response = $this->UserAPI->history($request)->getData();

        if($response->success) {
            return view('user.jobs.history')->with('requests' , $response->data)->with('header_page' , "history");
        } else {
            return back()->with('flash_error', $response->error);
        }
    
    }

    /**
     * Function name: requests_chat_history()
     *
     * @uses Get all messages for current request.
     *
     * @created Shobana C
     *
     * @updated Vidhya R
     *
     * @param integer id, string token
     *
     * @return JSON Response
     */  

    public function requests_chat_history(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
        ]);

        $response = $this->UserAPI->requests_chat_history($request)->getData();

        return response()->json($response->data);
    }

    /**
     * Function name: ajax_draft_requests()
     *
     * @uses drafts jobs list based on the users search/sort
     *
     * @created Shobana C
     *
     * @updated Vidhya R
     *
     * @param integer id
     *
     * @param integer token
     *
     * @return render view page 
     */    

    public function ajax_draft_requests(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
            'is_web' => DEVICE_WEB,
            'skip'=>$request->skip,
            's_k'=>$request->search_key,
        ]);
         
        $drafts_response = $this->UserAPI->drafts($request)->getData();

        if($drafts_response->success) 
            $drafts = $drafts_response->data;

        $view  = \View::make('user.jobs.ajax_request_responses.draft_requests')->with('drafts', $drafts)->render();

        return response()->json(['view'=>$view, 'length'=>count($drafts)]);

    }

    /**
     * Function name: ajax_posted_requests()
     *
     * @uses posted jobs list based on the users search/sort
     *
     * @created Shobana C
     *
     * @updated vidhya R
     *
     * @param integer id
     *
     * @param integer token
     *
     * @return render view page 
     */

    public function ajax_posted_requests(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
            'is_web' => DEVICE_WEB,
            'skip'=>$request->skip,
            's_k'=>$request->search_key,
        ]);

        $posted_response = $this->UserAPI->posted_requests($request)->getData();

        if($posted_response->success) 
            $posted_requests = $posted_response->data;

        $view  = \View::make('user.jobs.ajax_request_responses.posted_requests')->with('posted_requests', $posted_requests)->render();

        return response()->json(['view'=>$view, 'length'=>count($posted_requests)]);

    }

    /**
     * Function name: ajax_confirmed_requests()
     *
     * @uses confirmed jobs list based on the users search/sort
     *
     * @param integer id
     *
     * @param integer token
     *
     * @return render view page 
     */

    public function ajax_confirmed_requests(Request $request) {

         $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
            'is_web' => DEVICE_WEB,
            'skip'=>$request->skip,
            's_k'=>$request->search_key,
        ]);

        $confirmed_response = $this->UserAPI->confirmed_requests($request)->getData();

        if($confirmed_response->success) 
            $confirmed_requests = $confirmed_response->data;

        $view  = \View::make('user.jobs.ajax_request_responses.confirmed_requests')->with('confirmed_requests', $confirmed_requests)->render();

        return response()->json(['view'=>$view, 'length'=>count($confirmed_requests)]);

    }

    /**
     * Function name: ajax_ongoing_requests()
     *
     * @uses ongoing jobs list based on the users search/sort
     *
     * @created Shobana C
     *
     * @updated vidhya R
     *
     * @param integer id
     *
     * @param integer token
     *
     * @return render view page 
     */

    public function ajax_ongoing_requests(Request $request) {

         $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
            'is_web' => DEVICE_WEB,
            'skip'=>$request->skip,
            's_k'=>$request->search_key,
        ]);

        $ongoing_response = $this->UserAPI->ongoing_requests($request)->getData();

        if($ongoing_response->success) 
            $ongoing_requests = $ongoing_response->data;

        $view  = \View::make('user.jobs.ajax_request_responses.ongoing_requests')->with('ongoing_requests', $ongoing_requests)->render();

        return response()->json(['view'=>$view, 'length'=>count($ongoing_requests)]);

    }

    /**
     * Function name: ajax_cancelled_requests()
     *
     * @uses ongoing jobs list based on the users search/sort
     * 
     * @created Shobana C
     *
     * @updated vidhya R
     *
     * @param integer id
     *
     * @param integer token
     *
     * @return render view page 
     */

    public function ajax_cancelled_requests(Request $request) {

         $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
            'is_web' => DEVICE_WEB,
            'skip'=>$request->skip,
            's_k'=>$request->search_key,
        ]);

        $cancelled_response = $this->UserAPI->jobs_cancelled($request)->getData();

        if($cancelled_response->success) 
            $cancelled_requests = $cancelled_response->data;

        $view  = \View::make('user.jobs.ajax_request_responses.cancelled_requests')->with('cancelled_requests', $cancelled_requests)->render();

        return response()->json(['view'=>$view, 'length'=>count($cancelled_requests)]);

    }





    /**
     * Function name: see_all_notifications()
     *
     * @uses List of notifications
     *
     * @created Shobana C
     *
     * @updated vidhya R
     *
     * @param integer id
     *
     * @param integer token
     *
     * @return redirect to notifications list page 
     */

    public function see_all_notifications() {

        $model = UserNotification::select('user_notifications.id as notify_id', 
                        'user_notifications.notification as notification', 
                        'providers.picture as picture',
                        'providers.name as name',
                        'user_notifications.provider_id as provider_id',
                        'user_notifications.request_id as request_id',
                        'requests.unique_id as unique_id',
                        'user_notifications.created_at as created_at')
                    ->where('user_notifications.user_id', Auth::user()->id)
                    ->leftJoin('providers', 'providers.id', '=', 'user_notifications.provider_id')
                    ->leftJoin('requests', 'requests.id', '=', 'user_notifications.request_id')
                    ->orderBy('created_at', 'desc')->paginate(16);

        return view('user.jobs.notification')->with('model', $model);

    }

    /**
     * Function name: ajax_cancelled_requests()
     *
     * @uses cancelled jobs list based on the users search/ sort
     *
     * @created vidhya R
     *
     * @updated vidhya R
     * 
     * @param integer id
     *
     * @param integer token
     *
     * @return render view page 
     */

    public function cancelled_requests(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
            'device_token' => \Auth::user()->device_token,
            'is_web' => DEVICE_WEB,
            'skip'=>$request->skip,
            's_k'=>$request->search_key,
        ]);

        $cancelled_response = $this->UserAPI->cancelled_requests($request)->getData();

        if($cancelled_response->success) 
            $cancelled_requests = $cancelled_response->data;


        if($request->ajax()){

            $view  = \View::make('user.jobs.ajax_request_responses.cancelled_requests')->with('cancelled_requests', $cancelled_requests)->render();

            return response()->json(['view'=>$view, 'length'=>count($cancelled_requests)]);

        } else {
            return \View::make('user.jobs.cancelled')->with('cancelled_requests', $cancelled_requests);;
        }

    }

    /**
     * Function name: coupon_code_apply()
     *
     * @uses cancelled jobs list based on the users search/ sort
     *
     * @created vidhya R
     *
     * @updated vidhya R
     * 
     * @param integer id
     *
     * @param integer token
     *
     * @return render view page 
     */

    public function coupon_code_apply(Request $request) {

        $request->request->add([ 
            'id' => \Auth::user()->id,
            'token' => \Auth::user()->token,
        ]);

        $response = $this->UserAPI->coupon_code_apply($request)->getData();

        return response()->json($response);

    }

    /**
     * Function name: user_fav_providers_remove()
     *
     * @uses used to remove the user fav provider details
     *
     * @created Maheswari S
     *
     * @updated Maheswari S
     * 
     * @param integer provider_id
     *
     * @return success message
     */

    public function user_fav_providers_remove(Request $request) {

       if($request->user_fav_provider_id){

            $response = $this->UserAPI->fav_providers_remove($request)->getData();
            
            if($response->message) {

                $message = $response->message;

                return back()->with('flash_success',$message);

            } else {

                $message = $response->error;

                return back()->with('flash_error',$message);

            }
        } else {

            return back()->with('flash_error',tr('provider_not_found'));
        }

    }

    /**
     * @method  referral_code_signup()
     *
     * @uses signup user through referrals
     *
     * @created Vithya R
     *
     * @updated Anjana H
     *
     * @param string referral_code 
     *
     * @return redirect signup page
     */
    public function referrals_signup($referral_code){

        try {

            if(!$referral_code) {

                throw new Exception(tr('referral_code_invalid'), 101);
            }

            $check_referral_code =  UserReferrer::where('referral_code', $referral_code)->where('status', DEFAULT_TRUE)->first();

            if(!$check_referral_code) {

                throw new Exception(tr('referral_code_invalid'), 101);
            }

            $user_details = User::where('status', USER_APPROVED)->where('id', $check_referral_code->user_id)->first();

            if(!$user_details) {

                throw new Exception(tr('referral_code_invalid'), 101);
            }

            return redirect()->route('user.register.form', ['referral' => $referral_code]);

        } catch(Exception $e) {

            $error = $e->getMessage();

            $error_code = $e->getCode();

            return redirect(route('user.register.form'))->with('flash_error', $error);
        }

    }

    /**
     * @method referrals()
     *
     * @uses signup user through referrals
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param string referral_code 
     *
     * @return redirect signup page
     */
    public function referrals(Request $request) {

        try {

            $user_details =  Auth::user();

            $user_referrer_details = UserReferrer::where('user_id', $user_details->id)->first();

            if(!$user_referrer_details) {

                $user_referrer_details = new UserReferrer;

                $user_referrer_details->user_id = $user_details->id;

                $user_referrer_details->referral_code = uniqid();

                $user_referrer_details->save();

            }

            $referrals = Referral::where('parent_user_id', $user_details->id)->orderBy('created_at', 'desc')->get();
            
            foreach ($referrals as $key => $referral_details) {
                
                // $referral_user_details = $referral_details->userDetails;
                
                if($referral_user_details = $referral_details->userDetails) {

                    $referral_details->username = $referral_user_details->name ? : "";

                    $referral_details->picture = $referral_user_details->picture ? : "";

                }            
            }

            return view('user.account.referrals_index')
                    ->with('referrals', $referrals)
                    ->with('user_referrer_details', $user_referrer_details);

        } catch(Exception $e) {

            $error = $e->getMessage();

            $error_code = $e->getCode();

            return redirect()->back()->with('flash_error', $error);
        }

    }

    /**
     * @method referrals_view()
     *
     * @uses get the subscription & PPV details for selected referral user
     *
     * @created Anjana H
     *
     * @updated Anjana H
     *
     * @param integer $user_id
     *
     * @return response of success / failure message.
     */

    public function referrals_view(Request $request) {
        
        $user_details = User::find($request->user_id);

        if(!$user_details) {
            
            return back()->with('flash_error', tr('user_not_found'));
        }

        $user_referrer_details = UserReferrer::where('user_id', $request->parent_user_id)->first();

        return view('user.account.referrals_view')
                    ->with('page', 'users')
                    ->with('sub_page', 'view-user')
                    ->with('user_details', $user_details)
                    ->with('user_referrer_details', $user_referrer_details);

    } 
}










