<?php

namespace App;

use Illuminate\Database\Eloquent\Model;

use Illuminate\Database\Eloquent\SoftDeletes;

use Illuminate\Foundation\Auth\User as Authenticatable;

use App\Helpers\Helper;

use Setting;

class Provider extends Authenticatable
{

    use SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name', 'email', 'password','is_activated','login_by','device_type','picture','register_type','login_status',
        'verification_code', 'verification_code_expiry' ,'is_verified'
    ];


    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    
    protected $dates = ['deleted_at'];

     /**
     * Scope a query to only include active users.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeProviderResponse($query , $id)
    {
        return $query->where('providers.id' , $id)->select('providers.id as provider_id', 
                'providers.name as provider_name' , 
                'providers.email as email' , 
                'providers.social_unique_id as social_unique_id' , 
                'providers.unique_id as unique_id' , 
                'providers.token as token' , 
                'providers.picture as provider_picture' , 
                'providers.promo_video as promo_video' , 
                'providers.login_by' , 
                'providers.device_type' , 
                'providers.description' , 
                'providers.mobile' , 
                'providers.latitude' , 
                'providers.longitude' , 
                'providers.user_id' , 
                'providers.status' , 
                'providers.is_available' ,
                'providers.is_approved',
                'providers.is_verified',
                'providers.provider_type'
                );
    }

    public function providerGalleries()
    {
        return $this->hasMany('App\ProviderGallery');
    }

    public function providerRequests()
    {
        return $this->hasMany('App\Requests');
    }

    public function providerAbusePorviders()
    {
        return $this->hasMany('App\AbuseProvider');
    }

    public function providerRequestMeta()
    {
        return $this->hasMany('App\RequestsMeta' , 'provider_id');
    }

    public function providerServices() {
        return $this->hasMany('App\ProviderService' , 'provider_id');
    }

    public function providerDocuments() {
        return $this->hasMany('App\ProviderDocument' , 'provider_id');
    }

    public function providerChats() {
        return $this->hasMany('App\ChatMessage');
    }

    public function userRatings() {
        return $this->hasMany('App\UserRating')->orderby('rating' , 'desc');
    }

    public function providerRatings() {
        return $this->hasMany('App\ProviderRating');
    }

    public function completed() {
        return $this->hasMany('App\Requests' , 'provider_id')->where('status' , REQUEST_COMPLETED);
    }

    public function providerPayment() {
        return $this->hasOne('App\ProviderPayment');
    }

    public function providerRedeemRequests() {
        return $this->hasMany('App\RedeemRequest')->orderBy('created_at' , 'desc');
    }

    public function providerSubscriptionPayment() {
        return $this->hasMany('App\ProviderSubscriptionPayment', 'provider_id');
    }

    public function providerCards() {
        return $this->hasMany('App\ProviderCard');
    }

    public static function boot() {
    	
        // execute the parent's boot method 

        parent::boot();

        //delete your related models here, for example
        
        static::deleting(function($data) {

            Helper::delete_avatar('uploads/providers' , $data->picture);

            if(count($data->providerRequests) > 0) {

                foreach($data->providerRequests as $providerRequest) {

                    $providerRequest->delete();
                } 
            }

            if(count($data->providerAbusePorviders) > 0) {            

                foreach($data->providerAbusePorviders as $providerAbusePorvider) {
                    $providerAbusePorvider->delete();
                } 
            }

            if(count($data->providerRequestMeta) > 0) {

                foreach ($data->providerRequestMeta as $key => $requestMeta) {
                    $requestMeta->delete();
                }
            }

            if(count($data->providerServices) > 0) {

                foreach($data->providerServices as $providerService) {

                    $providerService->delete();
                } 

            }

            if(count($data->providerDocuments) > 0) {

                foreach($data->providerDocuments as $providerDocument) {

                    $providerDocument->delete();
                } 
            }

            if(count($data->providerChats) > 0) {

                foreach($data->providerChats as $providerChat) {

                    $providerChat->delete();
                } 
            }

            if(count($data->userRatings) > 0) {

                foreach($data->userRatings as $userRating) {
                    $userRating->delete();
                } 

            }

            if(count($data->providerRatings) > 0) {

                foreach($data->providerRatings as $providerRating) {

                    $providerRating->delete();
                }
            }

            if(count($data->providerPayment) > 0) {

                $data->providerPayment->delete();
            }

            if(count($data->providerRedeemRequests) > 0) {

                foreach($data->providerRedeemRequests as $providerRedeemRequest) {

                    $providerRedeemRequest->delete();
                }

            }

            if(count($data->providerGalleries) > 0) {

                foreach($data->providerGalleries as $provider_gallery_details) {

                    Helper::delete_avatar('uploads/provider-galleries' , $provider_gallery_details->picture);

                    $provider_gallery_details->delete();


                }

            }
        
        }); 
        
        static::creating(function ($model) {

            $model->generateEmailCode();

            $model->attributes['is_approved'] = PROVIDER_STATUS_PENDING;

            $model->attributes['status'] = true;

            $model->attributes['is_available'] = PROVIDER_AVAILABLE;

            $model->attributes['step'] = PROVIDER_STEP_INITIAL;
        });


        static::created(function($model) {

            $model->attributes['unique_id'] = "PID"."-".$model->id."-".routefreestring($model->name)."-".uniqid();

            $model->attributes['is_verified'] = PROVIDER_EMAIL_NOT_VERIFIED;

            if(Setting::get('email_verify_control') == EMAIL_VERIFICATION_OFF || in_array($model->login_by , ['facebook' , 'google'])) {

                $model->attributes['is_verified'] = PROVIDER_EMAIL_VERIFIED;

            }

            $model->save();
        });
    }



    /**
     * Generates Token and Token Expiry
     * 
     * @return bool returns true if successful. false on failure.
     */

    protected function generateEmailCode() {

        $this->attributes['verification_code'] = Helper::generate_email_code();

        $this->attributes['verification_code_expiry'] = Helper::generate_email_expiry();

        // $this->attributes['is_verified'] = 0;

        return true;
    }

}
