<?php

namespace App\Repositories;

use App\Repositories\ProviderRepository as ProviderRepo;
use App\Repositories\AdminRepository as AdminRepo;
use App\Repositories\UserRepository as UserRepo;
use App\Helpers\Helper;
use Illuminate\Http\Request;
use Validator;
use App\User;
use App\Provider;
use Hash;
use Log;

use Setting;

class CommonRepository {

	public static function profile_validation($data = [], &$errors = []) {

		$validator = Validator::make( $data,array(
                'id' => 'required|exists:admins,id',
                'name' => 'max:255',
                'email' => 'email|max:255',
                'mobile' => 'digits_between:4,16',
                'address' => 'max:300',
            )
        );
        
	    if($validator->fails()) {
	    	// The errors will be return using &$errors [Passing By References]
	        $errors = implode(',', $validator->messages()->all());
	        return false;
	    }

	    return true;
	
	}

	/**
	 * Usage : Register api - validation for the basic register fields 
	 *
	 */

	public static function basic_validation($data = [], &$errors = []) {

		$validator = Validator::make( $data,array(
                'device_type' => 'required|in:'.DEVICE_ANDROID.','.DEVICE_IOS.','.DEVICE_WEB,
                'device_token' => 'required',
                'login_by' => 'required|in:manual,facebook,google',
            )
        );
        
	    if($validator->fails()) {
	        $errors = implode(',', $validator->messages()->all());
	        return false;
	    }

	    return true;

	}

	/**
	 * Usage : Register api - validation for the social register or login 
	 *
	 */

	public static function social_validation($data = [] , &$errors = []) {

		$validator = Validator::make( $data,array(
                'social_unique_id' => 'required',
                'name' => 'required|min:2|max:150',
                'email' => 'required|email|max:255',
                'mobile' => 'digits_between:4,16',
                'picture' => '',
                'gender' => 'in:male,female,others',
            )
        );
        
	    if($validator->fails()) {
	        $errors = implode(',', $validator->messages()->all());
	        return false;
	    }

	    return true;

	}

	/**
	 * Usage : Register api - validation for the manual register fields 
	 *
	 */

	public static function manual_validation($data = [] , &$errors = []) {

		$validator = Validator::make( $data,array(
                'name' => 'required|min:2|max:150',
                'email' => 'required|email',
                'password' => 'required|min:6',
                'mobile' => 'digits_between:4,16',
                'picture' => 'mimes:jpeg,jpg,bmp,png',
            )
        );
        
	    if($validator->fails()) {
	        $errors = implode(',', $validator->messages()->all());
	        return false;
	    }

	    return true;

	}

	/**
	 * Usage : Login api - validation for the manual login fields 
	 *
	 */

	public static function login_validation($data = [] , &$errors = [] , $table = "users") {

		$validator = Validator::make( $data,[
                'email' => 'required|email|exists:'.$table.',email',
                'password' => 'required|min:6',
            ]
        );
        
	    if($validator->fails()) {
	        $errors = implode(',', $validator->messages()->all());
	        return false;
	    }

	    return true;

	}

	/**
	 * Usage : Login api - validation for the manual login fields 
	 *
	 */

	public static function email_validation($data = [] , &$errors = [] , $table = "users") {

		$validator = Validator::make( $data,[
                'email' => 'required|email|exists:'.$table.',email',
            ],
            [
            	'exists' => tr('email_id_not_found')
            ]
        );
        
	    if($validator->fails()) {
	        $errors = implode(',', $validator->messages()->all());
	        return false;
	    }

	    return true;

	}

	/**
	 *
	 */

	public static function change_password_validation($data = [] , &$errors = [] , $table = "users") {

		$validator = Validator::make( $data,[
                'password' => 'required|confirmed|min:6',
                'old_password' => 'required|min:6',
            ]
        );
        
	    if($validator->fails()) {
	    	
	        $errors = implode(',', $validator->messages()->all());
	        
	        return false;
	    }

	    return true;

	}

	public static function admin_add_validation($data = [] , &$errors = [] , $model) {

		if($model == USER)
			$email_exist_check = 'users';
		else
			$email_exist_check = 'providers';
		
		$validator = Validator::make( $data,array(
                'name' => 'required|min:2|max:150',
                'email' => 'required|email|max:255|unique:'.$email_exist_check.',email,NULL,id,deleted_at,NULL',
                'mobile' => 'digits_between:4,16',
                'password' => 'required|min:6|confirmed',
                'picture' => 'mimes:jpeg,bmp,png',
                'video' => 'mimes:mp4|max:20480',
                'description' =>'max:255'
            )
        );
        
	    if($validator->fails()) {

	        $errors = implode(',', $validator->messages()->all());
	        
	        return false;
	    }

	    return true;
	}

	public static function admin_edit_validation($data = [] , &$errors = [], $model, $id) {

		if($model == USER)

			$email_exist_check = 'users';

		else
			
			$email_exist_check = 'providers';

		$validator = Validator::make( $data,array(
			
                'name' => 'required|min:2|max:150',

                'email' => 'required|email|max:255|unique:'.$email_exist_check.',email,'.$id.',id,deleted_at,NULL',
                'picture' => 'mimes:jpeg,bmp,png',

                'mobile' => 'digits_between:4,16',

                'price_per_hour'=> 'numeric|min:0.1',

                'video' => 'mimes:mp4|max:20480',

                'description' =>'max:255'
            )
        );
        
	    if($validator->fails()) {
	        $errors = implode(',', $validator->messages()->all());
	        return false;
	    }

	    return true;
	}

	public static function admin_category_add_validation($data = [] , &$errors = [] , $model) {

		if($model == CATEGORY){
			$validator = Validator::make( $data,array(
                'name' => 'required|unique:categories,name|min:2|max:150',
                'description' => 'required',
                'picture' => 'required|mimes:jpeg,jpg,bmp,png'
            ));
		}else{
			$validator = Validator::make( $data,array(
                'name' => 'required|unique:sub_categories,name|min:2|max:150',
                'category_id' => 'required|exists:categories,id',
                'description' => 'required',
                'picture' => 'required|mimes:jpeg,jpg,bmp,png',
                'price'=> 'required|numeric|min:0.1',
            ));
		}
        
	    if($validator->fails()) {
	        $errors = implode(',', $validator->messages()->all());
	        return false;
	    }

	    return true;
	}

	public static function admin_category_edit_validation($data = [] , &$errors = [], $model,$id) {

		if($model == CATEGORY) {

			$validator = Validator::make( $data,array(
                'name' => 'required|min:2|max:150|unique:categories,name,'.$id,
                'description' => 'required',
                'picture' => 'mimes:jpeg,jpg,bmp,png',
            ));
		} else {

			$validator = Validator::make( $data,array(
                'name' => 'required|min:2|max:150|unique:sub_categories,name,'.$id,
                'category_id' => 'required|exists:categories,id',
                'description' => 'required',
                'picture' => 'mimes:jpeg,jpg,bmp,png',
                'price'=> 'required|numeric|min:0.1',
            ),
            [
            	'price.numeric' => tr('price_validation')
            ]
		);
		}
	    if($validator->fails()) {
	        $errors = implode(',', $validator->messages()->all());
	        return false;
	    }

	    return true;
	}

	/**
	 * While user register, make the user as provider.
	 * 
	 * @input : User model values
	 *
	 * @return : 
	 */

	public static function become_a_provider($user) {

		// Check this Become a User option is enabled by admin

		if(Setting::get('become_a_provider')) {

			$check_provider = $provider = Provider::where('email' , $user->email)->first();

			if(!$check_provider) {

				$provider = new Provider;
				$provider->name = $user->name ? $user->name : "";
				$provider->description = $user->description ? $user->description : "";
				$provider->picture = $user->picture ? $user->picture : asset('placeholder.png');
				$provider->email = $user->email ? $user->email : "";
				$provider->password = $user->password ? $user->password : "";
				$provider->token = $user->token ? $user->token : "";
				$provider->token_expiry = $user->token_expiry ? $user->token_expiry : "";
				$provider->gender = $user->gender ? $user->gender : "male";
				$provider->device_type = $user->device_type ? $user->device_type : "";
				$provider->login_by = $user->login_by ? $user->login_by :  "";
				$provider->user_id = $user->id ? $user->id : "";
        		$provider->login_status = "user";
        		$provider->register_type = "user";
				$provider->save();

				$provider->unique_id = encrypt($provider->id);
				$provider->save();
			
			} 

			$user->provider_id = $provider->id;
			$user->login_status = 'user';
			$user->save();

			$response_array = ['success' => true , 'data' => $provider];

		}  else {
			$response_array = ['success' => false , 'error' => Helper::error_message(147) , 'error_code' => 147];
		}

		return $response_array;
	
	}

	/**
	 * While provider register, make the provider as user.
	 * 
	 * @input : $request = Provider model values
	 *
	 * @return : 
	 */

	public static function become_a_user($provider) {

		// check this Become a provider option is enabled by admin

		if(Setting::get('become_a_user')) {

			// Check already user exists with the same details

			$check_data = User::where('email' , $provider->email)->first();

			if(!$check_data) {

				$data = new User;
				$data->name = $provider->name ?  $provider->name : "";
				$data->description = $provider->description ? $provider->description : "";
				$data->picture = $provider->picture ? $provider->picture : asset('placeholder.png');
				$data->email = $provider->email ? $provider->email : "";
				$data->password = $provider->password ? $provider->password : "";
				$data->token = $provider->token ? $provider->token : "";
				$data->token_expiry = $provider->token_expiry ? $provider->token_expiry : "";
				$data->gender = $provider->gender ? $provider->gender : "";
				$data->device_type = $provider->device_type ? $provider->device_type : "";
				$data->login_by = $provider->login_by ? $provider->login_by : "";
				$data->provider_id = $provider->id ? $provider->id : "";
				$data->payment_mode = 'cod';
				$provider->login_status = "provider";
        		$provider->register_type = "provider";
				$data->save();

				$data->unique_id = encrypt($data->id);
				$data->save();
			
			} 

			if($data)
				$provider->user_id = $data->id;

			$provider->login_status = 'provider';
			$provider->save();

			$response_array = ['success' => true , 'data' => $provider];

		} else {
			$response_array = ['success' => false , 'error' => Helper::error_message(147) , 'error_code' => 147];
		}

		return $response_array;
	}

}