<?php

namespace App\Repositories;

use App\Repositories\CommonRepository as CommonRepo;

use App\Provider;
use App\ProviderService; 
use App\ProviderDocument;
use App\Category;
use App\SubCategory;

use App\Requests;
use App\RequestsMeta;

use App\Helpers\Helper;
use Validator;
use App\User;
use Hash;
use Log;
use Setting;

class ProviderRepository {

    public static function request_validation($data = [] , &$errors = [] , $provider , $request_cancel = "") {

        if($request_cancel) {

            $validator = Validator::make($data,
                array(
                    'request_id' => 'required|integer|exists:requests,id',
                ),
                array(
                    'exists' => 'The :attribute not found'
                )
            );

        } else {
            $validator = Validator::make($data,
                array(
                    'request_id' => 'required|integer|exists:requests,id,provider_id,'.$provider->id,
                ),
                array(
                    'exists' => 'The :attribute doesn\'t belong to provider:'.$provider->id
                )
            );
        }

        if($validator->fails()) {

            $errors = implode(',' ,$validator->messages()->all());

            return false;

        }

        return true;

    }

	public static function all() {

		return Provider::orderBy('created_at' , 'desc')->get();
		
	}

	public static function login($request) {

        $provider_details = Provider::where('email', '=', $request->email)->first();

        if(!$provider_details) {
                    
            $response = array('success' => false , 'error' => Helper::error_message(504) , 'error_code' => 504);
            
            return $response;

            // return response()->json($response, 200);
        }

        if($provider_details->is_approved == PROVIDER_DECLINED) {
                    
            $response = array('success' => false , 'error' => Helper::error_message(502) , 'error_code' => 502);
            
            return $response;
            
            // return response()->json($response, 200);
        }

        if($provider_details->is_verified == PROVIDER_EMAIL_NOT_VERIFIED) {

            if(Setting::get('email_verify_control') == EMAIL_VERIFICATION_ON && !in_array($provider_details->login_by, ['facebook' , 'google'])) {

                // Check the verification code expiry

                Helper::check_email_verification("" , $provider_details, $error, PROVIDER);
                
                $response = array('success' => false , 'error' => Helper::error_message(503) , 'error_code' => 503);
                
                return $response;
                
                // return response()->json($response, 200);
            }
        }

        // Validate the user credentials

        if(\Hash::check($request->password, $provider_details->password)) {

        	// Generate new tokens
            $provider_details->token = Helper::generate_token();
            $provider_details->token_expiry = Helper::generate_token_expiry();

            // Save device details
            $provider_details->device_token = $request->device_token;
            $provider_details->device_type = $request->device_type;
            $provider_details->login_by = $request->login_by;
            $provider_details->timezone = $request->has('timezone') ? $request->timezone : "";

            $provider_details->save();

            $data = Provider::providerResponse($provider_details->id)->first();

            if($data) {
                
                $data = $data->toArray();

                $data['is_appstore_upload'] = Setting::get('is_appstore_upload') ?: 0;

            }

            $response_array = ['success' => true , 'message' => Helper::success_message(202),'data' => $data , 'code' => 202];

        } else {
            
            $response_array = array( 'success' => false, 'error' => tr('invalid_email_password'), 'error_code' => 105 );
        }

        return $response_array;
	}

    /**
     * Used to send mail to requested mail ID
     *
     *
     */

	public static function forgot_password($data = [] , &$errors = []) {

		$provider_details = Provider::where('email' , $data->email)->first();

		if($provider_details) {

            $provider_data = $provider_details->toArray();

            if(!envfile('MAIL_USERNAME') && !envfile('MAIL_PASSWORD')) {

                $response_array = ['success' => false , 'error' => Helper::error_message(505) , 'error_code' => 505];

                return $response_array;
            }

            // Check the provider approval status

            if($provider_details->is_approved == PROVIDER_DECLINED) {
                    
                $response = array('success' => false , 'error' => Helper::error_message(502) , 'error_code' => 502);
                return response()->json($response, 200);
            }

            // Check the provider email verfication status

            if($provider_details->is_verified == PROVIDER_EMAIL_NOT_VERIFIED) {

                if(Setting::get('email_verify_control') == EMAIL_VERIFICATION_ON && !in_array($provider_details->login_by, ['facebook' , 'google'])) {

                    // Check the verification code expiry

                    Helper::check_email_verification("" , $provider_details, $error, PROVIDER);
                    
                    $response = array('success' => false , 'error' => Helper::error_message(503) , 'error_code' => 503);
                    
                    return response()->json($response, 200);

                }
           
            }

			$new_password = Helper::generate_password();

	        $provider_details->password = \Hash::make($new_password);

	        $email_data = array();

	        $subject = tr('provider_forgot_email_title' , Setting::get('site_name'));

	        $page = "emails.provider.forgot-password";

            $email_data['username'] = $provider_details->name;

            $email_data['new_password'] = $new_password;

            // \Log::info("EMAIL DATA".print_r($email_data , true));

	        $email_send = Helper::send_email($page,$subject,$provider_details->email,$email_data);

            $message = "Thanks! Please check $provider_details->email for a link to reset your password.";

            $response_array = ['success' => true , 'message' => $message , 'code' => 205];

	        $provider_details->save();

		} else {
			$response_array = ['success' => false , 'error' => Helper::error_message(153) , 'error_code' => 153];
		}
        
        return $response_array;
	
	}

    /**
     * Function name: store()
     *
     * @uses store the provider details
     *
     * @created vidhya R
     *
     * @created vidhya R
     *
     * @param form data
     *
     * @return 
     */

	public static function store($request) {

		$create_user = new Provider;

		$create_user->name = $request->has('name') ? $request->name : "";

		$create_user->email = $request->has('email') ? $request->email : "";

		$create_user->password = $request->has('password') ? \Hash::make($request->password) : "";

		$create_user->gender = $request->has('gender') ? $request->gender : "male";

		$create_user->mobile = $request->has('mobile') ? $request->mobile : "";

		$create_user->token = Helper::generate_token();

        $create_user->token_expiry = Helper::generate_token_expiry();

        $check_device_exist = Provider::where('device_token', $request->device_token)->first();

        if($check_device_exist) {

            $check_device_exist->device_token = "";

            $check_device_exist->save();
        }

        $create_user->device_token = $request->has('device_token') ? $request->device_token : "";

        $create_user->device_type = $request->has('device_type') ? $request->device_type : "";

        $create_user->login_by = $request->has('login_by') ? $request->login_by : "manual";

        $create_user->login_status = 'provider';

        $create_user->register_type = $request->device_type ? $request->device_type : "web";

		$create_user->social_unique_id = $request->has('social_unique_id') ? $request->social_unique_id : "";

        $create_user->timezone = $request->has('timezone') ? $request->timezone : "";

        $create_user->picture = asset('placeholder.png');

        // Upload picture
        if($request->login_by == "manual") {

            if($request->hasFile('picture')) {

                $create_user->picture = Helper::upload_avatar('uploads/providers',$request->file('picture'));
            }

        } else {

            if($request->has('picture')) {

                $create_user->picture = $request->picture;

            }
        }
        
        $create_user->save();

        // Create Services for registered provider

        self::provider_services_create($create_user->id, 'new');

        CommonRepo::become_a_user($create_user);

        // Send welcome email to the new provider

        if($admin_mail = get_admin_mail()) {

            $subject = tr('new_provider_signup');

            $page = "emails.admin.provider-welcome";

            $email = $admin_mail->email;
            
            Helper::send_email($page,$subject,$email,$create_user);

        }

        $subject = tr('new_provider_signup');

        $page = "emails.provider.welcome";

        $email = $create_user->email;
        
        Helper::send_email($page,$subject,$email,$create_user);

        $response = [];

        if($create_user) {
            
            $response = $create_user->providerResponse($create_user->id)->first()->toArray();
        }

        return $response;

	}

    /**
     * Used to update the provider details
     *
     *
     */

	public static function update($request = [] , $id) {

        if($request->id) {
            $id = $request->id;
        }

		$data = Provider::find($id);
        
        $response = [];

        if($data) {

            if($request->has('name')) {
                $data->name = $request->name;
            }

            if($request->has('email')) {
                $data->email = $request->email;
            }

            if($request->has('mobile')) {
                $data->mobile = $request->mobile;
            }

            if($request->has('gender')) {
                $data->gender = $request->gender;
            }

            if($request->has('description')) {
                $data->description = $request->description;
            }
            if($request->has('timezone')) {
                $data->timezone = $request->timezone;
            }

            $check_device_exist = Provider::where('device_token', $request->device_token)->first();

            if($check_device_exist) {

                $check_device_exist->device_token = "";

                $check_device_exist->save();
            }

            $data->device_token = $request->has('device_token') ? $request->device_token : "";

            $data->device_type = $request->has('device_type') ? $request->device_type : $data->device_type;

            $data->login_by = $request->has('login_by') ? $request->login_by : $data->login_by;
            
            $data->login_status = 'provider';

            // For Social login we need to update the token and token expiry 

            if($request->is_social_login) {

                $data->token = Helper::generate_token();

                $data->token_expiry = Helper::generate_token_expiry();
            }

            // Upload picture
            
            if ($request->hasFile('picture')) {

                Helper::delete_avatar('uploads/providers',$data->picture); // Delete the old pic

                $data->picture = Helper::upload_avatar('uploads/providers',$request->file('picture'));
            }

            // $data->promo_video = $request->has('promo_video') ? Helper::youtube_vimeo_link_convert($request->promo_video) : $data->promo_video;

            if($request->has('promo_video')) {

                $data->promo_video = Helper::youtube_vimeo_link_convert($request->promo_video);

            } else {
                
                $data->promo_video = "";
            }

            $data->save();

            $response = $data->providerResponse($data->id)->first()->toArray();
        }

        return $response;

	}

    public static function provider_services_create($provider_id , $status = "new") {
        
        // Status = new provider  or update provider

        $categories = Category::where('status' , 1)->get();

        if($categories) {

            foreach ($categories as $key => $category) {

                if($sub_categories = $category->subcategory) {

                    foreach ($sub_categories as $key => $sub_category) {

                        $check_provider_services = ProviderService::where('category_id' , $category->id)->where('sub_category_id' , $sub_category->id)->where('provider_id' , $provider_id)->first();

                        if(!$check_provider_services) {

                            $provider_service = new ProviderService;
                            $provider_service->provider_id = $provider_id;
                            $provider_service->category_id = $category->id;
                            $provider_service->sub_category_id = $sub_category->id;
                            $provider_service->status = false;

                            $provider_service->save();
                        }
                    }
                }

            }
        }

    }

    /** 
    * Function : provider_services()
    *
    * @param : $request
    *
    * @usage: Only used to get the provider service ids, while request searching
    *
    */

    public static function provider_services($request) {
        
        $provider_services = ProviderService::where('provider_id' , $request->id)->where('status' , 1)->get();

        $result = [];

        if($provider_services) {

            foreach ($provider_services as $key => $provider_service) {
                $category_ids[] = $provider_service->category_id;
                $sub_category_ids[] = $provider_service->sub_category_id;
            }

            $category_ids = implode(',', $category_ids);
            $sub_category_ids = implode(',', $sub_category_ids);

            $result = ['success' => true , 'category_ids' => $category_ids , 'sub_category_ids' => $sub_category_ids];

        }

        return $result;
    
    }

    /**
     * Check the provider have any ongoing or confirmed requests
     *
     *
     */

    public static function check_ongoing_confirmed_requests($provider_id) {

        $check_status = [REQUEST_ONGOING, REQUEST_COMPLETE_PENDING , WAITING_PAYMENT_CONFIRM , REQUEST_RATING];

        $provider_status = [PROVIDER_NONE , PROVIDER_ASSIGNED ,PROVIDER_RATED];

        return Requests::where('provider_id' , $provider_id)->whereIn('status' , $check_status)->whereNotIn('provider_status' , $provider_status)->get();
    }

    /**
     * Used to get provider bidded requests ids
     *
     * @param : provide id
     * 
     * @return : array
     */

    public static function get_bidded_request_ids($provider_id) {

        $requests = RequestsMeta::where('provider_id' , $provider_id)->where('status' , REQUEST_META_BID)->select('request_id')->get();

        $ids = [];

        if($requests) {
            foreach ($requests as $key => $value) {
                $ids[] = $value->request_id;
            }
        }

        return $ids;
    }

}
