<?php

namespace App\Repositories;

use App\Repositories\CommonRepository as CommonRepo;

use App\Jobs\sendPushNotification;

use App\Category;

use App\SubCategory;

use App\Requests;

use App\TempRequests;

use App\Provider;

use App\ProviderService;

use App\RequestsMeta;

use App\RequestPayment;

use App\UserRating;

use App\NotificationTemplate;

use App\ProviderRating;

use App\User;

use App\Helpers\Helper;

use App\EmailTemplate;

use Validator;

use Setting;

use App\SmsTemplate;

use Log;

use App\ProviderNotification;

use App\UserFavProvider;

use Exception;

class RequestRepository {

    /** 
    * Used to check whether the job type and request type are eanbled or not
    *
    */

    public static function check_request_type_settings($job_type , $request_type) {

        $allow = false;

        if($job_type == REQUEST_NOW) {

            if(Setting::get('request_now')) {
                $allow = true;
            }
        } else {

            if($request_type == REQUEST_TYPE_LOCATION) {
                if(Setting::get('request_type_location')) {
                    $allow = true;
                }
            } elseif($request_type == REQUEST_TYPE_DIRECT) {

                if(Setting::get('request_type_direct')) {
                    $allow = true;
                }
            } else {
                if(Setting::get('request_type_category')) {
                    $allow = true;
                }
            }
        }

        return $allow;
    }

    /**
     * Used to validate the request form field before creating request
     * 
     * @param : Form data
     * 
     * @return : Boolean
     *
     */

    public static function basic_validation($data = [], &$errors = []) {

        $date = date('Y-m-d H:i:s' , strtotime("-60 minutes"));

        if(date('Y-m-d H:i:s' , strtotime($data['request_date'])) < $date) {

            $date = date('d-M-Y H:i' , strtotime($date));

            $errors = Helper::error_message(155)." ".$date;

            return false;

            // $response_array = ['success' => false , 'error' => Helper::error_message(155) , 'error_code' => 155];
       
        } else {

            $validator = Validator::make( $data,array(
                // 'request_date' => 'required|date|after:'.$date,
                's_latitude' => 'required',
                's_longitude' => 'required',
                's_address' => 'required',
                'name' => 'required',
                'description' => 'required' ,
                'note' => '',
                'user_price' => '',
                'temp_request_id' => 'required|exists:temp_requests,id'
                )
            );

            if($validator->fails()) {
                // The errors will be return using &$errors [Passing By References]
                $errors = implode(',', $validator->messages()->all());
                return false;
            }
        
        }

        return true;
    
    }

    /**
     * Used to check the selected provider availablity
     * 
     * @param :  integer provider_id
     * 
     * @return : Boolean
     *
     */

    public static function check_provider_availability($provider_id) {
        
        if($provider_id) {

            $provider = Provider::where('id' , $provider_id)->where('is_available' , PROVIDER_AVAILABLE)
                                    ->where('is_approved' , PROVIDER_APPROVED)
                                    ->where('waiting_to_respond' , WAITING_TO_RESPOND_NORMAL)
                                    ->count();
            if($provider) {
                return true;
            } else {
                return false;
            }

        } else {
            return false;
        }
    
    }

    /**
     * Used to check the user have any requests on selected request date
     * 
     * @param :  Object $request
     * 
     * @return : Boolean
     *
     */

    public static function request_date_validation($request) {

        $start_date = Helper::add_date($request->request_date,"-1");  

        $end_date = Helper::add_date($request->request_date,"+2");

        // Check any ongoing requests on the requested time

        $list_status = [REQUEST_CREATED, REQUEST_WAITING , REQUEST_PROVIDER_ASSIGNED , REQUEST_ONGOING ,WAITING_PAYMENT_CONFIRM,REQUEST_COMPLETE_PENDING, REQUEST_RATING];

        $check_request = Requests::where('user_id' , $request->id)
                            ->whereBetween('request_date' ,[$start_date,$end_date])
                            ->whereIn('status' , $list_status)
                            ->count();
        if($check_request) {

            $response_array = ['success' => false , 'error' => Helper::error_message(110) ,  'error_code' => 110];

            return $response_array;

        } else {
            return true;
        }

    }

    public static function requests_exists($data = [] , &$errors = []) {

        $validator = Validator::make($data , [
                'request_id' => 'required|exists:requests,id',
            ]);

        if($validator->fails()) {

            $errors = implode(',' ,$validator->messages()->all());

            return false;

        }

        return true;

    }

    /**
     * Used to check the user have any pending payment
     * 
     * @param :  integer $user_id
     * 
     * @return : Boolean
     *
     */

    public static function check_payment_pending($user_id) {

        $status = [REQUEST_COMPLETE_PENDING , WAITING_PAYMENT_CONFIRM];

        $requests = Requests::where('user_id' , $user_id)->whereIn('status' , $status)->count();

        if(!$requests) {

            return true;

        }
        return false;
    }

    /**
     * Used to store requests details as temporary
     *
     * @param : Form data
     * 
     * @return : 
     */

    public static function draft($request) {

        Log::info('Draft Request Start');

        $temp_request = new TempRequests;

        $allow = true;

        if($request->temp_request_id) {

            $temp_request = TempRequests::where('id' , $request->temp_request_id)->where('user_id' , $request->id)->first();

            if(!$temp_request) {
                $allow = false;
            }
        
        }

        if($allow) {

            if($request->has('id')) {
                $temp_request->user_id =  $request->id;
            }

            if($request->has('request_type')) {

                // Check the selected request type is available

                $allowed_request_types = [REQUEST_TYPE_DIRECT,REQUEST_TYPE_LOCATION,REQUEST_TYPE_CATEGORY];

                if(in_array($request->request_type, $allowed_request_types)) {
                    $temp_request->request_type =  $request->request_type;
                }
            }

            if($request->has('job_type')) {

                $allowed_job_types = [REQUEST_NOW,REQUEST_LATER];
                
                if(in_array($request->job_type, $allowed_job_types)) {
                    $temp_request->job_type =  $request->job_type;
                }
                // If the request is now means no need to save the request_tyep
                if($request->job_type == REQUEST_NOW) {
                    $temp_request->request_type = "";
                }
            }

            if($request->has('step')) {
                $temp_request->step =  $request->step;
            }

            if($request->has('category_id')) {

                if(Category::find($request->category_id))
                    $temp_request->category_id =  $request->category_id;
            }

            if($request->has('sub_category_id')) {

                $check_sub_category = SubCategory::where('category_id' , $temp_request->category_id)->where('id' , $request->sub_category_id)->first();

                if($check_sub_category)
                    $temp_request->sub_category_id =  $request->sub_category_id;
            }

            if($request->has('name')) {
                $temp_request->name =  $request->name;
            }

            if($request->has('description')) {
                $temp_request->description =  $request->description;
            }

            if($request->has('price_type')) {
                $temp_request->price_type =  $request->price_type;
            }

            if($request->has('note')) {
                $temp_request->note =  $request->note;
            }

            if($request->has('request_date')) {
                $temp_request->request_date = date('Y-m-d H:i:s' , strtotime($request->request_date));
            }

            if($request->has('s_latitude')) {
                $temp_request->s_latitude =  $request->s_latitude;
            }

            if($request->has('s_longitude')) {
                $temp_request->s_longitude =  $request->s_longitude;
            }

            if($request->has('s_address')) {
                $temp_request->s_address =  $request->s_address;
            }

            if($request->has('step')) {
                $temp_request->step = $request->step;
            }

            if($request->has('user_price')) {
                $temp_request->user_price = $request->user_price;
            }

            $temp_request->status = 1;
            $temp_request->save();

            $data = TempRequests::tempRequestResponse($temp_request->id)->first();

            $response_array = ['success' => true , 'data' =>$data];

        } else {
            $response_array = ['success' => false , 'error' => Helper::error_message(152) ,  'error_code' => 152];
        }

        Log::info('Draft Request end');

        return $response_array;
    
    }

    /**
     * @method provider_list()
     *
     * @uses Get provider List based on the category and sub category 
     *
     * @created vidhya R
     *
     * @updated vidhya R
     *
     * @param category_id and sub_category_id
     *
     * @return available provider ids
     */

    public static function provider_list($request , $temp_request) {

        // Check temp_request record is exists

        if(!$temp_request = TempRequests::find($temp_request->id)) {

            $response_array = ['success' => false , 'error' => Helper::error_message(112) , 'error_code' => 112];

            return response()->json($response_array , 200);
        
        }

        Log::info("Get Provider List Start");

        $user_id = $request->id;

        // Get the providers based on the availability and user selected category_id , sub_category_id

        $base_query = ProviderService::where('category_id' , $temp_request->category_id)
                        ->join('providers', 'provider_services.provider_id', '=', 'providers.id')
                        ->where('providers.user_id' , '!=' , $user_id)
                        ->where('provider_services.status' , true)
                        ->where('providers.is_available', PROVIDER_AVAILABLE)
                        ->where('providers.is_approved', PROVIDER_APPROVED)
                        ->where('providers.waiting_to_respond', WAITING_TO_RESPOND_NORMAL)
                        ->where('provider_services.status', APPROVED);

        /** check the category has paid subscriptions **/       
        $subscriptions = is_category_subscription($temp_request->category_id);

        /** if category has paid subscriptions **/
        if($subscriptions) {

            $category_subscribed_providers = get_category_subscribed_providers($temp_request->category_id);
           
            if($category_subscribed_providers) {
               
                $base_query = $base_query->whereIn('provider_services.provider_id', $category_subscribed_providers);
            }

        }

        if($temp_request->sub_category_id) {

            $base_query = $base_query->where('sub_category_id' , $temp_request->sub_category_id);
        }

        $category_base_providers = $base_query->select('provider_services.provider_id as provider_id')->get();

        Log::info("Category Base Providers Query Completed");

        // Check the category providers list is not empty

        if(!$category_base_providers) {

            Log::info("Category Base Providers - EMPTY RESULT");

            $response_array = ['success' => false , 'error' => Helper::error_message(111) , 'error_code' => 111];

            return $response_array;

        } else {

            Log::info("Request Date - Validation Start");

            // Convert category base providers list array to string conversion

            $ids = [];

             // Filter the providers who have the jobs on the current user requested date 

            $start_date = Helper::add_date($temp_request->request_date,"-1");  

            $end_date = Helper::add_date($temp_request->request_date,"+2");

            $list_status = [REQUEST_COMPLETED , REQUEST_CANCELLED , REQUEST_NO_PROVIDER_AVAILABLE];

            foreach ($category_base_providers as $key => $category_base_provider) {

                $cat_providers[] = $category_base_provider->provider_id;

                $check_request_provider = Requests::where('provider_id' , $category_base_provider->provider_id)
                                    ->whereBetween('request_date' , [$start_date , $end_date])
                                    ->whereNotIn('status' , $list_status)
                                    ->count();

                if(!$check_request_provider) {
                    $ids[] = $category_base_provider->provider_id;
                }

            }

            $ids = implode(',', $ids);

            Log::info("Request Date - Validation End");

            if($ids) {

                $response_array = ['success' => true , 'provider_ids' => $ids];

            } else {
                
                Log::info("No Providers Found");

                $response_array = ['success' => false , 'error' => Helper::error_message(111) , 'error_code' => 111];

                return $response_array;
            }

            Log::info("Request Date - Validation End");
        } 

        Log::info("Get Provider List END");

        return $response_array;
    
    }

    /**
     * @method location_providers_save()
     *
     * @uses Based on the location, the requests providers details will be saved
     *
     * @created vidhya R
     *
     * @updated vidhy R
     *
     * @param temp_request details, Form data and JOB TYPE
     *
     * @return array response
     */

    public static function location_providers_save($request , $temp_request  , $job_type = REQUEST_NOW) {

        Log::info("Location Based Providers Start");

        // Get the search radius from the admin

        $distance = Setting::get('search_radius' , 50);

        // Get the providers based on the availability and user selected category_id , sub_category_id

        Log::info("Get Providers List");

        $category_base_providers = Self::provider_list($request, $temp_request);

        // if no providers found sent failure messages

        if($category_base_providers['success'] == false) {

            Log::info("No Providers Found");

            $response_array = $category_base_providers;

            return $response_array;

        } else {

            // Initialize the variable

            $list_fav_providers = $list_fav_provider = [];

            /** Fav Providers SEARCH started */

            $fav_providers = Self::get_fav_providers($request->sub_category_id,$request->id);

            // Check Favourite Providers list is not empty

            if($fav_providers) {

                foreach ($fav_providers as $key => $favProvider) {

                    $list_fav_provider['id'] = $favProvider->provider_id;

                    $list_fav_provider['waiting'] = $favProvider->waiting;

                    $list_fav_provider['distance'] = 0;

                    array_push($list_fav_providers, $list_fav_provider);
                }                
            }
            /** Fav providers end */

            Log::info("Search Nearby providers Start");

            $providers = $category_base_providers['provider_ids'];
        
            $query = "SELECT providers.id,providers.waiting_to_respond as waiting, 1.609344 * 3956 * acos( cos( radians('$temp_request->s_latitude') ) * cos( radians(latitude) ) * cos( radians(longitude) - radians('$temp_request->s_longitude') ) + sin( radians('$temp_request->s_latitude') ) * sin( radians(latitude) ) ) AS distance FROM providers
                    WHERE id IN ($providers) AND (1.609344 * 3956 * acos( cos( radians('$temp_request->s_latitude') ) * cos( radians(latitude) ) * cos( radians(longitude) - radians('$temp_request->s_longitude') ) + sin( radians('$temp_request->s_latitude') ) * sin( radians(latitude) ) ) ) <= $distance
                    ORDER BY distance";

            $location_providers = \DB::select(\DB::raw($query));

            Log::info("Search query: " . $query);

            if($location_providers) {

                Log::info("New Request Create");

                $request_data = self::create($request);

                if($request_data['success']) {

                    // Initialize Final list of provider variable
                    $merge_providers = $search_providers = $search_provider = [];

                    foreach ($location_providers as $location_provider) {

                        $search_provider['id'] = $location_provider->id;

                        $search_provider['waiting'] = $location_provider->waiting;

                        $search_provider['distance'] = $location_provider->distance;
                        
                        array_push($search_providers, $search_provider);
                   
                    }

                    // Log::info("list_fav_providers".print_r($list_fav_providers , true));
                    
                    // Log::info("search_providers".print_r($search_providers , true));

                    // Merge the favourite providers and search providers

                    $merge_providers = array_merge($list_fav_providers,$search_providers);

                    // Sort the providers based on the waiting time
                    $sort_waiting_providers = Self::sort_waiting_providers($merge_providers);  

                    // Get the final providers list
                    $final_providers = $sort_waiting_providers['providers'];   

                    $check_waiting_provider_count = $sort_waiting_providers['check_waiting_provider_count'];

                    if(count($final_providers) == $check_waiting_provider_count) {

                        return response()->json($response_array = array('success' => false, 'error' => Helper::get_error_message(112), 'error_code' => 112) , 200);

                    }

                    Log::info("Request Meta Data Creation");

                    $request_id = $request_data['request_id'];

                    $request_details = Requests::find($request_id);

                    foreach ($final_providers as $key => $final_provider) {

                        $request_meta = new RequestsMeta;

                        $request_meta->status = REQUEST_META_NONE;

                        // For Request Now - The request will be offer to the first provider

                        // For Request Later -  Request will be assigned all the providers 

                        // Send push notifications to the first provider

                        $title = Helper::push_message(8010);

                        $message = Helper::push_message(8011);

                        if($job_type == REQUEST_NOW) {

                            if($key == 0) {

                                $request_meta->status = REQUEST_META_OFFERED;

                                // Availablity status change

                                if($current_provider = Provider::find($final_provider)) {

                                    $current_provider->waiting_to_respond = WAITING_TO_RESPOND;

                                    $current_provider->save();

                                    $email = EmailTemplate::getRowValue(ON_DEMAND_REQUEST, $current_provider, $request_details->user, $request_details);

                                    $notification = NotificationTemplate::getRowValue(ON_DEMAND_REQUEST, $current_provider,  $request_details->user, $request_details);

                                    $content = $notification ? $notification : ON_DEMAND_REQUEST;

                                    if (Setting::get('provider_notification_control')) {

                                        ProviderNotification::save_notification($current_provider, $content, $request_details->user_id, $request_details->id, ON_DEMAND_REQUEST);
                                    }

                                    if (Setting::get('sms_notification')) {

                                        SmsTemplate::getRowValue(ON_DEMAND_REQUEST, $current_provider);                          
                                    }

                                    dispatch(new sendPushNotification($final_provider,PROVIDER,PUSH_PROVIDER_HOME,$title,$message , NOW_REQUEST_IOS_STATUS , $request_details->id));
                                }
                            
                            }
                        
                        } else {

                            $request_meta->status = REQUEST_META_OFFERED;  

                            $page_type = PUSH_PROVIDER_AVAILABLE_TAB;

                            dispatch(new sendPushNotification($final_provider,PROVIDER,PUSH_PROVIDER_SINGLE,$title,$message , "" , $request_details->id , $page_type)); 

                            if ($current_provider = Provider::find($final_provider)) {

                                $email = EmailTemplate::getRowValue(NEARBY_PROVIDER, $current_provider, $request_details->user, $request_details);

                                $notification = NotificationTemplate::getRowValue(NEARBY_PROVIDER, $current_provider, $request_details->user, $request_details);

                                $content = $notification ? $notification : NEARBY_PROVIDER;

                                if (Setting::get('provider_notification_control')) {

                                    ProviderNotification::save_notification($current_provider, $content, $request_details->user_id, $request_details->id,NEARBY_PROVIDER);

                                }

                                if (Setting::get('sms_notification')) {

                                    SmsTemplate::getRowValue(NEARBY_PROVIDER, $current_provider);
                                        
                                }   

                            }
                        
                        }

                        $request_meta->provider_id = $final_provider;

                        $request_meta->request_id = $request_id;

                        $request_meta->save();
                    
                    }

                    $reqeusts_data = self::single_request_response($request_details);

                    // Delete Temp Request 

                    $temp_request->forceDelete();
                    
                    $response_array = ['success' => true , 'message' => Helper::success_message(206) , 'code' => 206 , 'data' => $reqeusts_data];

                } else {

                    $response_array = ['success' => false , 'error' => Helper::error_message(113) , 'error_code' => 113];
                }

            } else {

                Log::info("No Nearby providers found");

                // No Need send notifications to users
                $response_array = array('success' => false, 'error' => Helper::error_message(111), 'error_code' => 111);
            }

        }

        Log::info("Location Based Providers End");

        return $response_array;

    }

    /**
     * Used to create a request with selected provider
     * 
     * @param $request Request Form 
     * @param $temp_request Temp Request details
     * @param $job_type Later or Now
     *
     * @note request_meta status = REQUEST_META_OFFERED , requests status = REQUEST_CREATED
     *
     * @note requests provider_status = PROVIDER_NONE
     *
     * @note : The request shouldn't directly assigned to that provider. Only offere to that provider directly
     * Because the selected provider can also bid 
     */

    public static function direct_providers($request, $temp_request , $job_type = REQUEST_LATER) {

        // Create request

        $request_data = self::create($request);

        if($request_data['success']) {

            $request_id = $request_data['request_id'];

            $request_details = Requests::find($request_id);

            // Save the requests meta details

            $requests_meta =  new RequestsMeta;
            $requests_meta->provider_id = $request->provider_id;
            $requests_meta->request_id  = $request_id;
            $requests_meta->status = REQUEST_META_OFFERED;
            $requests_meta->save();

            // Save the requests details

            $request_details->provider_id = $request->provider_id;
            $request_details->status = REQUEST_CREATED;
            $request_details->provider_status = PROVIDER_NONE;
            $request_details->save();

            // Send notifications to the provider

            $title = Helper::push_message(8010);

            $message = Helper::push_message(8011);

            $page_type = PUSH_PROVIDER_DIRECT_TAB;

            dispatch(new sendPushNotification($request_details->provider_id,PROVIDER,PUSH_PROVIDER_SINGLE,$title,$message , "" , $request_details->id , $page_type));


            if ($request_details->provider) {

                $email = EmailTemplate::getRowValue(ON_LATER_REQUEST_DIRECT, $request_details->provider, $request_details->user, $request_details);

                $notification = NotificationTemplate::getRowValue(ON_LATER_REQUEST_DIRECT, $request_details->provider, $request_details->user, $request_details);

                $content = $notification ? $notification : ON_LATER_REQUEST_DIRECT;

                if (Setting::get('provider_notification_control')) {

                    ProviderNotification::save_notification($request_details->provider, $content,$request_details->user_id, $request_details->id, ON_LATER_REQUEST_DIRECT);

                }

                if (Setting::get('sms_notification')) {

                    SmsTemplate::getRowValue(ON_LATER_REQUEST_DIRECT, $request_details->provider);
                        
                }

            }

            $reqeusts_data = self::single_request_response($request_details);

            $response_array = ['success' => true , 'message' => Helper::success_message(206) , 'code' => 206 , 'data' => $reqeusts_data];
        
        }  else {

            Log::info("REQUEST CREATE API - ERROR");
            
            $response_array = ['success' => false , 'error' => Helper::error_message(113) , 'error_code' => 113];
        }

        return $response_array;

    }

    public static function create($request) {

        Log::info('CREATE REQUEST DETAILS::::::::::::'.print_r($request->all(), true));

        if($request->temp_request_id) {
            $temp_request = TempRequests::find($request->temp_request_id);
        }

        $data = New Requests;

        $user = User::find($request->id);

        if($temp_request) {

            $data->request_type = $request->has('request_type') ?  $request->request_type : $temp_request->request_type;

            $data->price_type = $request->has('price_type') ? $request->price_type : ($temp_request->price_type) ? $temp_request->price_type : PRICE_TYPE_HOUR_BASED;

            $data->job_type = $request->has('job_type') ? $request->job_type : $temp_request->job_type;

            $data->category_id = $request->has('category_id') ? $request->category_id : $temp_request->category_id ;

            $data->sub_category_id = $request->has('sub_category_id') ? $request->sub_category_id : $temp_request->sub_category_id;

            $data->s_latitude = $request->has('s_latitude') ? $request->s_latitude : $temp_request->s_latitude;

            $data->s_longitude = $request->has('s_longitude') ? $request->s_longitude : $temp_request->s_longitude;

            $data->s_address = $request->has('s_address') ? $request->s_address : $temp_request->s_address;

            $data->note = $request->has('note') ? $request->note : $temp_request->note;
        
            $data->user_price = $request->user_price ? $request->user_price : $temp_request->user_price;
        
        }

        $data->user_id  = $request->id;
        
        if($user)
            $data->payment_mode = $user->payment_mode;

        $data->request_start_time  = date('Y-m-d H:i:s');
        $data->step = $request->has('step') ? $request->step : "";
        $data->name = $request->has('name') ? $request->name : "";
        $data->description = $request->has('description') ? $request->description : "";
        
        \Log::info('REQUEST DATE - -'.$request->request_date);

        if($request->request_date) {

            if($data->job_type == REQUEST_NOW) {

                $data->request_date = date('Y-m-d H:i:s'  , strtotime($request->request_date));

            } else {
            
                $data->request_date =  $user->timezone ? convertTimeToUTCzone($request->request_date , $user->timezone) : date('Y-m-d H:i:s', strtotime($request->request_date));
            }

        } else {

            if($request->job_type == REQUEST_NOW) {

                $data->request_date = date('Y-m-d H:i:s');

            }
        }

        $data->status = REQUEST_CREATED;
        $data->currency = Setting::get('currency' , '$');
        $data->save();

        $name = $request->has('name') ? preg_replace('/[^A-Za-z0-9\-]/', '', str_replace(' ', '-', $request->name)) : "";
        
        $data->unique_id = uniqid(preg_replace("/[^a-zA-Z]/", "", $name));

        $data->save();

        if($data->price_type == PRICE_TYPE_FIXED_BASED) {

            $data->price_per_symbol = tr('fixed_price_symbol');

        } else {

            $data->price_per_symbol = tr('per_hour_symbol');

        }
        
        $data->save();

        $response_array  = ['success' => true , 'request_id' => $data->id ,'data' => $data->toArray()];

        Log::info('Request Create...'.print_r($response_array, true));

        return $response_array;

    }

    public static function update($request) {

        $data = Requests::find($request->request_id);

        if($data) {

            // Request Basic fields

            $data->request_type = $request->has('request_type') ?  $request->request_type : $data->request_type;
            $data->job_type = $request->has('job_type') ? $request->job_type : $data->job_type;
            $data->step = $request->has('step') ? $request->step : $data->step;

            // Category and Sub Category Selection

            $data->category_id = $request->has('category_id') ? $request->category_id : $data->category_id ;
            
            $data->sub_category_id = $request->has('sub_category_id') ? $request->sub_category_id : $data->sub_category_id;

            // Requests Form Details

            $data->name = $request->has('name') ? $request->name : $data->name;

            if($request->has('request_date')) {
                $data->request_date =  date('Y-m-d H:i:s' , strtotime($request->request_date));
            }

            $data->description = $request->has('description') ? $request->description : $data->description;

            $data->note = $request->has('note') ? $request->note : $data->note;
            $data->user_price = $request->has('user_price') ? $request->user_price : $data->user_price;

            // Requests location details

            $data->s_latitude = $request->has('s_latitude') ? $request->s_latitude : $data->s_latitude;
            $data->s_longitude = $request->has('s_longitude') ? $request->s_longitude : $data->s_longitude;

            $data->s_address = $request->has('s_address') ? $request->s_address : $data->s_longitude;
            $data->save();
        } 

        $response_array  = ['success' => true , 'data' => $data , 'message' => Helper::success_message(223) , 'code' => 223];

        return $response_array;

    }

    public static function view($request , $user_type = USER , $timezone = "") {

        $data =  Requests::find($request->request_id);

        $provider_id = ""; $bids = [];

        if($user_type == USER) {
            $bids = $data->bids;
        }
        
        if($user_type == PROVIDER) {

            $provider_id = $request->id;

        }

        $response = self::single_request_response($data , $provider_id , $timezone);

        $response_array = array('success' => true, 'data' => $response , 'bids' => $bids);

        return $response_array;
    }

    public static function all($request) {

        $data = Requests::orderBy('created_at' , 'desc')->get();
        
        $response_array = ['success' => true , 'data' => $data];

        return $response_array;

    }

    /**
     * Common Response for Requests
     * 
     * @param : $requests = list requests or single request details
     * 
     * @param : $group = RESPONSE_SINGLE (If single request means no need foreach)
     *
     * @param : $bid_response(provider_id)  - For provider (bidded_requests) we need send provider bid amount
     */

    public static function base_response($requests , $group = RESPONSE_SINGLE , $bid_response = "", $timezone = "" , $check_provider_later = "") {

        $result = [];
               
        $cancelled_status = [REQUEST_CANCELLED, REQUEST_NO_PROVIDER_AVAILABLE, REQUEST_TIME_EXCEED_CANCELLED];

        if($group == RESPONSE_MULTIPLE) {

            foreach ($requests as $key => $request_details) {

                $response_allow = 1;
                
                // Request Type - Nearby providers - check the request is assigned to that provider

                if($check_provider_later && $request_details->request_type == REQUEST_TYPE_LOCATION) {

                    $provider_id = $check_provider_later;

                    $check_meta_data = RequestsMeta::where('request_id' , $request_details->id)->where('provider_id' , $provider_id)->where('status' , REQUEST_META_OFFERED)->count();

                    if($check_meta_data == 0) {

                        $response_allow = 0;
                    }

                    Log::info("response_allow".$response_allow);

                }

                if($response_allow) {

                    $provider_id = "";

                    $data['request_id'] = $request_details->id;
                    $data['unique_id'] = $request_details->unique_id;
                    $data['user_id'] = $request_details->user_id;
                    $data['provider_id'] = $provider_id = $request_details->provider_id;
                    $data['job_type'] = $request_details->job_type;
                    $data['request_type'] = $request_details->request_type;
                    $data['s_address'] = $request_details->s_address;
                    $data['name'] = $request_details->name;
                    $data['description'] = $request_details->description;
                    $data['request_date'] = $timezone ? common_date($request_details->request_date , $timezone) : $request_details->request_date;
                    $data['user_name'] = $request_details->user ? $request_details->user->name : "";
                    $data['user_picture'] = $request_details->user ? $request_details->user->picture : "";

                    $data['provider_picture'] = $data['provider_name'] = $data['provider_mobile'] = $data['provider_rating']  = "";

                    $data['is_fav_provider'] = 0;

                    if($request_details->provider_id && $request_details->provider) {
                        $data['provider_name'] = $request_details->provider->name;
                        
                        $data['provider_picture'] = $request_details->provider->picture;
                        
                        $data['provider_mobile'] = $request_details->provider->mobile;
                       
                        // check the user ratings for provider for request
                        $check_user_rating = UserRating::where('user_id' , $request_details->user_id)->where('provider_id' , $request_details->provider_id)->where('request_id' , $request_details->id)->first();
                      
                        $data['provider_rating'] = $check_user_rating ? $check_user_rating->rating: 0;

                        // check the fav provider status

                        $check_fav_provider = UserFavProvider::where('user_id' , $request_details->user_id)->where('provider_id' , $request_details->provider_id)->where('status' , 1)->count();

                        $data['is_fav_provider'] = $check_fav_provider ? YES : NO;
                    
                    }
                    
                    $data['category_name'] = $data['sub_category_name'] = $data['service_picture'] = "";
                    
                    if($request_details->category_id && $request_details->category)
                        $data['category_name'] = $request_details->category->name;
                    
                    if($request_details->sub_category_id && $request_details->subCategory) {
                        $data['sub_category_name'] =  $request_details->subCategory->name;
                        $data['service_picture'] = $request_details->subCategory->picture;                    
                    
                    }

                    $data['user_price'] = $request_details->user_price;

                    $data['price_type'] = price_type($request_details->price_type);

                    $data['price_per_symbol'] = $request_details->price_per_symbol;

                    $data['price_per_hour'] = $request_details->price_per_hour;

                    // For Provider Bidding Details

                    $data['chat_status'] = $data['bid_amount'] = 0;

                    if($bid_response) {

                        $provider_id = $bid_response;

                        $bid_details = $request_details->requestMeta()->where('provider_id' , $bid_response)->first();

                        if($bid_details) {
                            $data['bid_amount'] = $bid_details->bid_amount;
                            $data['bid_status'] = $bid_details->status;
                        }
                    
                    }

                    if($provider_id) {
                        
                        // Check the user initiated any chat

                        $messages = check_provider_chat($request_details->user_id , $provider_id , $request_details->id);

                        if($messages) {
                            $data['chat_status'] = 1;
                        }
                    
                    }

                    $data['total'] = $request_details->total;
                    $data['currency'] = $request_details->currency;
                    $data['payment_mode'] = $request_details->payment_mode;
                    $data['status'] = $request_details->status;
                    $data['provider_status'] = $request_details->provider_status ? $request_details->provider_status : 0 ;
                    $data['created_at'] = $request_details->created_at;

                    $data['cancelled_date'] = $request_details->cancelled_date;

                    $data['cancelled_by'] = $request_details->cancelled_by;
                    
                    $data['cancelled_reason'] = $request_details->cancelled_reason;
                                        
                    $data['is_cancelled'] = in_array($request_details->status, $cancelled_status) ? YES: NO;

                    Helper::null_safe($data);
                
                    array_push($result, $data);
                }            
            }

        } else {

            $provider_id = "";

            $result['request_id'] = $requests->id;
            $result['unique_id'] = $requests->unique_id;
            $result['user_id'] = $requests->user_id;
            $result['provider_id'] = $provider_id = $requests->provider_id;
            $result['job_type'] = $requests->job_type;
            $result['request_type'] = $requests->request_type;
            $result['s_address'] = $requests->s_address;
            $result['name'] = $requests->name;
            $result['description'] = $requests->description;
            $result['request_date'] = $timezone ? common_date($requests->request_date , $timezone) : $requests->request_date;
            $result['user_name'] = $requests->user ? $requests->user->name : "";
            $result['user_picture'] = $requests->user ? $requests->user->picture : "";

            $result['provider_picture'] = $result['provider_name'] = $result['provider_mobile'] = $result['provider_rating']  = $result['service_picture'] = "";
            
            $result['is_fav_provider'] = 0;

            if($requests->provider_id) {
                
                $result['provider_name'] = $requests->provider ? $requests->provider->name : "";
                $result['provider_picture'] = $requests->provider ? $requests->provider->picture : "";
                $result['provider_mobile'] = $requests->provider ? $requests->provider->mobile : "";
                 // check the user ratings for provider for request
                $check_user_rating = UserRating::where('user_id' , $requests->user_id)->where('provider_id' , $requests->provider_id)->where('request_id' , $requests->id)->first();
              
                $result['provider_rating'] = $check_user_rating ? $check_user_rating->rating: 0;

                // check the fav provider status

                $check_fav_provider = UserFavProvider::where('user_id' , $requests->user_id)->where('provider_id' , $requests->provider_id)->where('status' , 1)->count();

                $result['is_fav_provider'] = $check_fav_provider ? YES : NO;


            }
            $result['category_name'] = $requests->category ? $requests->category->name : "";
            $result['sub_category_name'] = $requests->subCategory ? $requests->subCategory->name : "";
            $result['service_picture'] = $requests->subCategory ? $requests->subCategory->picture: "";
            
            $result['user_price'] = $requests->user_price;

            $result['price_per_hour'] = $requests->price_per_hour;
            $result['price_type'] = price_type($requests->price_type);
            $result['price_per_symbol'] = $requests->price_per_symbol;

            $result['chat_status'] = $result['bid_amount'] = 0;

            if($bid_response) {

                $provider_id = $bid_response;

                $bid_details = $requests->requestMeta()->where('provider_id' , $bid_response)->first();

                if($bid_details) {

                    $result['bid_amount'] = $bid_details->bid_amount;
                    $result['bid_status'] = $bid_details->status;
                }
            }

            if($provider_id) {
                    
                // Check the user initiated any chat

                $messages = check_provider_chat($requests->user_id , $provider_id , $requests->id);

                if($messages) {
                    $result['chat_status'] = 1;
                }
            }

            $result['total'] = $requests->total;
            $result['currency'] = $requests->currency;
            $result['payment_mode'] = $requests->payment_mode;
            $result['status'] = $requests->status;
            $result['provider_status'] = $requests->provider_status;
            $result['created_at'] = $requests->created_at;

            $result['cancelled_date'] = $requests->cancelled_date;

            $result['cancelled_by'] = $requests->cancelled_by;
            
            $result['cancelled_reason'] = $requests->cancelled_reason;
                                
            $data['is_cancelled'] = in_array($requests->status, $cancelled_status) ? YES: NO;
        }

        return $result;
        
    }

    /**
     * Temp Request response 
     *
     *
     */

    public static function temp_base_response($requests , $group = RESPONSE_SINGLE) {

        $result = [];

        if($group == RESPONSE_MULTIPLE) {

            foreach ($requests as $key => $request_details) {

                $data['temp_request_id'] = $request_details->id;
                $data['unique_id'] = $request_details->unique_id;
                $data['user_id'] = $request_details->user_id;
                $data['provider_id'] = $request_details->provider_id;
                $data['job_type'] = $request_details->job_type;
                $data['request_type'] = $request_details->request_type;
                $data['s_latitude'] = $request_details->s_latitude;
                $data['s_longitude'] = $request_details->s_longitude;
                $data['s_address'] = $request_details->s_address;
                $data['name'] = $request_details->name;
                $data['description'] = $request_details->description;
                $data['price_type'] = price_type($request_details->price_type);
                $data['request_date'] = $request_details->request_date;
                $data['user_name'] = $request_details->user->name;
                $data['user_picture'] = $request_details->user->picture;
                
                $data['category_name'] = $data['sub_category_name'] = "";

                if($request_details->category_id && $request_details->category)
                    $data['category_name'] = $request_details->category->name;
                
                if($request_details->sub_category_id && $request_details->subCategory)
                    $data['sub_category_name'] = $request_details->subCategory->name;

                $data['step'] = $request_details->step;
                $data['user_price'] = $request_details->user_price;
                $data['status'] = $request_details->status;
                $data['created_at'] = $request_details->created_at;
                Helper::null_safe($data);
            
                array_push($result, $data);
            }
        
        } else {

            $result['temp_request_id'] = $requests->id;
            $result['unique_id'] = $requests->unique_id;
            $result['user_id'] = $requests->user_id;
            $result['provider_id'] = $requests->provider_id;
            $result['job_type'] = $requests->job_type;
            $result['request_type'] = $requests->request_type;
            $result['s_latitude'] = $requests->s_latitude;
            $result['s_longitude'] = $requests->s_longitude;
            $result['s_address'] = $requests->s_address;            
            $result['name'] = $requests->name;
            $result['description'] = $requests->description;
            $result['request_date'] = $requests->request_date;
            $result['user_name'] = $requests->user->name;
            $result['user_picture'] = $requests->user->picture;
            $result['category_name'] = $requests->category ? $requests->category->name : "";
            $result['sub_category_name'] = $requests->subCategory ? $requests->subCategory->name : "";
            $result['user_price'] = $request_details->user_price;
            $result['price_type'] = $request_details->price_type;
            $result['status'] = $requests->status;
            $result['created_at'] = $requests->created_at;
        
        }

        return $result;
        
    }

    /**
     *
     *
     */

    public static function single_request_response($requests , $provider = "" , $timezone = "") {

        $data = [];

        $completed_status = [REQUEST_RATING, REQUEST_COMPLETED];
        
        $cancelled_status = [REQUEST_CANCELLED, REQUEST_NO_PROVIDER_AVAILABLE, REQUEST_TIME_EXCEED_CANCELLED];

        if($requests) {

            $data['request_id'] = $requests->id;
            $data['unique_id'] = $requests->unique_id;

            $data['user_id'] = $requests->user_id;
            $data['user_name'] = $requests->user ? $requests->user->name : "";
            $data['user_picture'] = $requests->user ? $requests->user->picture : "";
            $data['user_mobile'] = $requests->user ? $requests->user->mobile : "";

            $data['provider_id'] = $provider_id = $requests->provider_id;

            if($requests->provider_id) {
                $data['provider_name'] = $requests->provider ? $requests->provider->name : "";
                $data['provider_picture'] = $requests->provider ? $requests->provider->picture : "";
                $data['provider_mobile'] = $requests->provider ? $requests->provider->mobile : "";
            } else {
                $data['provider_name'] = $data['provider_picture'] = $data['provider_mobile'] = "";
            }

            $data['name'] = $requests->name;
            $data['description'] = $requests->description;
            $data['note'] = $requests->note;
            $data['request_date'] = $timezone ? common_date($requests->request_date , $timezone) : $requests->request_date;
            $data['status'] = $requests->status;
            $data['provider_status'] = $requests->provider_status;
            $data['request_type'] = $requests->request_type;
            $data['job_type'] = $requests->job_type;

            $data['category_id'] = $requests->category_id;
            $data['category_name'] = $requests->category ? $requests->category->name : "";
            $data['category_picture'] = $requests->category ? $requests->category->picture : "";

            $data['sub_category_id'] = $requests->sub_category_id;
            $data['sub_category_name'] = $requests->subCategory ? $requests->subCategory->name : "";
            $data['sub_category_picture'] = $requests->subCategory ? $requests->subCategory->picture : "";

            $data['service_picture'] = $requests->subCategory ? $requests->subCategory->picture: "";

            $data['currency'] = $requests->currency;
            $data['payment_mode'] = $requests->payment_mode;
            $data['user_price'] = $requests->user_price;
            $data['price_per_hour'] = $requests->price_per_hour;
            $data['price_type'] = price_type($requests->price_type);
            $data['price_per_symbol'] = $requests->price_per_symbol;

            $data['chat_status'] = $data['bid_amount'] = 0; $data['bid_status'] = "";

            if($provider) {

                $provider_id = $provider;

                $bid_request_details = $requests->requestMeta()->where('provider_id' , $provider)->first();

                if($bid_request_details) { 
                    $data['bid_amount'] = $bid_request_details->bid_amount;
                    $data['bid_status'] = $bid_request_details->status;
                }
            }

            // Check the provider id is not empty 

            if($provider_id) {

                 // Check the user initiated any chat

                $messages = check_provider_chat($requests->user_id , $provider_id , $requests->id);

                if($messages) {
                    $data['chat_status'] = 1;
                }
            }

            $data['s_latitude'] = $requests->s_latitude ? $requests->s_latitude : "0.0";
            $data['s_longitude'] = $requests->s_longitude ? $requests->s_longitude : "0.0";
            $data['s_address'] = $requests->s_address;
            $data['d_latitude'] = $requests->d_latitude;
            $data['d_longitude'] = $requests->d_longitude;
            $data['d_address'] = $requests->d_address;
            $data['before_image'] = $requests->before_image;
            $data['after_image'] = $requests->after_image;

            $data['provider_lat'] = $data['provider_long'] = 0.0000;

            $data['is_fav_provider'] = 0;

            //Provider lat and lon 

            if($requests->provider_id) {

                if($requests->provider) {
                    
                    $data['provider_lat'] = $requests->provider ? $requests->provider->latitude : 0.00000;

                    $data['provider_long'] = $requests->provider ? $requests->provider->longitude : 0.000000;
                }

                // check the fav provider status

                $check_fav_provider = UserFavProvider::where('user_id' , $requests->user_id)->where('provider_id' , $requests->provider_id)->where('status' , 1)->count();

                $data['is_fav_provider'] = $check_fav_provider ? YES : NO;
            
            }

            $invoice_query = RequestPayment::where('request_id' , $requests->id)
                            ->leftJoin('requests' , 'request_payments.request_id' , '=' , 'requests.id')
                            ->leftJoin('users' , 'requests.user_id' , '=' , 'users.id')
                            ->leftJoin('cards' , 'users.card_id' , '=' , 'cards.id');

            if($requests->user) {

                if($requests->user->payment_mode == CARD) {
                    $invoice_query = $invoice_query->where('cards.is_default' , DEFAULT_TRUE) ;  
                }
            }

            $invoice = $invoice_query->select('request_payments.total_time',
                                'request_payments.payment_mode as payment_mode' , 
                                'request_payments.base_price',
                                'request_payments.time_price' , 
                                'request_payments.tax_price' , 
                                'request_payments.total',
                                'request_payments.coupon_code',
                                'request_payments.coupon_type',
                                'request_payments.coupon_amount',
                                'request_payments.is_coupon_applied',
                                'request_payments.currency',
                                'request_payments.sub_total',
                                'request_payments.actual_total',
                                'request_payments.admin_amount',
                                'request_payments.provider_amount',
                                'cards.card_token','cards.customer_id','cards.last_four')
                            ->get()->toArray();

            
            $data = Helper::null_safe($data);

            $data['invoice'] = $invoice ?: [];

            $provider_rating = ProviderRating::where('provider_id', $requests->provider_id)->avg('rating') ? : 1;

            $user_rating = UserRating::where('user_id', $requests->user_id)->avg('rating') ? : 1;

            $data['provider_rating'] = ceil($provider_rating);

            $data['user_rating'] = ceil($user_rating);

            $data['rating'] = ceil($user_rating);

            $data['cancelled_date'] = $requests->cancelled_date;

            $data['cancelled_by'] = $requests->cancelled_by;

            $data['cancelled_reason'] = $requests->cancelled_reason;
            
            $data['is_cancelled'] = in_array($requests->status, $cancelled_status) ? YES: NO;

        }

        return $data;
    }

    public static function invoice($request_details) {

        // Initialize variables

        $base_price = $price_per_hour = $tax_price = $total_time = $total_time_price = $total = 0;
                
        // Invoice details

        // Get price per minute detials from admin panel

        $price_per_hour = $request_details->price_per_hour;

        // Get the total time from requests table

        $end_time = date('Y-m-d H:i:s');

        $start_time = $request_details->start_time;

        $get_time = strtotime($end_time) - strtotime($start_time);

        $total_time = round($get_time/3600);

        // If work has less than an hour consider as 1 hour

        $total_time = $total_time ? $total_time : 1;

        // Check extra miniutes

        // $total_time = $get_time->i ? $total_time + 1 : $total_time;

        // Calculate price based price_type (Hourly Bases and Fixed price)

        if($request_details->price_type == PRICE_TYPE_FIXED_BASED) {

            $total_time_price = $price_per_hour;

        } else {

            $total_time_price = $total_time * $price_per_hour;
        }

        // Get the tax details from admin panel

        $tax_percentage = Setting::get('tax_price' , 0);

        if($tax_percentage) {

            if($total_time_price) {
                
                $tax_percentage = $tax_percentage /100;

                $tax_price = $total_time_price * $tax_percentage;
            }
        }

        $total = $total_time_price + $base_price + $tax_price;

        $sub_total = $total_time_price + $base_price; 

        $admin_commission = Setting::get('admin_commission') ?: 0 ;

        $admin_amount = 0;

        if($admin_commission) {
            
            $admin_commission = $admin_commission/100;

            $admin_amount = $sub_total * $admin_commission;
        
        } 
                
        $provider_amount = $sub_total - $admin_amount;

        // Save the payment details
        
        if(!RequestPayment::where('request_id' , $request_details->id)->first()) {

            $request_payment = new RequestPayment;
            $request_payment->user_id = $request_details->user_id;
            $request_payment->provider_id = $request_details->provider_id;
            $request_payment->request_id = $request_details->id;
            $request_payment->payment_mode = $request_details->payment_mode;
            $request_payment->base_price = 0;
            $request_payment->time_price = $total_time_price;
            $request_payment->tax_price = $tax_price;
            $request_payment->total_time = $total_time;
            $request_payment->admin_amount = $admin_amount;
            $request_payment->provider_amount = $provider_amount;
            $request_payment->sub_total = $sub_total;
            $request_payment->total = $total;
            $request_payment->save();
            
        }

        $request_details->end_time = $end_time ? $end_time : date("Y-m-d H:i:s");

        $request_details->total = $total;

        $request_details->save();

        return true;
        
    }

    /**
     * @method sort_waiting_providers()
     *
     * @uses to sort the providers based on the waiting state
     *
     * @created vidhya R
     *
     * @updated vidhy R
     *
     * @param array providers
     *
     * @return array response
     */

    public static function sort_waiting_providers($merge_providers) {

        $waiting_array = $non_waiting_array = [];
        
        $check_waiting_provider_count = 0;

        foreach ($merge_providers as $key => $merge_provider_details) {

            if($merge_provider_details['waiting'] == WAITING_TO_RESPOND) {

                $waiting_array[] = $merge_provider_details['id'];

                $check_waiting_provider_count ++;

            } else {

                $non_waiting_array[] = $merge_provider_details['id'];

            }
        }

        $providers = array_unique(array_merge($non_waiting_array,$waiting_array));

        return array('providers' => $providers , 'check_waiting_provider_count' => $check_waiting_provider_count);
        
    }

    /**
     * @method get_fav_providers()
     *
     * @uses Fav providers list based on the selected service
     *
     * @created vidhya R
     *
     * @updated vidhy R
     *
     * @param integer sub_category_id
     *
     * @param integer user_id
     *
     * @return array response
     */

    public static function get_fav_providers($sub_category_id,$user_id) {

        /** Favourite Providers Search Start */

        Log::info('Favourite Providers Search Start');

        $fav_providers = [];  // Initialize the variable

         // Get the favourite providers list

        $fav_providers_query = UserFavProvider::leftJoin('providers' , 'user_fav_providers.provider_id' ,'=' , 'providers.id')
                ->where('user_fav_providers.user_id' , $user_id)
                ->where('providers.is_available' , PROVIDER_AVAILABLE)
                ->where('providers.status' , PROVIDER_APPROVED)
                ->select('provider_id' , 'providers.waiting_to_respond as waiting');

        if($sub_category_id) {

            $provider_services = ProviderService::where('sub_category_id' , $sub_category_id)
                                    ->where('status' , DEFAULT_TRUE)
                                    ->get();

            $provider_ids = [];

            // Get the fav providers based on the user selected services

            if($provider_services ) {

                foreach ($provider_services as $key => $provider_service) {

                    $provider_ids[] = $provider_service->provider_id;

                }

                $fav_providers = $fav_providers_query->whereIn('provider_id' , $provider_ids)->orderBy('waiting' , 'ASC')->get();
            }
                           
        } else {

            $fav_providers = $fav_providers_query->orderBy('waiting' , 'ASC')->get();
        }

        return $fav_providers;

        /** Favourite Providers Search End */
    
    }

}