<?php

namespace App\Repositories;

use App\Repositories\CommonRepository as CommonRepo;

use App\User;

use App\Helpers\Helper;

use Log;

use Validator;

use Setting;

use App\Referral;

use App\UserReferrer;

class UserRepository {

    public static function request_validation($data = [] , &$errors = [] , $user) {

        $validator = Validator::make($data,
            array(
                'request_id' => 'required|integer|exists:requests,id,user_id,'.$user->id,
            ),
            array(
                'exists' => 'The :attribute doesn\'t belong to User:'.$user->name
            )
        );

        if($validator->fails()) {

            $errors = implode(',' ,$validator->messages()->all());

            return false;

        }

        return true;

    }

	public static function all() {

		return User::orderBy('created_at' , 'desc')->get();
	}

	public static function login($request) {

        $user_details = User::where('email', '=', $request->email)->first();

        if(!$user_details) {
                    
            $response = array('success' => false , 'error' => Helper::error_message(504) , 'error_code' => 504);
            
            return response()->json($response, 200);
        }

        if($user_details->is_verified == USER_EMAIL_NOT_VERIFIED) {
                
            $response_array = array('success' => false , 'error' => Helper::error_message(503) , 'error_code' => 503);

            return $response_array;
        }

        if($user_details->status == USER_DECLINED) {
            
            $response_array = array('success' => false , 'error' => Helper::error_message(502) , 'error_code' => 502);
            return $response_array;
        }

        // Validate the user credentials

        if(\Hash::check($request->password, $user_details->password)) {

        	// Generate new tokens
            $user_details->token = Helper::generate_token();

            $user_details->token_expiry = Helper::generate_token_expiry();

            // Save device details
            $user_details->device_token = $request->device_token;

            $user_details->device_type = $request->device_type;

            $user_details->login_by = $request->login_by;

            $user_details->timezone = $request->has('timezone') ? $request->timezone : "";

            $user_details->save();

            $data = User::userResponse($user_details->id)->first();

            if($data) {
                $data = $data->toArray();
            }

            $response_array = ['success' => true , 'data' => $data];

        } else {
            $response_array = array( 'success' => false, 'error' => tr('invalid_email_password'), 'error_code' => 105 );
        }
    

        return $response_array;
	}

    public static function forgot_password($data = [] , &$errors = []) {

        // CASE 1: Check the Email configuration

        if(!envfile('MAIL_USERNAME') && !envfile('MAIL_PASSWORD')) {

            $response_array = ['success' => false , 'error' => Helper::error_message(505) , 'error_code' => 505];

            return $response_array;
        }

        $user_details = User::where('email' , $data->email)->first();

        if($user_details) {

            // CASE 2: Check the the user approved status 

            if($user_details->status == USER_DECLINED) {
                    
                $response = array('success' => false , 'error' => Helper::error_message(502) , 'error_code' => 502);
                return response()->json($response, 200);
            }

            // CASE 2: Check the the user email verification status 

            if($user_details->is_verified == USER_EMAIL_NOT_VERIFIED) {

                if(Setting::get('email_verify_control') == EMAIL_VERIFICATION_ON && !in_array($user_details->login_by, ['facebook' , 'google'])) {

                    // Check the verification code expiry

                    Helper::check_email_verification("" , $user_details, $error, USER);
                
                    $response = array('success' => false , 'error' => Helper::error_message(503) , 'error_code' => 503);

                    return response()->json($response, 200);

                }
            }

            $new_password = Helper::generate_password();

            $user_details->password = \Hash::make($new_password);

            $email_data = array();

            $subject = tr('user_forgot_email_title' , Setting::get('site_name'));

            $page = "emails.user.forgot-password";

            $email_data['username'] = $user_details->name;

            $email_data['new_password'] = $new_password;

            $email_send = Helper::send_email($page,$subject,$user_details->email,$email_data);

            $message = "Thanks! Please check $user_details->email for a link to reset your password.";

            $message = Helper::success_message(232 , $user_details->email);

            $response_array = ['success' => true , 'message' => $message , 'code' => 205];

            $user_details->save();

        } else {
            $response_array = ['success' => false , 'error' => Helper::error_message(153) , 'error_code' => 153];
        }
        
        return $response_array;
    
    }

	public static function store($request) {

		$user = new User;

        $user->name = $request->has('name') ? $request->name : "";
        $user->first_name = $request->has('first_name') ? $request->first_name : "";
        $user->last_name = $request->has('last_name') ? $request->last_name : "";
        
		$user->email = $request->has('email') ? $request->email : "";
		$user->password = $request->has('password') ? \Hash::make($request->password) : "";
        
		$user->gender = $request->has('gender') ? $request->gender : "male";
		$user->mobile = $request->has('mobile') ? $request->mobile : "";
        $user->address = $request->has('address') ? $request->address : "";
        $user->description = $request->has('description') ? $request->description : "";
        $user->timezone = $request->has('timezone') ? $request->timezone : "";

		$user->token = Helper::generate_token();
        $user->token_expiry = Helper::generate_token_expiry();

        $check_device_exist = User::where('device_token', $request->device_token)->first();

        if($check_device_exist) {
            $check_device_exist->device_token = "";
            $check_device_exist->save();
        }

        if($request->has('timezone')) {
            $user->timezone = $request->timezone;
        }

        $user->device_token = $request->has('device_token') ? $request->device_token : "";
        
        $user->device_type = $request->has('device_type') ? $request->device_type : "";
        
        $user->login_by = $request->has('login_by') ? $request->login_by : "manual";
        
		$user->social_unique_id = $request->has('social_unique_id') ? $request->social_unique_id : "";
        

        $user->picture = asset('placeholder.png');

        $user->payment_mode = $request->payment_mode ? $request->payment_mode : 'cod';

        // Upload picture
        
        if($request->login_by == "manual") {

            if($request->hasFile('picture')) {

                $user->picture = Helper::upload_avatar('uploads/users',$request->file('picture'));

            }

        } else {

            if($request->has('picture')) {

                $user->picture = $request->picture;

            }

        }

        $user->login_by = $request->login_by ?  $request->login_by : "manual";

        $user->status = $user->is_activated = 1;

        $user->payment_mode = 'cod';

        $user->login_status = "user";

        $user->save();

        $name = $request->has('name') ? preg_replace('/[^A-Za-z0-9\-]/', '', str_replace(' ', '-', $request->name)) : "";
        
        $user->unique_id = uniqid($name);


        $user->register_type = $request->device_type ? $request->device_type : "web";

        $user->save();
        
        $response = [];

        if($user) {

            register_mobile($user->device_type);

            // Check the default subscription and save the user type 

            if($request->referral_code) {

                self::referral_register($request->referral_code, $user);
            }

            if(!Setting::get('email_verify_control') == EMAIL_VERIFICATION_ON || in_array($user->login_by , ['facebook' , 'google'])) {

                $user->is_verified = 1;

                $user->save();
            }

            $response = User::userResponse($user->id)->first();

            if($response) {

                $response = $response->toArray();
            }

            if($admin_mail = get_admin_mail()) {

                $subject = tr('new_user_signup');

                $page = "emails.admin.user-welcome";

                $email = $admin_mail->email;
              
                Helper::send_email($page,$subject,$email,$user);
            }

            $subject = tr('new_user_signup');

            $page = "emails.user.welcome";

            $email = $user->email;

            Helper::send_email($page,$subject,$email,$user);
        }

        CommonRepo::become_a_provider($user);

        // Send welcome email to the new user:

        return $response;
	}

	public static function update($request , $user_id) {

        if($request->id) {
            $user_id = $request->id;
        }

		$user = User::find($user_id);

        $response = [];

        if($user) {

            if($request->has('name')) {
                $user->name = $request->name;
            }

            if($request->has('email')) {
                $user->email = $request->email;
            }

            if($request->has('description')) {
                $user->description = $request->description;
            }

            if($request->has('mobile')) {
                $user->mobile = $request->mobile;
            }

            if($request->has('timezone')) {

                $user->timezone = $request->timezone;

            }

            $user->device_token = $request->has('device_token') ? $request->device_token : "";

            $user->device_type = $request->has('device_type') ? $request->device_type : $user->device_type;

            $user->login_by = $request->has('login_by') ? $request->login_by : $user->login_by;
            
            $user->login_status = 'user';

            // Upload Picture

            if ($request->hasFile('picture')) {

                Helper::delete_avatar('uploads/users',$user->picture); // Delete the old pic

                $user->picture = Helper::upload_avatar('uploads/users',$request->file('picture'));
            }

            $user->login_by = $request->login_by ?  $request->login_by : $user->login_by;

            $user->save();

            if(in_array($user->login_by, ['facebook','google'])) {

                $user->is_verified = 1;

                $user->save();
            } 

            $response = $user->userResponse($user->id)->first()->toArray();
            
            \Log::info("Log of response".print_r($response , true));
        }

        return $response;

	}

	public static function delete($data = []) {

	}

	public static function find($data = []) {

	}

	public static function findBy($field , $value) {

	}

	public static function paginate($take , $skip) {

	}

    public static function referral_register($referral_code, $user_details) {

        $user_referral_details = UserReferrer::where('referral_code', $referral_code)->first();

        if($user_referral_details) {

            $referral_details =  Referral::where('user_id', $user_details->id)->where('referral_code', $referral_code)->first();

            if(!$referral_details) {

                $referral_details =  new Referral;
            }

            $referral_details->user_id = $user_details->id;

            $referral_details->parent_user_id = $user_referral_details->user_id;

            $referral_details->user_referrer_id = $user_referral_details->id;

            $referral_details->referral_code = $referral_code;

            $referral_details->source = $user_details->device_type." REGISTER";

            $referral_details->save();

            $user_referral_details->total_referrals = $user_referral_details->total_referrals + 1;

            $user_referral_details->total_referrals_earnings += (Setting::get('referral_commission') ?: 0);

            $user_referral_details->save();

        }

    }

}