<?php

namespace App;

use Illuminate\Foundation\Auth\User as Authenticatable;

use Illuminate\Database\Eloquent\SoftDeletes;

use App\Helpers\Helper;

class User extends Authenticatable {

    use SoftDeletes;

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    
    protected $dates = ['deleted_at'];
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name', 'email', 'password','is_activated','login_by','device_type','picture','register_type','login_status',
        'verification_code', 'verification_code_expiry' ,'is_verified'
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password', 'remember_token',
    ];

    /**
     * Scope a query to only include active users.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeUserResponse($query,$user_id) {

        return $query->where('users.id' , $user_id)->select('users.id as user_id', 
                'users.name as user_name' , 
                'users.email as email' , 
                'users.social_unique_id as social_unique_id' , 
                'users.token as token' , 
                'users.picture as user_picture' , 
                'users.login_by' , 
                'users.device_type' , 
                'users.description' , 
                'users.mobile' , 
                'users.latitude' , 
                'users.longitude' , 
                'users.payment_mode' , 
                'users.card_id' , 
                'users.provider_id' , 
                'users.status' , 
                'users.is_activated' ,
                'users.is_verified'
                );
    }

    public function userRequests() {
        return $this->hasMany('App\Requests');
    }

    public function userAbusePorviders() {
        return $this->hasMany('App\AbuseProvider');
    }

    public function favProviders() {
        return $this->hasMany('App\UserFavProvider');
    }

    public function userTempRequests() {
        return $this->hasMany('App\TempRequests');
    }

    public function userPayments() {
        return $this->hasMany('App\RequestPayment' , 'user_id');
    }

    public function userCards() {
        return $this->hasMany('App\Card');
    }

    public function userChats() {
        return $this->hasMany('App\ChatMessage');
    }

    public function userRatings() {
        return $this->hasMany('App\UserRating');
    }

    public function providerRatings() {
        return $this->hasMany('App\ProviderRating');
    }

    public static function boot() {

        //execute the parent's boot method 
        parent::boot();

        //delete your related models here, for example
        
        static::deleting(function($user) {

            Helper::delete_avatar('uploads/users' , $user->picture);

            if(count($user->userPayments) > 0 ) {

                foreach($user->userPayments as $userPayment) {

                    $userPayment->delete();
                } 
            }

            if(count($user->userRequests) > 0 ) {            

                foreach($user->userRequests as $userRequest) {

                    Helper::delete_avatar('uploads/jobs' , $userRequest->before_image);

                    Helper::delete_avatar('uploads/jobs' , $userRequest->after_image);

                    $userRequest->delete();
                
                } 
            }

            if(count($user->userTempRequests) > 0 ) {                        

                foreach($user->userTempRequests as $userTempRequest) {

                    $userTempRequest->delete();
                } 
            }

            if(count($user->userChats) > 0 ) {                        

                foreach($user->userChats as $userChat) {

                    $userChat->delete();
                } 
            } 

            if(count($user->userRatings) > 0 ) {                        
                foreach($user->userRatings as $userRating) {

                    $userRating->delete();
                }
            } 

            if(count($user->userCards) > 0 ) {                        

                foreach($user->userCards as $userCard) {

                    $userCard->delete();
                }
            } 

            if(count($user->providerRatings) > 0 ) {                        

                foreach($user->providerRatings as $providerRating) {

                    $providerRating->delete();
                }
            }

            if(count($user->userAbusePorviders) > 0 ) {

                foreach($user->userAbusePorviders as $userAbusePorvider) {

                    $userAbusePorvider->delete();
                } 
            }

            if(count($user->favProviders) > 0 ) {

                foreach($user->favProviders as $favProvider) {

                    $favProvider->delete();
                } 
            }
        
        }); 
        
        static::creating(function ($model) {

            $model->generateEmailCode();

        });
 
        static::created(function($model) {

            $model->attributes['unique_id'] = "UID"."-".$model->attributes['id']."-".$model->attributes['name'];

            $model->save();
        });
    }

    /**
     * Generates Token and Token Expiry
     * 
     * @return bool returns true if successful. false on failure.
     */

    protected function generateEmailCode() {

        $this->attributes['verification_code'] = Helper::generate_email_code();

        $this->attributes['verification_code_expiry'] = Helper::generate_email_expiry();

        return true;
    }

}
